package com.braintree.paypal.converters.impl;

import com.braintree.hybris.data.PayPalAddressData;
import com.braintree.paypal.validator.BraintreePayPalAddressDetailValidator;
import com.braintreegateway.Address;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.commercefacades.user.data.CountryData;
import de.hybris.platform.commercefacades.user.data.RegionData;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.servicelayer.dto.converter.ConversionException;
import de.hybris.platform.servicelayer.dto.converter.Converter;


public class BrainTreePayPalAddressDataConverter implements Converter<AddressModel, PayPalAddressData> {

    BraintreePayPalAddressDetailValidator payPalAddressDetailValidator;

    @Override
    public PayPalAddressData convert(final AddressModel hybrisAddress) throws ConversionException {
        final PayPalAddressData payPalAddressData = new PayPalAddressData();
        return convert(hybrisAddress, payPalAddressData);
    }

    public PayPalAddressData convert(final Address source, final String phone) {
        final PayPalAddressData payPalAddressData = new PayPalAddressData();
        return convert(source, phone, payPalAddressData);
    }

    @Override
    public PayPalAddressData convert(final AddressModel hybrisAddress, final PayPalAddressData paypalAddress)
        throws ConversionException {
        String recipientName = hybrisAddress.getFirstname() + " " + hybrisAddress.getLastname();
        if (hybrisAddress.getTitle() != null) {
            recipientName = hybrisAddress.getTitle().getName() + " " + recipientName;
        }

        paypalAddress.setLocality(hybrisAddress.getTown());
        paypalAddress.setPhone(hybrisAddress.getPhone1());
        paypalAddress.setPostalCode(hybrisAddress.getPostalcode());
        paypalAddress.setRecipientName(recipientName);
        paypalAddress.setStreetAddress(hybrisAddress.getLine1());
        paypalAddress.setExtendedAddress(hybrisAddress.getLine2());
        paypalAddress.setType("Personal");
        if (hybrisAddress.getRegion() != null) {
            paypalAddress.setRegion(hybrisAddress.getRegion().getIsocodeShort());
        }
        if (hybrisAddress.getCountry() != null) {
            paypalAddress.setCountryCodeAlpha2(hybrisAddress.getCountry().getIsocode());
        }
        return paypalAddress;
    }


    public PayPalAddressData convert(final Address source, final String phone, final PayPalAddressData paypalAddress)
        throws ConversionException {

        String recipientName = source.getFirstName();
        if(source.getLastName() != null) {
            recipientName = recipientName + " " + source.getLastName();
        }
        if(source.getLocality() != null) {
            paypalAddress.setLocality(source.getLocality());
        }
        if(phone != null) {
            paypalAddress.setPhone(phone);
        }
        if(source.getPostalCode() != null) {
            paypalAddress.setPostalCode(source.getPostalCode());
        }
        if(recipientName != null) {
            paypalAddress.setRecipientName(recipientName);
        }
        if(source.getStreetAddress() != null) {
            paypalAddress.setStreetAddress(source.getStreetAddress());
        }
        if(source.getExtendedAddress() != null) {
            paypalAddress.setExtendedAddress(source.getExtendedAddress());
        }
        paypalAddress.setType("Personal");
        if (source.getRegion() != null) {
            paypalAddress.setRegion(source.getRegion());
        }
        if (source.getCountryCodeAlpha2() != null) {
            paypalAddress.setCountryCodeAlpha2(source.getCountryCodeAlpha2());
        }
        return paypalAddress;
    }

    public AddressData convert(final PayPalAddressData address) {
        final AddressData addressData = new AddressData();
        final boolean isCountryIsoCodeValid = payPalAddressDetailValidator
            .validatePayPalCountryCode(address.getCountryCode());
        final boolean isRegionIsoCodeValid = payPalAddressDetailValidator
            .validatePayPalRegionCode(address.getCountryCode(), address.getState());
        if (isCountryIsoCodeValid) {
            final CountryData country = new CountryData();
            country.setIsocode(address.getCountryCode());
            addressData.setCountry(country);
        }

        if (isRegionIsoCodeValid) {
            final RegionData region = new RegionData();
            region.setIsocode(address.getCountryCode() + "-" + address.getState());
            addressData.setRegion(region);
        }

        addressData.setLine1(address.getLine1());
        addressData.setLine2(address.getLine2());
        addressData.setPhone(address.getPhone());
        addressData.setPostalCode(address.getPostalCode());
        if(address.getRecipientName() != null){
            final String[] names = address.getRecipientName().split(" ", 2);
            addressData.setFirstName(names[0]);
            if (names.length > 1) {
                addressData.setLastName(names[1]);
            }
        }
        addressData.setTown(address.getCity());

        return addressData;
    }

    public BraintreePayPalAddressDetailValidator getPayPalAddressDetailValidator() {
        return payPalAddressDetailValidator;
    }


    public void setPayPalAddressDetailValidator(final BraintreePayPalAddressDetailValidator payPalAddressDetailValidator) {
        this.payPalAddressDetailValidator = payPalAddressDetailValidator;
    }

}
