/**
 *
 */
package com.braintree.facade.impl;

import static com.braintree.constants.BraintreeConstants.BRAINTREE_TRY_DELETE_PAYMENT_DETAILS_WITH_NOT_CAPTURED_ORDER;
import static de.hybris.platform.servicelayer.util.ServicesUtil.validateParameterNotNullStandardMessage;

import com.braintree.command.request.BrainTreeAddressRequest;
import com.braintree.command.request.BrainTreeCreateCreditCardPaymentMethodRequest;
import com.braintree.command.request.BrainTreeCustomerRequest;
import com.braintree.command.request.BrainTreeUpdateCustomerRequest;
import com.braintree.command.request.BrainTreeDeletePaymentMethodRequest;
import com.braintree.command.request.BrainTreeUpdateCreditCardRequest;
import com.braintree.command.result.BrainTreeAddressResult;
import com.braintree.command.result.BrainTreeBillingAddressResult;
import com.braintree.command.result.BrainTreeCreatePaymentMethodResult;
import com.braintree.command.result.BrainTreeFindCustomerResult;
import com.braintree.command.result.BrainTreeUpdateCreditCardBillingAddressResult;
import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.customer.service.BrainTreeCustomerAccountService;
import com.braintree.exceptions.BrainTreeConnectWithPayPalException;
import com.braintree.exceptions.BraintreeDeletePaymentDetailsWithNotCapturedOrderException;
import com.braintree.facade.BrainTreeUserFacade;
import com.braintree.form.UpdateCreditCardBillingAddressForm;
import com.braintree.hybris.data.BrainTreeSubscriptionInfoData;
import com.braintree.hybris.data.BraintreeUserData;
import com.braintree.method.BrainTreePaymentService;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintree.order.service.BraintreeOrderBackofficeUtilService;
import com.braintree.payment.dto.BraintreeInfo;
import com.braintree.payment.info.service.BraintreePaymentInfoService;
import com.braintree.paypal.converters.impl.BraintreeBillingAddressConverter;
import com.braintree.servicelayer.i18n.BraintreeRegionI18NService;
import com.braintree.transaction.service.BrainTreeTransactionService;
import com.braintree.util.BrainTreeUtils;
import com.braintreegateway.PaymentMethod;
import com.google.common.collect.Lists;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.commercefacades.user.data.CustomerData;
import de.hybris.platform.commercefacades.user.impl.DefaultUserFacade;
import de.hybris.platform.commerceservices.enums.CustomerType;
import de.hybris.platform.commerceservices.strategies.CheckoutCustomerStrategy;
import de.hybris.platform.commerceservices.strategies.CustomerNameStrategy;
import de.hybris.platform.converters.Populator;
import de.hybris.platform.converters.impl.AbstractPopulatingConverter;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.c2l.CountryModel;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.core.model.user.UserModel;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.request.CreateSubscriptionRequest;
import de.hybris.platform.payment.commands.result.SubscriptionResult;
import de.hybris.platform.payment.dto.BillingInfo;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.exceptions.ModelNotFoundException;
import de.hybris.platform.servicelayer.i18n.CommonI18NService;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.user.UserService;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiPredicate;
import java.util.regex.Pattern;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.xml.security.utils.I18n;

/**
 * This class is a default implementation of BrainTreeUserFacade interface
 */
public class DefaultBrainTreeUserFacade extends DefaultUserFacade implements BrainTreeUserFacade {

    private Converter<BrainTreePaymentInfoModel, CCPaymentInfoData> brainTreePaymentInfoConverter;
    private Converter<CustomerModel, CustomerData> customerConverter;
    private Populator<CCPaymentInfoData, BrainTreePaymentInfoModel> braintreePaymentInfoReversePopulator;
    private BrainTreeCustomerAccountService brainTreeCustomerAccountService;
    private BrainTreePaymentService brainTreePaymentService;
    private CheckoutCustomerStrategy checkoutCustomerStrategy;
    private BrainTreeTransactionService brainTreeTransactionService;
    private CustomerNameStrategy customerNameStrategy;
    private Converter<AddressData, AddressModel> addressReverseConverter;
    private Converter<BrainTreeSubscriptionInfoData, BraintreeInfo> brainTreeSubscriptionInfoConverter;
    private BraintreePaymentInfoService paymentInfoService;
    private Converter<PaymentMethod, BrainTreePaymentInfoModel> paymentMethodConverter;
    private BraintreeBillingAddressConverter billingAddressConverter;
    private BrainTreeConfigService brainTreeConfigService;
    private BraintreeOrderBackofficeUtilService backofficeUtilService;
    private DefaultBrainTreePaymentFacade brainTreePaymentFacade;
    private UserService userService;
    private ModelService modelService;
    private Converter<BrainTreeBillingAddressResult, AddressModel> addressResultToAddressModelConverter;
    private BraintreeRegionI18NService regionI18NService;
    private CommonI18NService commonI18NService;


    @Override
    public void addAddress(final AddressData addressData) {
        validateParameterNotNullStandardMessage("addressData", addressData);

        if (Objects.nonNull(addressData.getRegion())) {
            if (Objects.isNull(regionI18NService.findRegion(commonI18NService.getCountry(addressData.getCountry().getIsocode()), addressData.getRegion().getIsocode()))) {
                addressData.setRegion(null);
            }
        }

        if(StringUtils.isBlank(addressData.getFirstName()) || StringUtils.isBlank(addressData.getLastName())) {
            addressData.setVisibleInAddressBook(false);
        }

        final CustomerModel currentCustomer = getCurrentUserForCheckout();

        final boolean makeThisAddressTheDefault = addressData.isDefaultAddress()
                || (currentCustomer.getDefaultShipmentAddress() == null && addressData.isVisibleInAddressBook());

        if (isDuplicateAddress(addressData, currentCustomer)) {
            addressData.setVisibleInAddressBook(false);
        }

        // Create the new address model
        final AddressModel newAddress = getModelService().create(AddressModel.class);
        getAddressReversePopulator().populate(addressData, newAddress);

        // Store the address against the user
        getCustomerAccountService().saveAddressEntry(currentCustomer, newAddress);

        // Update the address ID in the newly created address
        addressData.setId(newAddress.getPk().toString());

        if (makeThisAddressTheDefault) {
            getCustomerAccountService().setDefaultAddressEntry(currentCustomer, newAddress);
        }
    }

    private boolean isDuplicateAddress(final AddressData addressData, final CustomerModel customerModel) {
        BiPredicate<String, String> compare = (s1, s2) -> {
            if (StringUtils.isEmpty(s1) && s2 == null) {
                return true;
            }
            return s1 != null && s1.equals(s2);
        };

        BiPredicate<AddressModel, AddressData> compareRegion = (a1, a2) -> {
            if (a1.getRegion() == null || a2.getRegion() == null) {
                return true;
            }
            return a1.getRegion().getIsocode().equals(a2.getRegion().getIsocode());
        };

        for (AddressModel address : customerModel.getAddresses()) {
            if (address.getVisibleInAddressBook()
                    && compare.test(address.getCountry().getIsocode(), addressData.getCountry().getIsocode())
                    && compareRegion.test(address, addressData)
                    && compare.test(address.getFirstname(), addressData.getFirstName())
                    && compare.test(address.getLastname(), addressData.getLastName())
                    && compare.test(address.getTown(), addressData.getTown())
                    && compare.test(address.getPostalcode(), addressData.getPostalCode())
                    && compare.test(address.getLine1(), addressData.getLine1())
                    && compare.test(address.getLine2(), addressData.getLine2())
                    && compare.test(address.getPhone1(), addressData.getPhone())) {
                return true;
            }
        }
        return false;
    }

    @Override
    public void editAddress(final AddressData addressData) {
        final String brainTreeAddressId = getBrainTreeAddressId(addressData);

        if (StringUtils.isNotEmpty(brainTreeAddressId)) {
            final BrainTreeAddressRequest addressRequest = convertBrainTreeAddress(addressData);
            addressRequest.setAddressId(brainTreeAddressId);
            brainTreePaymentService.updateAddress(addressRequest);
        }

        super.editAddress(addressData);
    }

    @Override
    public void removeAddress(final AddressData addressData) {
        final String brainTreeAddressId = getBrainTreeAddressId(addressData);

        if (StringUtils.isNotEmpty(brainTreeAddressId)) {
            final BrainTreeAddressRequest addressRequest = convertBrainTreeAddress(addressData);
            addressRequest.setAddressId(brainTreeAddressId);
            brainTreePaymentService.removeAddress(addressRequest);
        }

        super.removeAddress(addressData);
    }

    private String getBrainTreeAddressId(final AddressData addressData) {
        validateParameterNotNullStandardMessage("addressData", addressData);
        final CustomerModel currentCustomer = getCurrentUserForCheckout();
        final AddressModel addressModel = getCustomerAccountService()
                .getAddressForCode(currentCustomer, addressData.getId());
        return addressModel.getBrainTreeAddressId();
    }

    private BrainTreeAddressRequest convertBrainTreeAddress(final AddressData address) {
        final String customerID = checkoutCustomerStrategy.getCurrentUserForCheckout().getBraintreeCustomerId();
        final BrainTreeAddressRequest addressRequest = new BrainTreeAddressRequest(customerID);
        addressRequest.setCompany(address.getTitle());
        addressRequest.setStreetAddress(address.getLine1());
        addressRequest.setExtendedAddress(address.getLine2());
        addressRequest.setFirstName(address.getFirstName());
        addressRequest.setLastName(address.getLastName());
        addressRequest.setLocality(address.getTown());
        addressRequest.setPostalCode(address.getPostalCode());

        if (address.getCountry() != null) {
            addressRequest.setCountryCodeAlpha2(address.getCountry().getIsocode());
        }
        if (address.getRegion() != null) {
            //The state or province. For PayPal addresses, the region must be a 2-letter abbreviation; for all other payment methods, it must be less than or equal to 255 characters.
            //because of hybris specific use the isocodeShort - 2 character isocode - and its right for braintree
            //the isocode return  2 character isocode US-CA - wrong for braintree
            addressRequest.setRegion(address.getRegion().getIsocodeShort());
        }

        return addressRequest;
    }

    @Override
    public void editPaymentMethod( final UpdateCreditCardBillingAddressForm updateCreditCardBillingAddressForm) {
        validateParameterNotNullStandardMessage("updateCreditCardBillingAddressForm", updateCreditCardBillingAddressForm);
        final CustomerModel currentCustomer = (CustomerModel) getUserService().getCurrentUser();
        final BrainTreePaymentInfoModel ccPaymentInfoModel = brainTreeCustomerAccountService
            .getBrainTreePaymentInfoForCode(
                currentCustomer, updateCreditCardBillingAddressForm.getPaymentMethodId());
        final BrainTreeUpdateCreditCardRequest request = new BrainTreeUpdateCreditCardRequest(
            ccPaymentInfoModel.getPaymentMethodToken());
        request.setToken(ccPaymentInfoModel.getPaymentMethodToken());
        final BillingInfo billingInfo = new BillingInfo();
        populateBillingInfo(billingInfo, updateCreditCardBillingAddressForm);
        request.setBillingInfo(billingInfo);
        final BrainTreeUpdateCreditCardBillingAddressResult result = getBrainTreePaymentService().updatePaymentMethod(request);
        if (result.isSuccess()) {
            final BrainTreeBillingAddressResult billingAddressResult = result.getBillingAddressResult();
            billingAddressResult.setCountyCode(updateCreditCardBillingAddressForm.getAddressData().getCountry().getIsocode());
            final AddressModel billingAddress = ccPaymentInfoModel.getBillingAddress();
            addressResultToAddressModelConverter.convert(billingAddressResult, billingAddress);
            modelService.save(billingAddress);
            ccPaymentInfoModel.setPaymentMethodToken(result.getPaymentMethodToken());
            ccPaymentInfoModel.setBillingAddress(billingAddress);
            modelService.save(ccPaymentInfoModel);
            modelService.refresh(ccPaymentInfoModel);
        } else {
            throw new AdapterException(result.getErrorMessage());
        }
    }

    private void populateBillingInfo(final BillingInfo billingInfo, final UpdateCreditCardBillingAddressForm updateCreditCardBillingAddressForm){
        billingInfo.setFirstName(updateCreditCardBillingAddressForm.getAddressData().getFirstName());
        billingInfo.setLastName(updateCreditCardBillingAddressForm.getAddressData().getLastName());
        billingInfo.setStreet1(updateCreditCardBillingAddressForm.getAddressData().getLine1());
        billingInfo.setStreet2(updateCreditCardBillingAddressForm.getAddressData().getLine2());
        billingInfo.setCity(updateCreditCardBillingAddressForm.getAddressData().getTown());
        billingInfo.setCountry(updateCreditCardBillingAddressForm.getAddressData().getCountry().getIsocode());

        if(updateCreditCardBillingAddressForm.getAddressData().getRegion()!=null) {
            billingInfo.setState(updateCreditCardBillingAddressForm.getAddressData().getRegion().getIsocode());
        }

        billingInfo.setPostalCode(updateCreditCardBillingAddressForm.getAddressData().getPostalCode());
    }

    @Override
    public boolean isCurrentUserHasBrainTreeCustomerId() {
        UserModel currentUser = userService.getCurrentUser();
        return !StringUtils.isEmpty(((CustomerModel) currentUser).getBraintreeCustomerId());
    }


    @Override
    public List<CCPaymentInfoData> getCCPaymentInfos(boolean saved) {

        final List<CCPaymentInfoData> ccPaymentInfos = super.getCCPaymentInfos(saved);
        final List<CCPaymentInfoData> brainTreeCCPaymentInfos = getBrainTreeCCPaymentInfos(saved);
        ccPaymentInfos.addAll(brainTreeCCPaymentInfos);
        return ccPaymentInfos;

    }

    @Override
    public List<CCPaymentInfoData> getBrainTreeCCPaymentInfos(final boolean saved) {

        final CustomerModel currentCustomer = (CustomerModel) getUserService().getCurrentUser();
        final Collection<BrainTreePaymentInfoModel> paymentInfos = brainTreeCustomerAccountService
                .getBrainTreePaymentInfos(
                        currentCustomer, saved);

        final List<BrainTreePaymentInfoModel> creditCards = Lists.newArrayList();

        boolean isPayPalEnabled = getBrainTreeConfigService().isPayPalStandardEnabled();
        boolean isHostedFieldsEnabled = getBrainTreeConfigService().isHostedFieldEnabled();
        boolean isVenmoEnabled = getBrainTreeConfigService().isVenmoEnabled();
        boolean isGooglePayEnabled = getBrainTreeConfigService().isGooglePayEnabled();
        boolean isSrcEnabled = getBrainTreeConfigService().isSrcEnabled();
        boolean isUsBankAccountEnabled = getBrainTreeConfigService().isUsBankAccountEnabled();
        boolean isAccountPaymentEnabled = isPayPalEnabled && isVenmoEnabled && isGooglePayEnabled;
        if (isHostedFieldsEnabled && isAccountPaymentEnabled && isSrcEnabled && isUsBankAccountEnabled) {
            addAllPaymentInfos(paymentInfos, creditCards);
        } else {
            for (final PaymentInfoModel paymentInfoModel : paymentInfos) {
                if (!(paymentInfoModel instanceof BrainTreePaymentInfoModel)) {
                    continue;
                }
                final BrainTreePaymentInfoModel brainTreePaymentInfo = (BrainTreePaymentInfoModel) paymentInfoModel;
                final String paymentProvider = (brainTreePaymentInfo).getPaymentProvider();
                if (isHostedFieldsEnabled && BrainTreeUtils.isCreditCardPayment(paymentProvider)) {
                    creditCards.add(brainTreePaymentInfo);
                    continue;
                }
                if (isPayPalEnabled && BrainTreeUtils.isPayPalPayment(paymentProvider)) {
                    creditCards.add(brainTreePaymentInfo);
                    continue;
                }
                if (isVenmoEnabled && BrainTreeUtils.isVenmoPayment(paymentProvider)) {
                    creditCards.add(brainTreePaymentInfo);
                    continue;
                }
                if (isGooglePayEnabled && BrainTreeUtils.isGooglePayment(paymentProvider)) {
                    creditCards.add(brainTreePaymentInfo);
                    continue;
                }
                if (isSrcEnabled && BrainTreeUtils.isSrcPayment(paymentProvider)) {
                    creditCards.add(brainTreePaymentInfo);
                }
                if (isUsBankAccountEnabled && BrainTreeUtils.isUsBankAccountPayment(paymentProvider)) {
                    creditCards.add(brainTreePaymentInfo);
                }
            }
        }

        final List<CCPaymentInfoData> ccPaymentInfos = new ArrayList<CCPaymentInfoData>();
        final PaymentInfoModel defaultPaymentInfoModel = currentCustomer.getDefaultPaymentInfo();

        for (final BrainTreePaymentInfoModel ccPaymentInfoModel : creditCards) {
            final CCPaymentInfoData defaultPaymentInfoData = getBrainTreePaymentInfoConverter()
                    .convert(ccPaymentInfoModel);

            if (ccPaymentInfoModel.equals(defaultPaymentInfoModel) &&
                    Optional.ofNullable(defaultPaymentInfoData).isPresent()) {
                defaultPaymentInfoData.setDefaultPaymentInfo(true);
            }

            if (isPaymentMethodDeletedOnBraintree(defaultPaymentInfoData)) {
                ccPaymentInfoModel.setIsDeletedOnBraintree(true);
                modelService.save(ccPaymentInfoModel);
                continue;
            }

            ccPaymentInfos.add(defaultPaymentInfoData);
        }
        return ccPaymentInfos;
    }

    private boolean isPaymentMethodDeletedOnBraintree(CCPaymentInfoData paymentInfoData) {
        brainTreePaymentFacade.setPaymentMethodNonce(paymentInfoData);
        return !Optional.ofNullable(paymentInfoData.getPaymentMethodNonce()).isPresent();
    }

    private void addAllPaymentInfos(Collection<BrainTreePaymentInfoModel> paymentInfos,
                                    List<BrainTreePaymentInfoModel> creditCards) {
        for (final PaymentInfoModel paymentInfoModel : paymentInfos) {
            if (paymentInfoModel instanceof BrainTreePaymentInfoModel) {
                creditCards.add((BrainTreePaymentInfoModel) paymentInfoModel);
            }
        }
    }

    @Override
    public BrainTreeFindCustomerResult findBrainTreeCustomer(String braintreeCustomerId) {
        if (StringUtils.isEmpty(braintreeCustomerId)) {
            return new BrainTreeFindCustomerResult(false);
        }
        final BrainTreeCustomerRequest findCustomerRequest = new BrainTreeCustomerRequest(braintreeCustomerId);
        findCustomerRequest.setCustomerId(braintreeCustomerId);
        final BrainTreeFindCustomerResult response = getBrainTreePaymentService().findCustomer(findCustomerRequest);
        return response;
    }

    @Override
    public CustomerData getCustomerByPayerId(String payerId) {
        return brainTreeCustomerAccountService.getCustomerByPayerId(payerId)
                .map(customerConverter::convert)
                .orElseThrow(() -> new BrainTreeConnectWithPayPalException("Customer with this payer id does not exist."));
    }

    @Override
    public boolean isCustomerWithPayerIdExist(String payerId) {
        return brainTreeCustomerAccountService.getCustomerByPayerId(payerId).isPresent();
    }

    public CustomerData getCustomerDataByUid(final String uId){
        return customerConverter.convert(userService.getUserForUID(uId, CustomerModel.class));
    }

    @Override
    public AddressData getBillingAddressByPaymentMethodId(String paymentMethodId) {
        validateParameterNotNullStandardMessage("paymentMethodId", paymentMethodId);
        final CustomerModel currentCustomer = (CustomerModel) getUserService().getCurrentUser();
        final BrainTreePaymentInfoModel ccPaymentInfoModel = brainTreeCustomerAccountService
            .getBrainTreePaymentInfoForCode(
                currentCustomer, paymentMethodId);
        return getAddressConverter().convert(ccPaymentInfoModel.getBillingAddress());
    }

    private void updateCurrentAddress(final CustomerModel currentCustomer, final AddressData addressData,
                                      final BrainTreeAddressResult brainTreeAddress) {
        for (final AddressModel addressModel : currentCustomer.getAddresses()) {
            if (addressModel.getPk().toString().equals(addressData.getId())) {
                addressModel.setBrainTreeAddressId(brainTreeAddress.getAddress().getId());
                getModelService().save(addressModel);
            }
        }
        getModelService().save(currentCustomer);
    }

    @Override
    public void unlinkCCPaymentInfo(final String id) {
        validateParameterNotNullStandardMessage("id", id);
        final CustomerModel currentCustomer = (CustomerModel) getUserService().getCurrentUser();

        final BrainTreePaymentInfoModel brainTreePaymentInfo = brainTreeCustomerAccountService
                .getBrainTreePaymentInfoForCode(
                        currentCustomer, id);

        brainTreeCustomerAccountService.unlinkCCPaymentInfo(currentCustomer, brainTreePaymentInfo);
    }

    @Override
    public void updateCCPaymentInfo(final CCPaymentInfoData paymentInfo) {
        validateParameterNotNullStandardMessage("paymentInfo", paymentInfo);
        validateParameterNotNullStandardMessage("paymentInfoID", paymentInfo.getId());
        final CustomerModel currentCustomer = (CustomerModel) getUserService().getCurrentUser();

        BrainTreePaymentInfoModel brainTreePaymentInfoForCode = brainTreeCustomerAccountService
                .getBrainTreePaymentInfoForCode(currentCustomer, paymentInfo.getId());
        if (brainTreePaymentInfoForCode != null) {
            braintreePaymentInfoReversePopulator.populate(paymentInfo, brainTreePaymentInfoForCode);
            modelService.save(brainTreePaymentInfoForCode);
            if (brainTreePaymentInfoForCode.getBillingAddress() != null) {
                getModelService().save(brainTreePaymentInfoForCode.getBillingAddress());
                getModelService().refresh(brainTreePaymentInfoForCode);
            }

        } else {
            super.updateCCPaymentInfo(paymentInfo);
        }

    }

    protected void updateDefaultPaymentInfo(final CustomerModel currentCustomer) {
        if (currentCustomer.getDefaultPaymentInfo() == null) {
            List<BrainTreePaymentInfoModel> brainTreePaymentInfoModelList = brainTreeCustomerAccountService
                    .getBrainTreePaymentInfos(currentCustomer, true);

            if (CollectionUtils.isNotEmpty(brainTreePaymentInfoModelList)) {
                BrainTreePaymentInfoModel paymentInfoModel = brainTreePaymentInfoModelList.get(brainTreePaymentInfoModelList.size() - 1);
                paymentInfoModel.setIsDefault(Boolean.TRUE);
                modelService.save(paymentInfoModel);
                getCustomerAccountService().setDefaultPaymentInfo(currentCustomer, paymentInfoModel);
            } else {
                super.updateDefaultPaymentInfo(currentCustomer);
            }
        }
    }

    @Override
    public void removeBTCCPaymentInfo(final String paymentMethodToken) {
        validateParameterNotNullStandardMessage("paymentMethodToken", paymentMethodToken);
        final CustomerModel currentCustomer = (CustomerModel) getUserService().getCurrentUser();
        final BrainTreeDeletePaymentMethodRequest request = new BrainTreeDeletePaymentMethodRequest(
                currentCustomer.getBraintreeCustomerId(), paymentMethodToken);
        brainTreePaymentService.deletePaymentMethod(request);
    }

    @Override
    public CCPaymentInfoData getCCPaymentInfoForCode(final String code) {
        if (StringUtils.isNotBlank(code)) {
            final CustomerModel currentCustomer = (CustomerModel) getUserService().getCurrentUser();
            final BrainTreePaymentInfoModel ccPaymentInfoModel = brainTreeCustomerAccountService
                    .getBrainTreePaymentInfoForCode(
                            currentCustomer, code);
            if (ccPaymentInfoModel != null) {
                final PaymentInfoModel defaultPaymentInfoModel = currentCustomer.getDefaultPaymentInfo();
                final CCPaymentInfoData paymentInfoData = getBrainTreePaymentInfoConverter()
                        .convert(ccPaymentInfoModel);
                if (ccPaymentInfoModel.equals(defaultPaymentInfoModel)) {
                    paymentInfoData.setDefaultPaymentInfo(true);
                }
                return paymentInfoData;
            } else {
                return super.getCCPaymentInfoForCode(code);
            }
        }

        return null;
    }

    @Override
    public void setDefaultPaymentInfo(final CCPaymentInfoData paymentInfoData) {
        validateParameterNotNullStandardMessage("paymentInfoData", paymentInfoData);
        final CustomerModel currentCustomer = getCurrentUserForCheckout();

        BrainTreePaymentInfoModel paymentInfoModel = brainTreeCustomerAccountService
                .getBrainTreePaymentInfoForCode(currentCustomer, paymentInfoData.getId());

        if (paymentInfoModel != null) {
            paymentInfoModel.setIsDefault(Boolean.TRUE);
            getCustomerAccountService().setDefaultPaymentInfo(currentCustomer, paymentInfoModel);
            modelService.save(paymentInfoModel);
            setBrainTreeDefaultPaymentInfo(currentCustomer, paymentInfoModel);
        } else {
            super.setDefaultPaymentInfo(paymentInfoData);
        }
    }

    private void setBrainTreeDefaultPaymentInfo(CustomerModel customer, BrainTreePaymentInfoModel paymentInfo) {
        BrainTreeUpdateCustomerRequest customerRequest = new BrainTreeUpdateCustomerRequest(paymentInfo.getCode());
        customerRequest.setId(customer.getBraintreeCustomerId());
        customerRequest.setDefaultPaymentMethodToken(paymentInfo.getPaymentMethodToken());
        getBrainTreePaymentService().setBrainTreeDefaultPaymentInfo(customerRequest);
    }

    @Override
    public void removeCCPaymentInfo(final String id) {
        validateParameterNotNullStandardMessage("id", id);
        final CustomerModel currentCustomer = (CustomerModel) getUserService().getCurrentUser();

        final BrainTreePaymentInfoModel brainTreePaymentInfo = brainTreeCustomerAccountService
                .getBrainTreePaymentInfoForCode(
                        currentCustomer, id);

        if (brainTreePaymentInfo != null) {
            if (brainTreeConfigService.isRestrictionPaymentMethodRemovalEnable()) {
                boolean isAvailableDeletePaymentDetails =
                        BrainTreeUtils.isPayPalPayment(brainTreePaymentInfo.getPaymentProvider())
                                || !brainTreeTransactionService.isCustomerHaveNotCapturedOrdersForPaymentDetails(currentCustomer, brainTreePaymentInfo);

                if (!isAvailableDeletePaymentDetails) {
                    throw new BraintreeDeletePaymentDetailsWithNotCapturedOrderException(
                            BRAINTREE_TRY_DELETE_PAYMENT_DETAILS_WITH_NOT_CAPTURED_ORDER);
                }
            }

            brainTreeCustomerAccountService.deleteBrainTreePaymentInfo(currentCustomer, brainTreePaymentInfo);

            if (brainTreePaymentInfo.isIsDefault()) {
                brainTreePaymentInfo.setIsDefault(Boolean.FALSE);
                currentCustomer.setDefaultPaymentInfo(null);
                modelService.save(currentCustomer);
            }

            if (brainTreePaymentInfo.getPaymentMethodToken() != null) {
                removeBTCCPaymentInfo(brainTreePaymentInfo.getPaymentMethodToken());
                brainTreePaymentInfo.setIsDeletedOnBraintree(true);
                modelService.save(brainTreePaymentInfo);
            }


        } else {
            super.removeCCPaymentInfo(id);
        }

        updateDefaultPaymentInfo(currentCustomer);
    }


    @Override
    public boolean isPayerIdInCustomer(String uid) {
        return StringUtils.isNotBlank(((CustomerModel) userService.getUserForUID(uid)).getPayPalPayerId());
    }

    /**
     * Used to check is customer has not captures for saved payment method type
     * @param paymentProvider
     * @return boolean
     */
    public boolean isCustomerHasNotCapturedOrdersForSavedPaymentMethodType(String paymentProvider) {
        final CustomerModel customerModel = (CustomerModel) getUserService().getCurrentUser();
        return customerModel.getOrders().stream()
                .filter(order -> {
                    PaymentInfoModel paymentInfo = order.getPaymentInfo();
                    if (paymentInfo instanceof BrainTreePaymentInfoModel) {
                        String curPaymentProvider = ((BrainTreePaymentInfoModel) (paymentInfo)).getPaymentProvider();
                        String token = ((BrainTreePaymentInfoModel) paymentInfo).getPaymentMethodToken();
                        if (token == null || StringUtils.isBlank(token)) {
                            return false;
                        }
                        return Optional.ofNullable(curPaymentProvider)
                                .map(s -> s.equalsIgnoreCase(paymentProvider))
                                .orElse(Boolean.FALSE);
                    }
                    return false;
                })
                .anyMatch(orderModel -> !OrderStatus.COMPLETED.equals(orderModel.getStatus())
                        && !backofficeUtilService.isOrderVoided(orderModel));
    }

    public BraintreeUserData getUserData() {
        UserModel currentUser = getCartService().getSessionCart().getUser();

        BraintreeUserData userData = new BraintreeUserData();

        if (currentUser.getName() != null) {
            final String[] names = currentUser.getName().split(" ", 2);
            userData.setFirstName(names[0]);
            if (names.length > 1) {
                userData.setLastName(names[1]);
            }
        }

        if (CustomerType.GUEST.getCode().equalsIgnoreCase(currentUser.getName())) {
            String[] splatUid = currentUser.getUid().split(Pattern.quote("|"));
            userData.setEmail(splatUid[splatUid.length - 1]);
            userData.setLastName(currentUser.getName());

        } else {
            userData.setEmail(currentUser.getUid());
        }

        return userData;
    }

    /**
     * @return the checkoutCustomerStrategy
     */
    @Override
    public CheckoutCustomerStrategy getCheckoutCustomerStrategy() {
        return checkoutCustomerStrategy;
    }

    /**
     * @param checkoutCustomerStrategy the checkoutCustomerStrategy to set
     */
    @Override
    public void setCheckoutCustomerStrategy(final CheckoutCustomerStrategy checkoutCustomerStrategy) {
        this.checkoutCustomerStrategy = checkoutCustomerStrategy;
    }

    /**
     * @return the brainTreePaymentService
     */
    public BrainTreePaymentService getBrainTreePaymentService() {
        return brainTreePaymentService;
    }

    /**
     * @param brainTreePaymentService the brainTreePaymentService to set
     */
    public void setBrainTreePaymentService(final BrainTreePaymentService brainTreePaymentService) {
        this.brainTreePaymentService = brainTreePaymentService;
    }

    public Converter<BrainTreePaymentInfoModel, CCPaymentInfoData> getBrainTreePaymentInfoConverter() {
        return brainTreePaymentInfoConverter;
    }

    public void setBrainTreePaymentInfoConverter(
            final Converter<BrainTreePaymentInfoModel, CCPaymentInfoData> brainTreePaymentInfoConverter) {
        this.brainTreePaymentInfoConverter = brainTreePaymentInfoConverter;
    }

    /**
     * @return the brainTreeCustomerAccountService
     */
    public BrainTreeCustomerAccountService getBrainTreeCustomerAccountService() {
        return brainTreeCustomerAccountService;
    }

    /**
     * @param brainTreeCustomerAccountService the brainTreeCustomerAccountService to set
     */
    public void setBrainTreeCustomerAccountService(
            final BrainTreeCustomerAccountService brainTreeCustomerAccountService) {
        this.brainTreeCustomerAccountService = brainTreeCustomerAccountService;
    }

    /**
     * @return the customerNameStrategy
     */
    public CustomerNameStrategy getCustomerNameStrategy() {
        return customerNameStrategy;
    }

    /**
     * @param customerNameStrategy the customerNameStrategy to set
     */
    public void setCustomerNameStrategy(final CustomerNameStrategy customerNameStrategy) {
        this.customerNameStrategy = customerNameStrategy;
    }

    /**
     * @return the brainTreeTransactionService
     */
    public BrainTreeTransactionService getBrainTreeTransactionService() {
        return brainTreeTransactionService;
    }

    /**
     * @param brainTreeTransactionService the brainTreeTransactionService to set
     */
    public void setBrainTreeTransactionService(final BrainTreeTransactionService brainTreeTransactionService) {
        this.brainTreeTransactionService = brainTreeTransactionService;
    }

    /**
     * @return the addressReverseConverter
     */
    public Converter<AddressData, AddressModel> getAddressReverseConverter() {
        return addressReverseConverter;
    }

    /**
     * @param addressReverseConverter the addressReverseConverter to set
     */
    public void setAddressReverseConverter(final Converter<AddressData, AddressModel> addressReverseConverter) {
        this.addressReverseConverter = addressReverseConverter;
    }

    /**
     * @return the brainTreeSubscriptionInfoConverter
     */
    public Converter<BrainTreeSubscriptionInfoData, BraintreeInfo> getBrainTreeSubscriptionInfoConverter() {
        return brainTreeSubscriptionInfoConverter;
    }

    /**
     * @param brainTreeSubscriptionInfoConverter the brainTreeSubscriptionInfoConverter to set
     */
    public void setBrainTreeSubscriptionInfoConverter(
            final Converter<BrainTreeSubscriptionInfoData, BraintreeInfo> brainTreeSubscriptionInfoConverter) {
        this.brainTreeSubscriptionInfoConverter = brainTreeSubscriptionInfoConverter;
    }

    public BraintreePaymentInfoService getPaymentInfoService() {
        return paymentInfoService;
    }

    public BraintreeRegionI18NService getRegionI18NService() {
        return regionI18NService;
    }

    public CommonI18NService getCommonI18NService() {
        return commonI18NService;
    }

    public void setPaymentInfoService(BraintreePaymentInfoService paymentInfoService) {
        this.paymentInfoService = paymentInfoService;
    }

    public Converter<PaymentMethod, BrainTreePaymentInfoModel> getPaymentMethodConverter() {
        return paymentMethodConverter;
    }

    public void setPaymentMethodConverter(Converter<PaymentMethod, BrainTreePaymentInfoModel> paymentMethodConverter) {
        this.paymentMethodConverter = paymentMethodConverter;
    }

    public BrainTreeConfigService getBrainTreeConfigService() {
        return brainTreeConfigService;
    }

    public void setBrainTreeConfigService(BrainTreeConfigService brainTreeConfigService) {
        this.brainTreeConfigService = brainTreeConfigService;
    }

    @Override
    public UserService getUserService() {
        return userService;
    }

    public DefaultBrainTreePaymentFacade getBrainTreePaymentFacade() {
        return brainTreePaymentFacade;
    }

    public void setBrainTreePaymentFacade(DefaultBrainTreePaymentFacade brainTreePaymentFacade) {
        this.brainTreePaymentFacade = brainTreePaymentFacade;
    }

    @Override
    public void setUserService(UserService userService) {
        this.userService = userService;
    }

    public BraintreeOrderBackofficeUtilService getBackofficeUtilService() {
        return backofficeUtilService;
    }

    public void setBackofficeUtilService(BraintreeOrderBackofficeUtilService backofficeUtilService) {
        this.backofficeUtilService = backofficeUtilService;
    }

    @Override
    public ModelService getModelService() {
        return modelService;
    }

    @Override
    public void setModelService(ModelService modelService) {
        this.modelService = modelService;
    }

    public Populator<CCPaymentInfoData, BrainTreePaymentInfoModel> getBraintreePaymentInfoReversePopulator() {
        return braintreePaymentInfoReversePopulator;
    }

    public void setBraintreePaymentInfoReversePopulator(
            Populator<CCPaymentInfoData, BrainTreePaymentInfoModel> braintreePaymentInfoReversePopulator) {
        this.braintreePaymentInfoReversePopulator = braintreePaymentInfoReversePopulator;
    }

    public BraintreeBillingAddressConverter getBillingAddressConverter() {
        return billingAddressConverter;
    }

    public void setBillingAddressConverter(
            BraintreeBillingAddressConverter billingAddressConverter) {
        this.billingAddressConverter = billingAddressConverter;
    }

    public Converter<BrainTreeBillingAddressResult, AddressModel> getAddressResultToAddressModelConverter() {
        return addressResultToAddressModelConverter;
    }

    public void setAddressResultToAddressModelConverter(
        Converter<BrainTreeBillingAddressResult, AddressModel> addressResultToAddressModelConverter) {
        this.addressResultToAddressModelConverter = addressResultToAddressModelConverter;
    }

    public void setCustomerConverter(Converter<CustomerModel, CustomerData> customerConverter) {
        this.customerConverter = customerConverter;
    }

    public void setRegionI18NService(BraintreeRegionI18NService regionI18NService) {
        this.regionI18NService = regionI18NService;
    }

    public void setCommonI18NService(CommonI18NService commonI18NService) {
        this.commonI18NService = commonI18NService;
    }
}
