package com.braintree.facade.impl;

import static com.braintree.constants.BraintreeConstants.PayPalIntent.PAYPAL_INTENT_SALE;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.FAILED_VERIFICATION_ERROR_MESSAGE;
import static com.braintree.constants.BraintreeConstants.US_BANK_ACCOUNT;
import static de.hybris.platform.servicelayer.util.ServicesUtil.validateParameterNotNullStandardMessage;
import static org.apache.commons.lang.StringUtils.isNotEmpty;

import com.braintree.command.result.BrainTreeCreatePaymentMethodResult;
import com.braintree.command.result.BrainTreePaymentMethodNonceResult;
import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.constants.BraintreeConstants;
import com.braintree.customer.service.BrainTreeCustomerAccountService;
import com.braintree.enums.BrainTreePaymentMethod;
import com.braintree.enums.ExpirationStatus;
import com.braintree.exceptions.BrainTreeCardVerifyException;
import com.braintree.exceptions.BrainTreeDeliveryAddressNotFoundException;
import com.braintree.facade.BrainTreeUserFacade;
import com.braintree.hybris.data.BrainTreePaymentInfoData;
import com.braintree.hybris.data.BrainTreeSubscriptionInfoData;
import com.braintree.hybris.data.BraintreeCardDetailsData;
import com.braintree.hybris.data.BraintreePaymentMethodNonceData;
import com.braintree.method.BrainTreePaymentService;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintree.model.BraintreeLocalPaymentMethodsModel;
import com.braintree.payment.dto.BraintreeInfo;
import com.braintree.payment.info.service.BraintreePaymentInfoService;
import com.braintree.payment.local.methods.service.BraintreeLocalPaymentMethodsService;
import com.braintree.payment.vault.StoreInVaultSupplier;
import com.braintree.paypal.converters.impl.BraintreeBillingAddressConverter;
import com.braintree.transaction.service.BrainTreeTransactionService;
import com.braintree.util.BrainTreeUtils;
import com.braintree.util.GenericBuilder;
import com.braintreegateway.WebhookNotification;
import de.hybris.platform.acceleratorfacades.payment.impl.DefaultPaymentFacade;
import de.hybris.platform.braintree.data.BrainTreeWebhookNotificationRequest;
import de.hybris.platform.commercefacades.customer.CustomerFacade;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commercefacades.order.data.OrderData;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.commerceservices.customer.CustomerEmailResolutionService;
import de.hybris.platform.commerceservices.customer.CustomerService;
import de.hybris.platform.commerceservices.enums.CustomerType;
import de.hybris.platform.commerceservices.order.CommerceCartService;
import de.hybris.platform.core.model.order.AbstractOrderModel;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.core.model.user.UserModel;
import de.hybris.platform.order.CartService;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.user.UserService;
import de.hybris.platform.store.BaseStoreModel;
import de.hybris.platform.store.services.BaseStoreService;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.servlet.http.HttpServletRequest;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

/**
 * This class extends DefaultPaymentFacade
 */
public class DefaultBrainTreePaymentFacade extends DefaultPaymentFacade {

    private static final Logger LOG = Logger.getLogger(DefaultBrainTreePaymentFacade.class);
    private BrainTreeUserFacade brainTreeUserFacade;
    private BrainTreePaymentService brainTreePaymentService;
    private CartService cartService;
    private BaseStoreService baseStoreService;
    private BrainTreeCustomerAccountService brainTreeCustomerAccountService;
    private BraintreePaymentInfoService braintreePaymentInfoService;
    private CommerceCartService commerceCartService;
    private ModelService modelService;
    private UserService userService;
    private CustomerFacade customerFacade;
    private CustomerService customerService;
    private CustomerEmailResolutionService customerEmailResolutionService;
    private BrainTreeTransactionService brainTreeTransactionService;

    private Converter<BrainTreePaymentInfoModel, BrainTreePaymentInfoData> brainTreePaymentInfoDataConverter;
    private Converter<AddressModel, AddressData> addressConverter;
    private Converter<AddressData, AddressModel> addressReverseConverter;
    private BraintreeBillingAddressConverter billingAddressConverter;
    private Converter<BrainTreeSubscriptionInfoData, BraintreeInfo> brainTreeSubscriptionInfoConverter;
    private BrainTreeConfigService brainTreeConfigService;
    private BraintreeLocalPaymentMethodsService braintreeLocalPaymentMethodsService;
    private Converter<OrderModel, OrderData> orderConverter;
    private Converter<BrainTreePaymentInfoModel, CCPaymentInfoData> brainTreePaymentInfoConverter;
    private Map<String, StoreInVaultSupplier> storeInVaultSupplierMap;

    /**
     * This method create LocalPaymentMethodSubscription
     *
     * @param paymentId payment id
     */
    public void createLocalPaymentMethodSubscription(String paymentId) {
        final CartModel cart = getCartService().getSessionCart();
        completeCreateLocalPaymentSubscription(paymentId, cart);

    }

    /**
     * This method is used to update LocalPaymentMethodSubscription
     *
     * @param nonce      nonce
     * @param deviceData deviceData
     * @param payerEmail payerEmail
     * @return Credit Card payment info data
     * @throws BrainTreeDeliveryAddressNotFoundException if smth get wrong
     */
    public CCPaymentInfoData updateLocalPaymentMethodSubscription(final String nonce, final String deviceData,
        final String payerEmail)
        throws BrainTreeDeliveryAddressNotFoundException {
        validateParameterNotNullStandardMessage("nonce", nonce);
        validateParameterNotNullStandardMessage("deviceData", deviceData);

        PaymentInfoModel paymentInfo = getCartService().getSessionCart().getPaymentInfo();
        AddressModel address = getCartService().getSessionCart().getDeliveryAddress();
        if (address == null) {
            throw new BrainTreeDeliveryAddressNotFoundException();
        }
        AddressModel billingAddress = getModelService().clone(address);

        if (paymentInfo instanceof BrainTreePaymentInfoModel) {
            UserModel user = getCartService().getSessionCart().getUser();
            String userUid = getCartService().getSessionCart().getUser().getUid();

            if (CustomerType.GUEST.getCode().equalsIgnoreCase(user.getName())) {
                String[] splatUid = userUid.split(Pattern.quote("|"));
                String email = splatUid[splatUid.length - 1];
                ((BrainTreePaymentInfoModel) paymentInfo).setPayer(email);
                billingAddress.setEmail(email);
            } else {
                ((BrainTreePaymentInfoModel) paymentInfo).setPayer(userUid);
            }
            ((BrainTreePaymentInfoModel) paymentInfo).setNonce(nonce);
            ((BrainTreePaymentInfoModel) paymentInfo).setDeviceData(deviceData);
            billingAddress.setOwner(paymentInfo);
            paymentInfo.setBillingAddress(billingAddress);
            getModelService().saveAll(paymentInfo, billingAddress);
            return getBrainTreePaymentInfoConverter().convert((BrainTreePaymentInfoModel) paymentInfo);
        }
        return null;
    }

    /**
     * This method is used to complete creating LocalPaymentSubscription
     *
     * @param paymentId paymentId
     * @param cart      cart
     */
    private void completeCreateLocalPaymentSubscription(String paymentId, CartModel cart) {
        validateParameterNotNullStandardMessage("cart", cart);

        final BrainTreePaymentInfoModel paymentInfo = getModelService().create(BrainTreePaymentInfoModel.class);
        final CustomerModel customerModel = getCurrentUserForCheckout();

        paymentInfo.setUser(customerModel);
        paymentInfo.setCode(customerModel.getUid() + "_" + UUID.randomUUID().toString());
        paymentInfo.setThreeDSecureConfiguration(getBrainTreeConfigService().is3dSecureConfiguration());
        paymentInfo.setPaymentId(paymentId);
        paymentInfo.setUsePaymentMethodToken(false);
        paymentInfo.setPayPalIntent(PAYPAL_INTENT_SALE);
        paymentInfo.setPaymentProvider(BrainTreePaymentMethod.LOCALPAYMENT.toString());
        paymentInfo.setOwner(cart);
        paymentInfo.setMerchantAccountIdForCurrentSite(
            getBrainTreeConfigService().getMerchantAccountIdForCurrentSiteAndCurrency());
        paymentInfo.setAdvancedFraudTools(brainTreeConfigService.isAdvancedFraudTools());

        cart.setPaymentInfo(paymentInfo);
        getModelService().saveAll(paymentInfo, cart);
    }

    /**
     * This method is used to completeCreateSubscription
     *
     * @param subscriptionData subscriptionInfo
     * @return Credit Card payment info data
     */
    public CCPaymentInfoData completeCreateSubscription(final BrainTreeSubscriptionInfoData subscriptionData) {
        final CustomerModel customer = getCurrentUserForCheckout();
        final CartModel cart = getCartService().getSessionCart();

        final AddressData addressData = subscriptionData.getAddressData();
        final AddressModel billingAddress = resolveBillingAddress(addressData, cart, customer,
            subscriptionData);

        if (brainTreeConfigService.isStoreInVault() && StringUtils.isEmpty(customer.getBraintreeCustomerId())) {
            LOG.debug("... creating customer on the braintree side");
            getBrainTreePaymentService().createCustomer(customer, billingAddress);
        }

        if(BrainTreeUtils.isUsBankAccountPayment(subscriptionData.getPaymentProvider())) {
            checkOnDuplicateAndVaultingAndVerifyingUsBankAccountPaymentMethod(subscriptionData, customer, billingAddress);
        }

        final BrainTreePaymentInfoModel paymentInfoModel = brainTreeTransactionService
            .createPaymentMethodInfo(billingAddress, customer, subscriptionData);
        cart.setPaymentInfo(paymentInfoModel);
        modelService.save(cart);

        return getBrainTreePaymentInfoConverter().convert(paymentInfoModel);
    }

    public CCPaymentInfoData completeCreateSubscriptionForMyAccount(
            final BrainTreeSubscriptionInfoData subscriptionData, CustomerModel customer) {
        final AddressModel billingAddress = buildBillingAddress(subscriptionData.getAddressData(), customer);

        final BrainTreeCreatePaymentMethodResult braintreePaymentMethod = brainTreePaymentService
                .createPaymentMethodOnBraintreeOrGetExisting(customer, billingAddress, subscriptionData);
        checkBraintreeResult(braintreePaymentMethod);
        if (BrainTreeUtils.isUsBankAccountPayment(subscriptionData.getPaymentProvider()) && !braintreePaymentMethod.getVerified()) {
            brainTreeUserFacade.removeBTCCPaymentInfo(braintreePaymentMethod.getPaymentMethodToken());
            throw new BrainTreeCardVerifyException(FAILED_VERIFICATION_ERROR_MESSAGE);
        }
        addAdditionalPaymentMethodFields(subscriptionData, braintreePaymentMethod);

        final BrainTreePaymentInfoModel paymentInfoModel = brainTreeTransactionService
                .createPaymentMethodInfo(billingAddress, customer, subscriptionData);
        braintreePaymentInfoService.checkOnDuplicatePaymentMethod(paymentInfoModel, customer, billingAddress);
        return getBrainTreePaymentInfoConverter().convert(paymentInfoModel);
    }

    public CCPaymentInfoData completeCreateSubscriptionForMyAccount(
        final BrainTreeSubscriptionInfoData subscriptionData) {
        final CustomerModel customer = getCurrentUserForCheckout();
        return completeCreateSubscriptionForMyAccount(subscriptionData, customer);
    }

    public CCPaymentInfoData createPaymentInfoForLogin(
        final BrainTreeSubscriptionInfoData subscriptionData,
        final String email) {
        final CustomerModel customer = (CustomerModel)userService.getUserForUID(email);
        return completeCreateSubscriptionForMyAccount(subscriptionData, customer);
    }

    public void setPaymentMethodNonce(final CCPaymentInfoData paymentInfoData) {
        final String token = paymentInfoData.getPaymentMethodToken();
        if (token != null) {
            final BrainTreePaymentMethodNonceResult nonceResult = getBrainTreePaymentService().createPaymentMethodNonce(token);
            paymentInfoData.setPaymentMethodNonce(nonceResult.getNonce());
        }
    }

    public BraintreePaymentMethodNonceData createPaymentMethodNonce(String token){
        final BrainTreePaymentMethodNonceResult nonceResult = getBrainTreePaymentService().createPaymentMethodNonce(token);
        return GenericBuilder.of(BraintreePaymentMethodNonceData::new)
                .with(BraintreePaymentMethodNonceData::setNonce, nonceResult.getNonce())
                .with(BraintreePaymentMethodNonceData::setDetails,
                        GenericBuilder.of(BraintreeCardDetailsData::new)
                                .with(BraintreeCardDetailsData::setBin, nonceResult.getBin())
                                .build()
                )
                .with(BraintreePaymentMethodNonceData::setShouldPerform3dSecure, nonceResult.isShouldPerform3dSecure())
                .build();
    }

    /**
     * This method is used to updateLocalPaymentMethodSubscription
     *
     * @param nonce nonce
     * @param cart  cart
     * @return payment info model
     */
    public PaymentInfoModel updateLocalPaymentMethodSubscription(final String nonce, final CartModel cart) {
        PaymentInfoModel paymentInfo = cart.getPaymentInfo();
        AddressModel billingAddress = getModelService().clone(cart.getDeliveryAddress());
        if (paymentInfo instanceof BrainTreePaymentInfoModel) {
            ((BrainTreePaymentInfoModel) paymentInfo).setNonce(nonce);
            paymentInfo.setBillingAddress(billingAddress);
            billingAddress.setOwner(paymentInfo);
            getModelService().saveAll(paymentInfo, cart, billingAddress);
        }
        return paymentInfo;
    }

    /**
     * This method is used to setDeviceData
     *
     * @param deviceData  deviceData
     * @param paymentInfo paymentInfo
     */
    public void setDeviceData(String deviceData, BrainTreePaymentInfoModel paymentInfo) {
        paymentInfo.setDeviceData(deviceData);
        getModelService().save(paymentInfo);
    }

    private void checkBraintreeResult(final BrainTreeCreatePaymentMethodResult paymentMethodResult)
        throws AdapterException {
        if (!paymentMethodResult.isSuccess()) {
            throw new AdapterException(paymentMethodResult.getErrorMessage());
        }
    }

    private AddressModel resolveBillingAddress(final AddressData addressBillingData, final AbstractOrderModel cart,
        final CustomerModel customer, final BrainTreeSubscriptionInfoData brainTreeSubscriptionInfoData) {
        AddressModel billingAddress = getModelService().create(AddressModel.class);

        if (addressBillingData != null) {
            billingAddress = buildBillingAddress(addressBillingData, customer);
        } else {
            // double convert instead of cloning
            final AddressData deliveryAddress = getAddressConverter().convert(cart.getDeliveryAddress());
            getAddressReverseConverter().convert(deliveryAddress, billingAddress);

            billingAddress.setBrainTreeAddressId(cart.getDeliveryAddress().getBrainTreeAddressId());
            if (billingAddress.getEmail() == null && brainTreeSubscriptionInfoData.getEmail() != null) {
                billingAddress.setEmail(brainTreeSubscriptionInfoData.getEmail());
            }
        }

        return billingAddress;
    }

    private AddressModel buildBillingAddress(final AddressData addressData, final CustomerModel customer) {
        final AddressModel billingAddress = getModelService().create(AddressModel.class);
        getAddressReverseConverter().convert(addressData, billingAddress);

        billingAddress.setBrainTreeAddressId(addressData.getBrainTreeAddressId());

        if (isNotEmpty(addressData.getEmail()) && !BraintreeConstants.GUEST_USER_TYPE
            .equals(Optional.ofNullable(customer.getType()).orElse(CustomerType.REGISTERED).getCode())) {
            billingAddress.setEmail(addressData.getEmail());
        } else {
            billingAddress.setEmail(customerEmailResolutionService.getEmailForCustomer(customer));
        }

        return billingAddress;
    }

    private void addAdditionalPaymentMethodFields(final BrainTreeSubscriptionInfoData brainTreeSubscriptionInfoData,
        final BrainTreeCreatePaymentMethodResult createPaymentMethodResult) {
        if (createPaymentMethodResult != null) {
            brainTreeSubscriptionInfoData.setPaymentMethodToken(createPaymentMethodResult.getPaymentMethodToken());
            brainTreeSubscriptionInfoData.setExpirationMonth(createPaymentMethodResult.getExpirationMonth());
            brainTreeSubscriptionInfoData.setExpirationYear(createPaymentMethodResult.getExpirationYear());
            brainTreeSubscriptionInfoData.setImageSource(createPaymentMethodResult.getImageSource());
            brainTreeSubscriptionInfoData.setCardNumber(createPaymentMethodResult.getCardNumber());
            brainTreeSubscriptionInfoData.setCardType(createPaymentMethodResult.getCardType());
            brainTreeSubscriptionInfoData.setCardholder(createPaymentMethodResult.getCardholderName());
            brainTreeSubscriptionInfoData.setVerified(createPaymentMethodResult.getVerified());
            if (StringUtils.isNotBlank(createPaymentMethodResult.getEmail())) {
                brainTreeSubscriptionInfoData.setEmail(createPaymentMethodResult.getEmail());
            }
            if (StringUtils.isNotBlank(createPaymentMethodResult.getLast4())) {
                brainTreeSubscriptionInfoData.setLast4(createPaymentMethodResult.getLast4());
            }
            if (StringUtils.isNotBlank(createPaymentMethodResult.getRoutingNumber())) {
                brainTreeSubscriptionInfoData.setRoutingNumber(createPaymentMethodResult.getRoutingNumber());
            }
        }
    }

    private void checkOnDuplicateAndVaultingAndVerifyingUsBankAccountPaymentMethod(
            BrainTreeSubscriptionInfoData subscriptionData, CustomerModel customer, AddressModel billingAddress) {
        final BrainTreeCreatePaymentMethodResult braintreePaymentMethod = brainTreePaymentService
                .createPaymentMethodOnBraintreeOrGetExisting(customer, billingAddress, subscriptionData);
        checkBraintreeResult(braintreePaymentMethod);
        if (!braintreePaymentMethod.getVerified()) {
            brainTreeUserFacade.removeBTCCPaymentInfo(braintreePaymentMethod.getPaymentMethodToken());
            throw new BrainTreeCardVerifyException(FAILED_VERIFICATION_ERROR_MESSAGE);
        }
        addAdditionalPaymentMethodFields(subscriptionData, braintreePaymentMethod);
    }

    /**
     * This method is used to get BrainTreePaymentInfoData
     *
     * @return Payment info data
     */
    public BrainTreePaymentInfoData getBrainTreePaymentInfoData() {
        final PaymentInfoModel paymentInfo = getCartService().getSessionCart().getPaymentInfo();
        if (paymentInfo instanceof BrainTreePaymentInfoModel) {

            validateParameterNotNullStandardMessage("paymentInfo", paymentInfo);

            final BrainTreePaymentInfoData paymentData = getBrainTreePaymentInfoDataConverter()
                .convert((BrainTreePaymentInfoModel) paymentInfo);
            return paymentData;
        }
        return null;
    }

    public String getExpirationStatusForCustomer() {
        CustomerModel customerModel = (CustomerModel) userService.getCurrentUser();
        String brainTreeCustomerId = customerModel.getBraintreeCustomerId();
        ExpirationStatus result = ExpirationStatus.NOT_EXPIRED;
        if (StringUtils.isNotBlank(brainTreeCustomerId)) {
            List<BrainTreePaymentInfoModel> cards = braintreePaymentInfoService.getAllSavedCreditCardByCustomer(brainTreeCustomerId);
            Predicate<ExpirationStatus> isExpirationStatusCart = status -> cards.stream()
                    .anyMatch(card -> card.getExpirationStatus().equals(status));
            if (isExpirationStatusCart.test(ExpirationStatus.EXPIRED)) {
                result = ExpirationStatus.EXPIRED;
            } else if (isExpirationStatusCart.test(ExpirationStatus.EXPIRE_SOON)
                    && customerModel.isCustomerHasNewExpireSoonCard()) {
                customerModel.setCustomerHasNewExpireSoonCard(false);
                modelService.save(customerModel);
                result = ExpirationStatus.EXPIRE_SOON;
            }
        }
        return result.getCode();
    }

    /**
     * This method is used to get BrainTreePaymentInfoData
     *
     * @param orderCode orderCode
     * @return Payment info data
     */
    public BrainTreePaymentInfoData getBrainTreePaymentInfoData(final String orderCode) {
        final BaseStoreModel baseStoreModel = getBaseStoreService().getCurrentBaseStore();
        final OrderModel orderModel = getBrainTreeCustomerAccountService().getOrderForCode(orderCode, baseStoreModel);
        final PaymentInfoModel paymentInfo = orderModel.getPaymentInfo();
        if (paymentInfo instanceof BrainTreePaymentInfoModel) {

            validateParameterNotNullStandardMessage("paymentInfo", paymentInfo);

            final BrainTreePaymentInfoData paymentData = getBrainTreePaymentInfoDataConverter()
                .convert((BrainTreePaymentInfoModel) paymentInfo);
            return paymentData;
        }
        return null;
    }

    /**
     * This method is used to get BrainTreePaymentInfoData by Cart
     *
     * @param cartCode cartCode
     * @return Payment info data
     */
    public BrainTreePaymentInfoData getBrainTreePaymentInfoDataByCart(final String cartCode) {
        final CartModel cartModel = getCommerceCartService()
            .getCartForCodeAndUser(cartCode, getUserService().getCurrentUser());
        final BrainTreePaymentInfoModel paymentInfo = (BrainTreePaymentInfoModel) cartModel.getPaymentInfo();
        validateParameterNotNullStandardMessage("paymentInfo", paymentInfo);
        final BrainTreePaymentInfoData paymentData = getBrainTreePaymentInfoDataConverter().convert(paymentInfo);
        return paymentData;
    }

    /**
     * This method is used to force unduplicate cart for replenishment
     *
     * @param cartCode cartCode
     */
    public void forceUnduplicateCartForReplenishment(final String cartCode) {
        final CartModel cartModel = getCommerceCartService()
            .getCartForCodeAndUser(cartCode, getUserService().getCurrentUser());
        final BrainTreePaymentInfoModel paymentInfo = (BrainTreePaymentInfoModel) cartModel.getPaymentInfo();
        if (paymentInfo.getDuplicate() && paymentInfo.isSaved()) {
            paymentInfo.setSaved(false);
        }
        paymentInfo.setDuplicate(false);
        getModelService().save(paymentInfo);
    }

    public List<BraintreeLocalPaymentMethodsModel> getLocalPaymentMethods() {
        return braintreeLocalPaymentMethodsService.getAllLocalPaymentMethods();
    }

    /**
     * This method is used to get Order By PaymentMethodNonce
     *
     * @param paymentMethodNonce paymentMethodNonce
     * @return Order data
     */
    public OrderData getOrderByPaymentMethodNonce(String paymentMethodNonce) {
        final OrderData orderData = new OrderData();
        OrderModel orderModel = braintreeLocalPaymentMethodsService.getOrderByPaymentMethodNonce(paymentMethodNonce);
        return (orderModel == null) ?
            null :
            getOrderConverter()
                .convert(braintreeLocalPaymentMethodsService.getOrderByPaymentMethodNonce(paymentMethodNonce),
                    orderData);
    }

    /**
     * This method is used to get Order By PaymentId
     *
     * @param paymentId paymentId
     * @return Order model
     */
    public OrderModel getOrderByPaymentId(String paymentId) {
        return braintreeLocalPaymentMethodsService.getOrderByPaymentId(paymentId);
    }

    /**
     * This method is used to get Cart By PaymentId
     *
     * @param paymentId paymentId
     * @return Cart model
     */
    public CartModel getCartByPaymentId(String paymentId) {
        return braintreeLocalPaymentMethodsService.getCartByPaymentId(paymentId);
    }

    /**
     * This method is used to get WebhookNotification
     *
     * @param request request
     * @return Webhook notification
     */
    public WebhookNotification getWebhookNotification(final HttpServletRequest request) {
        BrainTreeWebhookNotificationRequest webhookNotificationRequest = new BrainTreeWebhookNotificationRequest();
        webhookNotificationRequest.setBtPayload(request.getParameter("bt_payload"));
        webhookNotificationRequest.setBtSignature(request.getParameter("bt_signature"));

        return brainTreePaymentService.getWebhookNotification(webhookNotificationRequest);
    }

    /**
     * This method is used to get AvailablePayments
     *
     * @param brainTreeCCPaymentInfos brainTreeCCPaymentInfos
     * @return List of available payment info data
     */
    public List<CCPaymentInfoData> getAvailablePayments(List<CCPaymentInfoData> brainTreeCCPaymentInfos) {
        return brainTreeCCPaymentInfos.stream().filter(payment -> payment.getPaymentMethodNonce() != null)
            .collect(Collectors.toList());
    }

    /**
     * @return the brainTreePaymentService
     */
    public BrainTreePaymentService getBrainTreePaymentService() {
        return brainTreePaymentService;
    }

    /**
     * @param brainTreePaymentService the brainTreePaymentService to set
     */
    public void setBrainTreePaymentService(final BrainTreePaymentService brainTreePaymentService) {
        this.brainTreePaymentService = brainTreePaymentService;
    }

    /**
     * @return the cartService
     */
    public CartService getCartService() {
        return cartService;
    }

    /**
     * @param cartService the cartService to set
     */
    public void setCartService(final CartService cartService) {
        this.cartService = cartService;
    }

    /**
     * @return the baseStoreService
     */
    public BaseStoreService getBaseStoreService() {
        return baseStoreService;
    }

    /**
     * @param baseStoreService the baseStoreService to set
     */
    public void setBaseStoreService(final BaseStoreService baseStoreService) {
        this.baseStoreService = baseStoreService;
    }

    /**
     * @return the customerAccountService
     */
    public BrainTreeCustomerAccountService getBrainTreeCustomerAccountService() {
        return brainTreeCustomerAccountService;
    }

    /**
     * @param brainTreeCustomerAccountService the customerAccountService to set
     */
    public void setBrainTreeCustomerAccountService(
        final BrainTreeCustomerAccountService brainTreeCustomerAccountService) {
        this.brainTreeCustomerAccountService = brainTreeCustomerAccountService;
    }

    /**
     * @return the commerceCartService
     */
    public CommerceCartService getCommerceCartService() {
        return commerceCartService;
    }

    /**
     * @param commerceCartService the commerceCartService to set
     */
    public void setCommerceCartService(final CommerceCartService commerceCartService) {
        this.commerceCartService = commerceCartService;
    }

    /**
     * @return the modelService
     */
    public ModelService getModelService() {
        return modelService;
    }

    /**
     * @param modelService the modelService to set
     */
    public void setModelService(final ModelService modelService) {
        this.modelService = modelService;
    }

    /**
     * @return the addressConverter
     */
    public Converter<AddressModel, AddressData> getAddressConverter() {
        return addressConverter;
    }

    /**
     * @param addressConverter the addressConverter to set
     */
    public void setAddressConverter(final Converter<AddressModel, AddressData> addressConverter) {
        this.addressConverter = addressConverter;
    }

    /**
     * @return the billingAddressConverter
     */
    public BraintreeBillingAddressConverter getBillingAddressConverter() {
        return billingAddressConverter;
    }

    /**
     * @param billingAddressConverter the billingAddressConverter to set
     */
    public void setBillingAddressConverter(final BraintreeBillingAddressConverter billingAddressConverter) {
        this.billingAddressConverter = billingAddressConverter;
    }

    /**
     * @return the userService
     */
    @Override
    public UserService getUserService() {
        return userService;
    }

    /**
     * @param userService the userService to set
     */
    @Override
    public void setUserService(final UserService userService) {
        this.userService = userService;
    }

    public CustomerService getCustomerService() {
        return customerService;
    }

    public void setCustomerService(CustomerService customerService) {
        this.customerService = customerService;
    }

    public CustomerFacade getCustomerFacade() {
        return customerFacade;
    }

    public void setCustomerFacade(CustomerFacade customerFacade) {
        this.customerFacade = customerFacade;
    }

    /**
     * @return the brainTreeTransactionService
     */
    public BrainTreeTransactionService getBrainTreeTransactionService() {
        return brainTreeTransactionService;
    }

    /**
     * @param brainTreeTransactionService the brainTreeTransactionService to set
     */
    public void setBrainTreeTransactionService(final BrainTreeTransactionService brainTreeTransactionService) {
        this.brainTreeTransactionService = brainTreeTransactionService;
    }

    /**
     * @return the customerEmailResolutionService
     */
    public CustomerEmailResolutionService getCustomerEmailResolutionService() {
        return customerEmailResolutionService;
    }

    /**
     * @param customerEmailResolutionService the customerEmailResolutionService to set
     */
    public void setCustomerEmailResolutionService(final CustomerEmailResolutionService customerEmailResolutionService) {
        this.customerEmailResolutionService = customerEmailResolutionService;
    }

    /**
     * @return the brainTreeSubscriptionInfoConverter
     */
    public Converter<BrainTreeSubscriptionInfoData, BraintreeInfo> getBrainTreeSubscriptionInfoConverter() {
        return brainTreeSubscriptionInfoConverter;
    }

    /**
     * @param brainTreeSubscriptionInfoConverter the brainTreeSubscriptionInfoConverter to set
     */
    public void setBrainTreeSubscriptionInfoConverter(
        final Converter<BrainTreeSubscriptionInfoData, BraintreeInfo> brainTreeSubscriptionInfoConverter) {
        this.brainTreeSubscriptionInfoConverter = brainTreeSubscriptionInfoConverter;
    }

    /**
     * @return the addressReverseConverter
     */
    public Converter<AddressData, AddressModel> getAddressReverseConverter() {
        return addressReverseConverter;
    }

    /**
     * @param addressReverseConverter the addressReverseConverter to set
     */
    public void setAddressReverseConverter(final Converter<AddressData, AddressModel> addressReverseConverter) {
        this.addressReverseConverter = addressReverseConverter;
    }

    /**
     * @return the brainTreePaymentInfoDataConverter
     */
    public Converter<BrainTreePaymentInfoModel, BrainTreePaymentInfoData> getBrainTreePaymentInfoDataConverter() {
        return brainTreePaymentInfoDataConverter;
    }

    /**
     * @param brainTreePaymentInfoDataConverter the brainTreePaymentInfoDataConverter to set
     */
    public void setBrainTreePaymentInfoDataConverter(
        final Converter<BrainTreePaymentInfoModel, BrainTreePaymentInfoData> brainTreePaymentInfoDataConverter) {
        this.brainTreePaymentInfoDataConverter = brainTreePaymentInfoDataConverter;
    }

    public BrainTreeConfigService getBrainTreeConfigService() {
        return brainTreeConfigService;
    }

    public void setBrainTreeConfigService(BrainTreeConfigService brainTreeConfigService) {
        this.brainTreeConfigService = brainTreeConfigService;
    }

    public BrainTreeUserFacade getBrainTreeUserFacade() {
        return brainTreeUserFacade;
    }

    public void setBrainTreeUserFacade(BrainTreeUserFacade brainTreeUserFacade) {
        this.brainTreeUserFacade = brainTreeUserFacade;
    }

    public BraintreeLocalPaymentMethodsService getBraintreeLocalPaymentMethodsService() {
        return braintreeLocalPaymentMethodsService;
    }

    public void setBraintreeLocalPaymentMethodsService(
        BraintreeLocalPaymentMethodsService braintreeLocalPaymentMethodsService) {
        this.braintreeLocalPaymentMethodsService = braintreeLocalPaymentMethodsService;
    }

    public Converter<OrderModel, OrderData> getOrderConverter() {
        return orderConverter;
    }

    public void setOrderConverter(Converter<OrderModel, OrderData> orderConverter) {
        this.orderConverter = orderConverter;
    }

    public Converter<BrainTreePaymentInfoModel, CCPaymentInfoData> getBrainTreePaymentInfoConverter() {
        return brainTreePaymentInfoConverter;
    }

    public void setBrainTreePaymentInfoConverter(
        Converter<BrainTreePaymentInfoModel, CCPaymentInfoData> brainTreePaymentInfoConverter) {
        this.brainTreePaymentInfoConverter = brainTreePaymentInfoConverter;
    }

    public Map<String, StoreInVaultSupplier> getStoreInVaultSupplierMap() {
        return storeInVaultSupplierMap;
    }

    public void setStoreInVaultSupplierMap(
        Map<String, StoreInVaultSupplier> storeInVaultSupplierMap) {
        this.storeInVaultSupplierMap = storeInVaultSupplierMap;
    }

    public BraintreePaymentInfoService getBraintreePaymentInfoService() {
        return braintreePaymentInfoService;
    }

    public void setBraintreePaymentInfoService(
        BraintreePaymentInfoService braintreePaymentInfoService) {
        this.braintreePaymentInfoService = braintreePaymentInfoService;
    }
}
