package com.braintree.facade.impl;

import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.customer.service.BrainTreeCustomerAccountService;
import com.braintree.facade.BrainTreeConnectWithPayPalFacade;
import com.braintree.hybris.data.BrainTreeConnectWithPayPalAddressData;
import com.braintree.hybris.data.BrainTreeConnectWithPayPalUserData;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintree.service.BrainTreeConnectWithPayPalService;
import com.braintree.util.BrainTreeUtils;
import de.hybris.platform.core.model.c2l.CountryModel;
import de.hybris.platform.core.model.c2l.RegionModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.servicelayer.i18n.CommonI18NService;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.user.UserService;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.function.BiPredicate;
import java.util.stream.Collectors;
import org.apache.commons.lang.StringUtils;

public class DefaultBrainTreeConnectWithPayPalFacade implements BrainTreeConnectWithPayPalFacade {

    private static final String EMPTY = "";
    private static final String SPACE = " ";

    private BrainTreeConnectWithPayPalService brainTreeConnectWithPayPalService;
    private BrainTreeCustomerAccountService brainTreeCustomerAccountService;
    private UserService userService;
    private ModelService modelService;
    private CommonI18NService commonI18NService;
    private BrainTreeConfigService brainTreeConfigService;

    @Override
    public BrainTreeConnectWithPayPalUserData getUserDataByAccessToken(String accessToken) {
        return brainTreeConnectWithPayPalService.getUserDataByAccessToken(accessToken);
    }

    @Override
    public String exchangeAuthorizationCodeToAccessToken(String authorizationToken) {
        return brainTreeConnectWithPayPalService.exchangeAuthorizationCodeToAccessToken(authorizationToken);
    }

    @Override
    public void setAccessTokenForCustomer(String accessTokenGuid, String payerId) {
        brainTreeCustomerAccountService.setAccessTokenForCustomer(accessTokenGuid, payerId);
    }

    @Override
    public void setDefaultAccessToken(String payerId) {
        brainTreeCustomerAccountService.setDefaultAccessToken(payerId);
    }

    @Override
    public boolean isPayPalAddressPresent(String email, BrainTreeConnectWithPayPalAddressData addressData) {
        Collection<AddressModel> addresses = userService.getUserForUID(email).getAddresses();
        BiPredicate<String, String> compare = (s1, s2) -> {
            if (s1 == null && s2 == null) {
                return true;
            }
            return s1 != null && s1.equals(s2);
        };

        BiPredicate<AddressModel, BrainTreeConnectWithPayPalAddressData> compareRegion = (s1, s2) -> {
            if (s1.getRegion() == null && s2.getRegion() == null) {
                return true;
            }
            if(s1.getRegion() == null){
                return false;
            }
            return s1.getRegion().getIsocodeShort().equals(s2.getRegion());
        };

        for (AddressModel addressModel: addresses){
            if(compare.test(addressModel.getCountry().getIsocode(), addressData.getCountry())
                && compareRegion.test(addressModel, addressData)
                && compare.test(addressModel.getPostalcode(), addressData.getPostalCode())
                && compare.test(addressModel.getLine1(), addressData.getStreetAddress())
                && compare.test(addressModel.getTown(), addressData.getLocality())){
                return true;
            }
        }
        return false;
    }

    @Override
    public boolean isPayPalPaymentMethodPresent(String email) {
        final CustomerModel customerModel = (CustomerModel) userService.getUserForUID(email);
        final List<BrainTreePaymentInfoModel> paymentInfoList = brainTreeCustomerAccountService.getBrainTreePaymentInfos(customerModel, true);
        for(BrainTreePaymentInfoModel paymentInfoModel: paymentInfoList ){
            if (BrainTreeUtils.isPayPalPayment(paymentInfoModel.getPaymentProvider()) && paymentInfoModel.getPayer().equalsIgnoreCase(email)){
                return true;
            }
        }
        return false;
    }

    @Override
    public boolean isPayPalPaymentMethodPresentOrIntentOrder(String email){
        if (brainTreeConfigService.isPayPalStandardEnabled() && brainTreeConfigService.isStoreInVault()) {
            return isPayPalPaymentMethodPresent(email)
                    || BrainTreeUtils.isIntentOrder(brainTreeConfigService.getIntent());
        } else {
            return true;
        }
    }

    @Override
    public boolean shouldSavePayPalPaymentInfo() {
        return brainTreeConfigService.isPayPalStandardEnabled()
                && !BrainTreeUtils.isIntentOrder(brainTreeConfigService.getIntent())
                && brainTreeConfigService.isStoreInVault();
    }

    @Override
    public void addAddressForNewUser(BrainTreeConnectWithPayPalAddressData addressData, String email) {
        final AddressModel newAddress = modelService.create(AddressModel.class);
        final CountryModel countryModel = commonI18NService.getCountry(addressData.getCountry());
        final CustomerModel customer = (CustomerModel) userService.getUserForUID(email);

        if (StringUtils.isNotBlank(addressData.getRegion())) {
            final String regionIsoCode = String.format("%s-%s", countryModel.getIsocode(), addressData.getRegion());
            final RegionModel regionModel = commonI18NService.getRegion(countryModel, regionIsoCode);
            newAddress.setRegion(regionModel);
            modelService.save(regionModel);
        }

        List<String> fullName = Arrays.asList(customer.getName().split(SPACE));
        newAddress.setFirstname(fullName.stream().findFirst().orElse(EMPTY));
        newAddress.setLastname(fullName.stream().skip(1).collect(Collectors.joining(" ")));
        newAddress.setEmail(customer.getContactEmail());
        newAddress.setCountry(countryModel);
        newAddress.setTown(addressData.getLocality());
        newAddress.setLine1(addressData.getStreetAddress());
        newAddress.setPostalcode(addressData.getPostalCode());
        newAddress.setShippingAddress(true);

        customer.setDefaultShipmentAddress(newAddress);
        customer.setDefaultPaymentAddress(newAddress);
        brainTreeCustomerAccountService.saveAddressEntry(customer, newAddress);
    }

    public BrainTreeConnectWithPayPalService getBrainTreeConnectWithPayPalService() {
        return brainTreeConnectWithPayPalService;
    }

    public void setBrainTreeConnectWithPayPalService(
        BrainTreeConnectWithPayPalService brainTreeConnectWithPayPalService) {
        this.brainTreeConnectWithPayPalService = brainTreeConnectWithPayPalService;
    }

    public BrainTreeCustomerAccountService getBrainTreeCustomerAccountService() {
        return brainTreeCustomerAccountService;
    }

    public void setBrainTreeCustomerAccountService(
        BrainTreeCustomerAccountService brainTreeCustomerAccountService) {
        this.brainTreeCustomerAccountService = brainTreeCustomerAccountService;
    }

    public void setUserService(UserService userService) {
        this.userService = userService;
    }

    public void setModelService(ModelService modelService) {
        this.modelService = modelService;
    }

    public void setCommonI18NService(CommonI18NService commonI18NService) {
        this.commonI18NService = commonI18NService;
    }

    public void setBrainTreeConfigService(BrainTreeConfigService brainTreeConfigService) {
        this.brainTreeConfigService = brainTreeConfigService;
    }
}
