package com.braintree.facade.backoffice.impl;

import com.braintree.exceptions.BraintreeErrorException;
import com.braintree.facade.backoffice.BraintreeBackofficePartialRefundFacade;
import com.braintree.order.refund.partial.services.BraintreePartialRefundService;
import com.braintree.order.service.BraintreeOrderBackofficeUtilService;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.ordermanagementfacades.payment.data.PaymentTransactionEntryData;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.model.ModelService;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import java.math.BigDecimal;

/**
 * This class is a default implementation of the BraintreeBackofficePartialRefundFacade interface
 */
public class DefaultBraintreeBackofficePartialRefundFacade implements BraintreeBackofficePartialRefundFacade {

    private static final Logger LOG = Logger.getLogger(DefaultBraintreeBackofficePartialRefundFacade.class);

    @Autowired
    private BraintreePartialRefundService partialRefundService;
    @Autowired
    private BraintreeOrderBackofficeUtilService backofficeUtilService;

    @Autowired
    @Qualifier("ordermanagementPaymentTransactionEntryConverter")
    private Converter<PaymentTransactionEntryModel, PaymentTransactionEntryData> ordermanagementPaymentTransactionEntryConverter;
    @Autowired
    private ModelService modelService;

    @Override
    public boolean isPartialRefundPossible(final OrderModel order) {
        if (order == null) {
            return false;
        }
        LOG.info("isPartialRefundPossible, order.getTotalPrice: " + order.getTotalPrice());

        modelService.refresh(order);
        if(backofficeUtilService.isOrderVoided(order)){
            return false;
        }

        BigDecimal capturedAmount = BigDecimal.ZERO;
        for (final PaymentTransactionModel paymentTransaction : order.getPaymentTransactions()) {
            for (final PaymentTransactionEntryModel transactionEntry : paymentTransaction.getEntries()) {
                if ("SUCCESFULL".equals(transactionEntry.getTransactionStatusDetails()) &&
                    TransactionStatus.ACCEPTED.name().equals(transactionEntry.getTransactionStatus())) {
                    capturedAmount = checkTransactionTypeAndAddCapturedAmount(transactionEntry, capturedAmount);
                }
            }
        }
        return capturedAmount.compareTo(BigDecimal.ZERO) > 0;
    }

    private BigDecimal checkTransactionTypeAndAddCapturedAmount(PaymentTransactionEntryModel transactionEntry,
                                                                BigDecimal capturedAmount) {
        final PaymentTransactionType transactionType = transactionEntry.getType();
        if (PaymentTransactionType.CAPTURE.equals(transactionType) || PaymentTransactionType.PARTIAL_CAPTURE
                .equals(transactionType)) {
            capturedAmount = capturedAmount.add(transactionEntry.getAmount());
        } else if (PaymentTransactionType.REFUND_PARTIAL.equals(transactionType)) {
            capturedAmount = capturedAmount.subtract(transactionEntry.getAmount());
        }
        return capturedAmount;
    }

    @Override
    public PaymentTransactionEntryData partialRefundTransaction(final OrderModel order,
        final PaymentTransactionEntryModel paymentTransactionEntry, final BigDecimal amount)
        throws BraintreeErrorException {
        final PaymentTransactionEntryModel response = partialRefundService.partialRefundTransaction(order,
            paymentTransactionEntry, amount);
        return ordermanagementPaymentTransactionEntryConverter.convert(response);
    }
}
