package com.braintree.facade.backoffice.impl;

import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.exceptions.BraintreeErrorException;
import com.braintree.facade.backoffice.BraintreeBackofficeMultiCaptureFacade;
import com.braintree.method.BrainTreePaymentService;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintree.order.capture.partial.services.BraintreePartialCaptureService;
import com.braintree.order.service.BraintreeOrderBackofficeUtilService;
import com.braintree.order.submitForSettlement.service.BraintreeSubmitForSettlementService;
import com.braintree.util.BrainTreeUtils;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.ordermanagementfacades.payment.data.PaymentTransactionEntryData;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.model.ModelService;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.function.Predicate;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

/**
 * This class is a default implementation of the BraintreeBackofficeMultiCaptureFacade interface
 */
public class DefaultBraintreeBackofficeMultiCaptureFacade implements BraintreeBackofficeMultiCaptureFacade {

    private static final Logger LOG = Logger.getLogger(DefaultBraintreeBackofficeMultiCaptureFacade.class);

    @Autowired
    private BrainTreePaymentService brainTreePaymentService;
    @Autowired
    private ModelService modelService;
    @Autowired
    private BrainTreeConfigService brainTreeConfigService;
    @Autowired
    @Qualifier("braintreePartialCaptureService")
    private BraintreePartialCaptureService braintreePartialCaptureService;
    @Autowired
    private BraintreeSubmitForSettlementService braintreeSubmitForSettlementService;
    @Autowired
    private BraintreeOrderBackofficeUtilService backofficeUtilService;
    @Autowired
    @Qualifier("ordermanagementPaymentTransactionEntryConverter")
    private Converter<PaymentTransactionEntryModel, PaymentTransactionEntryData> ordermanagementPaymentTransactionEntryConverter;

    @Override
    public boolean isMultiCapturePossible(final OrderModel order) {
        if (null == order) {
            LOG.warn("order: " + order);
            return false;
        }
        LOG.info("isMultiCapturePossible, order.getTotalPrice: " + order.getTotalPrice());

        boolean isMultiCapturePossible = false;
        modelService.refresh(order);
        Boolean settlementConfigParameter = brainTreeConfigService.isSubmitForSettlement();

        if (order.getPaymentInfo() instanceof BrainTreePaymentInfoModel) {
            final String intent = getOrderIntent(order);
            final String paymentProvider = order.getPaymentInfo().getProperty("paymentProvider");

            if (backofficeUtilService.paidByCard(order) || BrainTreeUtils.isVenmoPayment(paymentProvider)) {
                if (!settlementConfigParameter && !backofficeUtilService.isOrderVoided(order) && !(OrderStatus.COMPLETED
                    .equals(order.getStatus()))) {
                    isMultiCapturePossible = true;
                }
            } else if (!backofficeUtilService.isOrderVoided(order) && (getPossibleAmountForCapture(order).compareTo(BigDecimal.ZERO) != 0)) {
                isMultiCapturePossible = isSuccessfulTransactionPresent(order, PaymentTransactionType.AUTHORIZATION) &&
                    !BrainTreeUtils.isIntentSale(intent);
            }
        }
        return isMultiCapturePossible;
    }

    private String getOrderIntent(final OrderModel order) {
        return ((BrainTreePaymentInfoModel) order.getPaymentInfo()).getPayPalIntent();
    }

    @Override
    public List<PaymentTransactionEntryModel> getMultiCaptureableTransactions(final OrderModel order) {
        List<PaymentTransactionEntryModel> result = new ArrayList<>();

        for (PaymentTransactionModel transaction : order.getPaymentTransactions()) {
            for (PaymentTransactionEntryModel paymentEntry : transaction.getEntries()) {
                if (PaymentTransactionType.AUTHORIZATION.equals(paymentEntry.getType())
                    && TransactionStatus.ACCEPTED.name().equals(paymentEntry.getTransactionStatus())) {
                    result.add(paymentEntry);
                }
            }
        }

        return result;
    }

    @Override
    public void executeMultiCapture(final PaymentTransactionEntryModel transaction) throws BraintreeErrorException {
        LOG.warn("brainTreePaymentService: " + brainTreePaymentService);
    }

    private boolean isSuccessfulTransactionPresent(final OrderModel order,
        final PaymentTransactionType transactionType) {
        for (PaymentTransactionModel paymentTransaction : order.getPaymentTransactions()) {
            for (PaymentTransactionEntryModel transactionEntry : paymentTransaction.getEntries()) {
                if ((transactionType.equals(transactionEntry.getType()) && transactionEntry
                    .getTransactionStatusDetails().startsWith("SUCCESFULL"))
                    && TransactionStatus.ACCEPTED.name().equals(transactionEntry.getTransactionStatus())) {
                    return true;
                }
            }
        }
        return false;
    }

    @Override
    public PaymentTransactionEntryData partialCapture(OrderModel orderModel, BigDecimal captureAmount,
        final String authorizeTransactionID) throws BraintreeErrorException {
        return ordermanagementPaymentTransactionEntryConverter
            .convert(braintreePartialCaptureService.partialCapture(orderModel, captureAmount, authorizeTransactionID));
    }

    @Override
    public BigDecimal getPossibleAmountForCapture(OrderModel orderModel) {
        return braintreePartialCaptureService.getPossibleAmountForCapture(orderModel);
    }

    @Override
    public boolean isPartialCapturePossible(OrderModel orderModel) {
        return braintreePartialCaptureService.isPartialCapturePossible(orderModel);
    }

    /**
     * This method is used to get Transaction Amount
     *
     * @param currentModel current model
     * @param orderModel   order model
     * @return transaction amount
     */
    public BigDecimal getTransactionAmount(final PaymentTransactionEntryModel currentModel,
        final OrderModel orderModel) {
        return braintreePartialCaptureService.getTransactionAmount(currentModel, orderModel);
    }

    @Override
    public PaymentTransactionEntryData submitForSettlement(OrderModel orderModel, BigDecimal amount,
        String authorizeTransactionID)
        throws BraintreeErrorException {
        return ordermanagementPaymentTransactionEntryConverter
            .convert(
                braintreeSubmitForSettlementService.submitForSettlement(orderModel, amount, authorizeTransactionID));
    }

    @Override
    public BigDecimal getAuthorizationAmountByTransactionId(OrderModel orderModel, String transactionId) {
        List<PaymentTransactionEntryModel> multiCaptureableTransactions = getMultiCaptureableTransactions(orderModel);
        PaymentTransactionEntryModel authorizeTransaction = null;
        for (PaymentTransactionEntryModel transaction : multiCaptureableTransactions) {
            if (transaction.getRequestId().equals(transactionId)) {
                authorizeTransaction = transaction;
            }
        }
        return Optional.ofNullable(authorizeTransaction).map(PaymentTransactionEntryModel::getAmount)
            .orElse(BigDecimal.ZERO);
    }

    @Override
    public boolean isSubmitForSettlementAvailable(OrderModel orderModel) {
        return braintreeSubmitForSettlementService.isSubmitForSettlementAvailable(orderModel);
    }

    @Override
    public Predicate<PaymentTransactionEntryModel> getCaptureTransactionTypePredicate() {
        return braintreePartialCaptureService.getCaptureTransactionTypePredicate();
    }

    @Override
    public Predicate<PaymentTransactionEntryModel> getAuthorizeTransactionTypePredicate() {
        return braintreePartialCaptureService.getAuthorizeTransactionTypePredicate();
    }

    @Override
    public Predicate<PaymentTransactionEntryModel> getRefundTransactionTypePredicate() {
        return braintreePartialCaptureService.getRefundTransactionTypePredicate();
    }

}
