package com.braintree.customersupportbackoffice.widgets.order.multiplecapture;

import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.exceptions.BraintreeErrorException;
import com.braintree.facade.backoffice.BraintreeBackofficeMultiCaptureFacade;
import com.braintree.facade.impl.DefaultBrainTreeCheckoutFacade;
import com.braintree.transaction.service.BrainTreePaymentTransactionService;
import com.hybris.cockpitng.annotations.SocketEvent;
import com.hybris.cockpitng.annotations.ViewEvent;
import com.hybris.cockpitng.util.DefaultWidgetController;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.ordermanagementfacades.payment.data.PaymentTransactionEntryData;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.zkoss.util.Locales;
import org.zkoss.zk.ui.WrongValueException;
import org.zkoss.zk.ui.select.annotation.Wire;
import org.zkoss.zul.Combobox;
import org.zkoss.zul.Comboitem;
import org.zkoss.zul.Messagebox;
import org.zkoss.zul.Textbox;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.List;

import static com.braintree.customersupportbackoffice.constants.BraintreecustomersupportbackofficeConstants.OrderManagementActions.MULTI_CAPTURE_TEXT_1;
import static com.braintree.customersupportbackoffice.constants.BraintreecustomersupportbackofficeConstants.OrderManagementActions.MULTI_CAPTURE_TEXT_2;
import static com.braintree.customersupportbackoffice.constants.BraintreecustomersupportbackofficeConstants.OrderManagementActions.MULTI_CAPTURE_TEXT_3;
import static com.braintree.customersupportbackoffice.constants.BraintreecustomersupportbackofficeConstants.OrderManagementActions.MULTI_CAPTURE_TITLE;
import static com.braintree.customersupportbackoffice.constants.BraintreecustomersupportbackofficeConstants.OrderManagementActions.PARTIAL_CAPTURE_AMOUNT_NOT_MATH_ERROR;
import static com.braintree.customersupportbackoffice.constants.BraintreecustomersupportbackofficeConstants.OrderManagementActions.PARTIAL_CAPTURE_ERROR;
import static com.braintree.customersupportbackoffice.constants.BraintreecustomersupportbackofficeConstants.OrderManagementActions.PARTIAL_CAPTURE_SUCCESS;
import static com.braintree.customersupportbackoffice.constants.BraintreecustomersupportbackofficeConstants.OrderManagementActions.PARTIAL_CAPTURE_TITLE;

public class BrainTreeMultipleCaptureController extends DefaultWidgetController {

    private static final Logger LOG = Logger.getLogger(BrainTreeMultipleCaptureController.class);

    private static final String IN_SOCKET = "inputObject";
    private static final String OUT_MODIFIED_ITEM = "modifiedItem";
    private static final String ON_OK_EVENT = "onOK";

    private OrderModel order;
    private String transactionId;

    @Wire
    private Textbox orderCode;
    @Wire
    private Textbox customer;
    @Wire
    private Textbox amount;
    @Wire
    private Combobox transactions;

    @Resource
    private BrainTreePaymentTransactionService brainTreePaymentTransactionService;

    @Resource
    private BrainTreeConfigService brainTreeConfigService;

    @Resource
    private DefaultBrainTreeCheckoutFacade brainTreeCheckoutFacade;

    @Autowired
    private BraintreeBackofficeMultiCaptureFacade braintreeBackofficeMultiCaptureFacade;


    @SocketEvent(socketId = IN_SOCKET)
    public void initCreateReturnRequestForm(OrderModel inputOrder) {
        this.setOrder(inputOrder);
        this.getWidgetInstanceManager().setTitle(
            this.getWidgetInstanceManager().getLabel("braintreecustomersupportbackoffice.multiplecapture.confirm.title")
                + " " + this.getOrder().getCode());
        this.orderCode.setValue(this.getOrder().getCode());
        this.customer.setValue(this.getOrder().getUser().getDisplayName());
        configureTransactionsCombobox(inputOrder);
        transactions.setSelectedIndex(0);
        this.amount.setValue(getAmount());
        transactions.addEventListener("onChange", event -> this.amount.setValue(getAmount()));
    }

    private String getAmount() {
        PaymentTransactionEntryModel currentModel = (PaymentTransactionEntryModel) transactions.getSelectedItem()
            .getValue();
        return formatAmount(braintreeBackofficeMultiCaptureFacade.getTransactionAmount(currentModel, this.getOrder()));
    }

    private void configureTransactionsCombobox(final OrderModel inputOrder) {
        List<PaymentTransactionEntryModel> trans = braintreeBackofficeMultiCaptureFacade
            .getMultiCaptureableTransactions(inputOrder);
        LOG.info("MultiCaptureable transactions: " + trans);

        for (PaymentTransactionEntryModel v : trans) {
            Comboitem ci = new Comboitem();
            ci.setValue(v);
            ci.setLabel(v.getRequestId());
            if (!brainTreeConfigService.isStoreInVault()
                && !brainTreeCheckoutFacade.isPayPalCheckout(order)) {
                amount.setValue(v.getAmount().toString());
                amount.setDisabled(true);
            }
            transactions.appendChild(ci);
        }

        if (!transactions.getItems().isEmpty()) {
            transactions.setSelectedItem(transactions.getItems().get(0));
        }
    }

    @ViewEvent(componentID = "multiplecapturerequest", eventName = "onClick")
    public void confirm() {
        if (transactions.getSelectedIndex() != -1) {
            transactionId = transactions.getSelectedItem().getLabel();
        }
        if (!validateAmount()) {
            Messagebox.show(
                getLabel(MULTI_CAPTURE_TEXT_1) + System.lineSeparator() + System.lineSeparator()
                    + getLabel(MULTI_CAPTURE_TEXT_2) + System.lineSeparator() + System.lineSeparator()
                    + getLabel(MULTI_CAPTURE_TEXT_3), getLabel(MULTI_CAPTURE_TITLE),
                Messagebox.OK | Messagebox.CANCEL, Messagebox.ERROR, event -> {
                    if (event.getName().equalsIgnoreCase(ON_OK_EVENT)) {
                        capture();
                    }
                });
            sendOutput(OUT_MODIFIED_ITEM, order);
            return;
        }
        capture();
    }

    private void capture() {
        try {
            if (order != null && brainTreePaymentTransactionService.isValidTransactionId(order, transactionId)) {
                PaymentTransactionEntryData orderCaptureResult = processCapture();
                if (TransactionStatus.ACCEPTED.name().equals(orderCaptureResult.getTransactionStatus())) {
                    Messagebox.show(getLabel(PARTIAL_CAPTURE_SUCCESS), getLabel(PARTIAL_CAPTURE_TITLE), Messagebox.OK,
                        Messagebox.INFORMATION);
                } else {
                    Messagebox
                        .show(getLabel(PARTIAL_CAPTURE_ERROR) + ". " + orderCaptureResult.getTransactionStatusDetails(),
                            getLabel(PARTIAL_CAPTURE_TITLE), Messagebox.OK, Messagebox.ERROR);
                }

            } else {
                Messagebox.show(getLabel(PARTIAL_CAPTURE_AMOUNT_NOT_MATH_ERROR) + " " + formatAmount(
                    getAmountAvailableForMultiCapture()),
                    getLabel(PARTIAL_CAPTURE_TITLE), Messagebox.OK, Messagebox.ERROR);
            }
        } catch (BraintreeErrorException e) {
            Messagebox.show(e.getMessage(), getLabel(PARTIAL_CAPTURE_ERROR), Messagebox.OK, Messagebox.ERROR);
        }
        sendOutput(OUT_MODIFIED_ITEM, order);
    }

    private PaymentTransactionEntryData processCapture() throws BraintreeErrorException {
        final BigDecimal currentAmount = new BigDecimal(this.amount.getValue());
        if (braintreeBackofficeMultiCaptureFacade.isPartialCapturePossible(order)
            && brainTreeCheckoutFacade.isPayPalCheckout(order) && !isCaptureAmountEqualsTotal(currentAmount)) {
            return braintreeBackofficeMultiCaptureFacade.partialCapture(order, currentAmount, transactionId);
        } else {
            if (braintreeBackofficeMultiCaptureFacade.isSubmitForSettlementAvailable(order)) {
                return braintreeBackofficeMultiCaptureFacade.submitForSettlement(order, currentAmount, transactionId);
            }
            if (brainTreePaymentTransactionService.canPerformDelayedCapture(order, currentAmount)) {
                return brainTreeCheckoutFacade.capturePayment(order, currentAmount);
            }

        }
        throw new BraintreeErrorException("Error during order capture process");
    }

    private String formatAmount(final BigDecimal amount) {
        final DecimalFormat decimalFormat = (DecimalFormat) NumberFormat.getNumberInstance(Locales.getCurrent());
        decimalFormat.applyPattern("#0.00");
        return decimalFormat.format(amount);
    }

    private boolean isCaptureAmountEqualsTotal(final BigDecimal amount) {
        return amount.doubleValue() >= order.getTotalPrice();
    }

    private BigDecimal getAmountAvailableForMultiCapture() {
        if (order != null) {
            return braintreeBackofficeMultiCaptureFacade.getPossibleAmountForCapture(order);
        }
        return BigDecimal.ZERO;
    }

    private boolean validateAmount() {
        String value = amount.getValue();
        if (StringUtils.isBlank(value)) {
            throw new WrongValueException(amount, getLabel("bt.customersupport.order.amount.error.empty"));
        }
        try {
            BigDecimal bigDecimalValue = new BigDecimal(value);
            if (BigDecimal.ZERO.equals(new BigDecimal(value))) {
                throw new WrongValueException(amount, getLabel("bt.customersupport.order.amount.error.zero"));
            } else if (braintreeBackofficeMultiCaptureFacade.getAuthorizationAmountByTransactionId(order, transactionId)
                .compareTo(bigDecimalValue) < 0) {
                return false;
            }
        } catch (NumberFormatException e) {
            throw new WrongValueException(amount, getLabel("bt.customersupport.order.amount.error.number.format"));
        }
        return true;
    }

    public OrderModel getOrder() {
        return order;
    }

    public void setOrder(OrderModel order) {
        this.order = order;
    }
}
