package com.braintree.service;

import com.braintree.command.result.BrainTreeAuthorizationResult;
import com.braintree.command.result.BrainTreeRefundTransactionResult;
import com.braintree.command.result.BrainTreeTransactionResult;
import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.enums.BTFraudToolReviewStatus;
import com.braintree.enums.BrainTreePaymentMethod;
import com.braintree.enums.BraintreePaymentTransactionStatus;
import com.braintree.method.impl.DefaultBrainTreePaymentService;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintree.service.impl.DefaultBrainTreeFraudCheckToolService;
import com.braintree.transaction.service.BrainTreeTransactionService;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.model.order.AbstractOrderModel;
import de.hybris.platform.payment.PaymentService;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.model.ModelService;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.ArgumentCaptor;
import org.mockito.Captor;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;

import static org.junit.Assert.assertEquals;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.any;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyZeroInteractions;
import static org.mockito.Mockito.when;

@UnitTest
@RunWith(MockitoJUnitRunner.class)
public class DefaultBrainTreeFraudCheckToolServiceTest {

    private static final String TRANSACTION_ID = "request id";
    private static final String DECLINE = "DECLINE";
    private static final String CREDIT_CARD = "CreditCard";
    private static final String APPROVED = "APPROVED";
    private static final String PAY_PAL = "PayPal";
    private static final String TRANSACTION_GRAPH_QL_ID = "transaction graphQL id";
    private static final BigDecimal transactionAmount = new BigDecimal("12.4");
    private static final String REFUND_ENTRY_CODE = "refund entry code";
    private static final String CANCEL_ENTRY_CODE = "cancel entry code";
    private static final String REFUND_TRANSACTION_ID = "refund transaction id";
    private static final String REFUND_GRAPHQL_TRANSACTION_ID = "refund graphql transaction id";
    @Mock
    private AbstractOrderModel order;
    @Mock
    private ModelService modelService;
    @Mock
    private PaymentTransactionModel paymentTransaction;
    @Mock
    private Map<String, BTFraudToolReviewStatus> statusMapper;
    @Mock
    private List<BrainTreePaymentMethod> paymentMethodsApplicableForBTFraudToolReview;
    @Mock
    private BrainTreeAuthorizationResult brainTreeAuthorizationResult;
    @Mock
    private BrainTreeConfigService brainTreeConfigService;
    @Mock
    private DefaultBrainTreePaymentService brainTreePaymentService;
    @Mock
    private BrainTreeTransactionResult brainTreeTransactionResult;
    @Mock
    private BrainTreeRefundTransactionResult refundTransactionResult;
    @Mock
    private PaymentService paymentService;
    @Mock
    private BrainTreeTransactionService brainTreeTransactionService;
    @Captor
    private ArgumentCaptor<PaymentTransactionEntryModel> entryModelArgumentCaptor;
    @Mock
    private BrainTreePaymentInfoModel paymentInfoModel;

    @InjectMocks
    private DefaultBrainTreeFraudCheckToolService brainTreeFraudCheckToolService;

    @Before
    public void initSetUp() {
        when(paymentTransaction.getRequestId()).thenReturn(TRANSACTION_ID);
        when(paymentTransaction.getOrder()).thenReturn(order);

        when(statusMapper.get(DECLINE)).thenReturn(BTFraudToolReviewStatus.REJECTED);
        when(statusMapper.get(APPROVED)).thenReturn(BTFraudToolReviewStatus.ACCEPTED);

        when(paymentMethodsApplicableForBTFraudToolReview.contains(BrainTreePaymentMethod.CREDITCARD)).thenReturn(true);
        when(paymentMethodsApplicableForBTFraudToolReview.contains(BrainTreePaymentMethod.PAYPAL)).thenReturn(false);

        when(paymentTransaction.getRequestId()).thenReturn(TRANSACTION_ID);
        when(paymentTransaction.getRequestIdGraphQL()).thenReturn(TRANSACTION_GRAPH_QL_ID);
        when(paymentService.getNewPaymentTransactionEntryCode(paymentTransaction, PaymentTransactionType.REFUND_PARTIAL)).thenReturn(REFUND_ENTRY_CODE);
        when(paymentService.getNewPaymentTransactionEntryCode(paymentTransaction, PaymentTransactionType.CANCEL)).thenReturn(CANCEL_ENTRY_CODE);

    }

    @Test
    public void shouldSaveBtFraudToolReviewStatus() {
        brainTreeFraudCheckToolService.saveBtFraudToolReviewStatus(paymentTransaction, BTFraudToolReviewStatus.FOR_REVIEW);

        verify(order).setBtFraudToolReviewStatus(BTFraudToolReviewStatus.FOR_REVIEW);
        verify(paymentTransaction).setBtFraudToolReviewStatus(BTFraudToolReviewStatus.FOR_REVIEW);
        verify(modelService).saveAll(order, paymentTransaction);
        verify(modelService).refresh(order);
        verify(modelService).refresh(paymentTransaction);
    }

    @Test(expected = IllegalArgumentException.class)
    public void shouldThrowExceptionWhenAnyParameterForSaveIsNull() {
        brainTreeFraudCheckToolService.saveBtFraudToolReviewStatus(paymentTransaction, null);
    }

    @Test
    public void shouldTranslateStatusFromBtToHybris() {
        BTFraudToolReviewStatus actual = brainTreeFraudCheckToolService.translateStatusFromBtToHybris(DECLINE);

        assertEquals(BTFraudToolReviewStatus.REJECTED, actual);
    }

    @Test(expected = IllegalArgumentException.class)
    public void shouldThrowExceptionWhenParameterForTranslateIsNull() {
        brainTreeFraudCheckToolService.translateStatusFromBtToHybris(null);
    }

    @Test
    public void shouldUpdateBtFraudToolReviewStatusWhenDecisionIsPresentAndPaymentProviderApplicable() {
        when(paymentTransaction.getOrder()).thenReturn(order);
        when(order.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(paymentInfoModel.getPaymentProvider()).thenReturn(CREDIT_CARD);
        when(brainTreeAuthorizationResult.getBtFraudCheckToolDecision()).thenReturn(APPROVED);

        brainTreeFraudCheckToolService.updateBtFraudToolReviewStatus(paymentTransaction, brainTreeAuthorizationResult);

        verify(order).setBtFraudToolReviewStatus(BTFraudToolReviewStatus.ACCEPTED);
        verify(paymentTransaction).setBtFraudToolReviewStatus(BTFraudToolReviewStatus.ACCEPTED);
        verify(modelService).saveAll(order, paymentTransaction);
        verify(modelService).refresh(paymentTransaction);
        verify(modelService).refresh(order);
    }

    @Test
    public void shouldNotUpdateBtFraudToolReviewStatusWhenDecisionIsNull() {
        when(paymentTransaction.getOrder()).thenReturn(order);
        when(order.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(paymentInfoModel.getPaymentProvider()).thenReturn(CREDIT_CARD);
        when(brainTreeAuthorizationResult.getBtFraudCheckToolDecision()).thenReturn(null);

        brainTreeFraudCheckToolService.updateBtFraudToolReviewStatus(paymentTransaction, brainTreeAuthorizationResult);

        verify(order, never()).setBtFraudToolReviewStatus(any(BTFraudToolReviewStatus.class));
        verify(modelService, never()).save(paymentTransaction);
        verify(modelService, never()).refresh(paymentTransaction);
        verify(paymentTransaction, never()).setBtFraudToolReviewStatus(any(BTFraudToolReviewStatus.class));
        verifyZeroInteractions(modelService);
    }

    @Test
    public void shouldUpdateBtFraudToolReviewStatusWhenDecisionIsNullAndPaymentProviderNotApplicable() {
        when(paymentTransaction.getOrder()).thenReturn(order);
        when(order.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(paymentInfoModel.getPaymentProvider()).thenReturn(PAY_PAL);
        when(brainTreeAuthorizationResult.getBtFraudCheckToolDecision()).thenReturn(null);

        brainTreeFraudCheckToolService.updateBtFraudToolReviewStatus(paymentTransaction, brainTreeAuthorizationResult);

        verify(order).setBtFraudToolReviewStatus(BTFraudToolReviewStatus.NOT_APPLICABLE);
        verify(paymentTransaction).setBtFraudToolReviewStatus(BTFraudToolReviewStatus.NOT_APPLICABLE);
        verify(modelService).saveAll(order, paymentTransaction);
        verify(modelService).refresh(paymentTransaction);
        verify(modelService).refresh(order);
    }

    @Test(expected = IllegalArgumentException.class)
    public void shouldThrowExceptionWhenAnyParameterForUpdateIsNull() {
        brainTreeFraudCheckToolService.updateBtFraudToolReviewStatus(paymentTransaction, null);
    }

    @Test
    public void shouldSuccessfullyUpdateTransactionStatusForGraphQLFlowAndCreateRefundEntry() {
        when(brainTreeConfigService.isBrainTreeGraphQLEnabled()).thenReturn(true);
        when(brainTreePaymentService.findTransactionById(TRANSACTION_GRAPH_QL_ID)).thenReturn(brainTreeTransactionResult);
        when(brainTreeTransactionResult.getStatus()).thenReturn(BraintreePaymentTransactionStatus.SETTLED.getCode());
        when(brainTreeTransactionResult.getRefunds()).thenReturn(List.of(refundTransactionResult));
        when(refundTransactionResult.getAmount()).thenReturn(transactionAmount);
        when(refundTransactionResult.getTransactionId()).thenReturn(REFUND_TRANSACTION_ID);
        when(refundTransactionResult.getTransactionGraphQLId()).thenReturn(REFUND_GRAPHQL_TRANSACTION_ID);
        when(brainTreeTransactionResult.getTotalAmount()).thenReturn(transactionAmount);

        brainTreeFraudCheckToolService.updateTransactionStatus(paymentTransaction, BTFraudToolReviewStatus.REJECTED);

        verify(paymentTransaction).setEntries(any());
        verify(modelService).saveAll(eq(paymentTransaction), entryModelArgumentCaptor.capture());
        verify(modelService).refresh(entryModelArgumentCaptor.getValue());
        verify(modelService).refresh(paymentTransaction);
        verify(brainTreeTransactionService).saveBraintreeTransactionStatus(BraintreePaymentTransactionStatus.REFUNDED.getCode(),
                paymentTransaction);

        PaymentTransactionEntryModel transactionEntry = entryModelArgumentCaptor.getValue();
        assertEquals(TransactionStatus.ACCEPTED.name(), transactionEntry.getTransactionStatus());
        assertEquals(TransactionStatusDetails.SUCCESFULL.name(), transactionEntry.getTransactionStatusDetails());
        assertEquals(PaymentTransactionType.REFUND_PARTIAL, transactionEntry.getType());
        assertEquals(REFUND_TRANSACTION_ID, transactionEntry.getRequestId());
        assertEquals(REFUND_GRAPHQL_TRANSACTION_ID, transactionEntry.getRequestIdGraphQL());
    }

    @Test
    public void shouldSuccessfullyUpdateTransactionStatusForGraphQLFlowAndCreateVoidEntry() {
        when(brainTreeConfigService.isBrainTreeGraphQLEnabled()).thenReturn(true);
        when(brainTreePaymentService.findTransactionById(TRANSACTION_GRAPH_QL_ID)).thenReturn(brainTreeTransactionResult);
        when(brainTreeTransactionResult.getStatus()).thenReturn(BraintreePaymentTransactionStatus.VOIDED.getCode());
        when(brainTreeTransactionResult.getId()).thenReturn(TRANSACTION_ID);
        when(brainTreeTransactionResult.getGraphQLId()).thenReturn(TRANSACTION_GRAPH_QL_ID);

        brainTreeFraudCheckToolService.updateTransactionStatus(paymentTransaction, BTFraudToolReviewStatus.REJECTED);

        verify(paymentTransaction).setEntries(any());
        verify(modelService).saveAll(eq(paymentTransaction), entryModelArgumentCaptor.capture());
        verify(modelService).refresh(entryModelArgumentCaptor.getValue());
        verify(modelService).refresh(paymentTransaction);
        verify(brainTreeTransactionService).saveBraintreeTransactionStatus(BraintreePaymentTransactionStatus.VOIDED.getCode(),
                paymentTransaction);

        PaymentTransactionEntryModel transactionEntry = entryModelArgumentCaptor.getValue();
        assertEquals(TransactionStatus.ACCEPTED.name(), transactionEntry.getTransactionStatus());
        assertEquals(TransactionStatusDetails.SUCCESFULL.name(), transactionEntry.getTransactionStatusDetails());
        assertEquals(PaymentTransactionType.CANCEL, transactionEntry.getType());
        assertEquals(TRANSACTION_ID, transactionEntry.getRequestId());
        assertEquals(TRANSACTION_GRAPH_QL_ID, transactionEntry.getRequestIdGraphQL());
    }

    @Test
    public void shouldFindTransactionByRequestIdForRestFlow() {
        when(brainTreeConfigService.isBrainTreeGraphQLEnabled()).thenReturn(false);
        when(brainTreeTransactionResult.getStatus()).thenReturn(BraintreePaymentTransactionStatus.AUTHORIZED.getCode());
        when(brainTreePaymentService.findTransactionById(TRANSACTION_ID)).thenReturn(brainTreeTransactionResult);

        brainTreeFraudCheckToolService.updateTransactionStatus(paymentTransaction, BTFraudToolReviewStatus.FOR_REVIEW);

        verify(brainTreePaymentService).findTransactionById(TRANSACTION_ID);
    }

    @Test
    public void shouldDoNothingWhenDecisionIsAccept() {
        when(brainTreeConfigService.isBrainTreeGraphQLEnabled()).thenReturn(false);
        when(brainTreePaymentService.findTransactionById(TRANSACTION_ID)).thenReturn(brainTreeTransactionResult);
        when(brainTreeTransactionResult.getStatus()).thenReturn(BraintreePaymentTransactionStatus.SETTLING.getCode());

        brainTreeFraudCheckToolService.updateTransactionStatus(paymentTransaction, BTFraudToolReviewStatus.ACCEPTED);

        verify(paymentTransaction, never()).setEntries(any());
        verify(modelService, never()).saveAll(any(PaymentTransactionModel.class), any(PaymentTransactionEntryModel.class));
        verify(modelService,never()).refresh(any(PaymentTransactionModel.class));
        verify(modelService, never()).refresh(PaymentTransactionEntryModel.class);
        verify(brainTreeTransactionService, never()).saveBraintreeTransactionStatus(any(String.class), any(PaymentTransactionModel.class));
    }

}
