/**
 *
 */
package com.braintree.order.submitForSettlement.service.impl;

import static org.mockito.Matchers.any;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import com.braintree.command.result.BrainTreeSubmitForSettlementTransactionResult;
import com.braintree.exceptions.BraintreeErrorException;
import com.braintree.method.BrainTreePaymentService;
import com.braintree.transaction.service.BrainTreePaymentTransactionService;
import com.braintree.transaction.service.BrainTreeTransactionService;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.user.UserModel;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.model.ModelService;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;

@UnitTest
public class DefaultBraintreeSubmitForSettlementServiceTest {

    private static final BigDecimal AMOUNT = BigDecimal.TEN;
    private static final String TRANSACTION_ID = "testTransactionId";
    private static final String USER_UID = "userUid";

    private OrderModel orderModel;
    private BrainTreeSubmitForSettlementTransactionResult settlementTransactionResult;
    private PaymentTransactionModel paymentTransactionModel;
    private PaymentTransactionEntryModel entryModel;
    private List<PaymentTransactionModel> paymentTransactionModels;

    @Mock
    private BrainTreePaymentTransactionService brainTreePaymentTransactionService;

    @Mock
    private BrainTreeTransactionService brainTreeTransactionService;

    @Mock
    private BrainTreePaymentService brainTreePaymentService;

    @Mock
    private ModelService modelService;

    @InjectMocks
    private DefaultBraintreeSubmitForSettlementService unit;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
        orderModel = Mockito.mock(OrderModel.class);
        UserModel userModel = Mockito.mock(UserModel.class);
        settlementTransactionResult = Mockito.mock(BrainTreeSubmitForSettlementTransactionResult.class);

        when(brainTreePaymentService.submitForSettlementTransaction(any())).thenReturn(settlementTransactionResult);
        when(userModel.getUid()).thenReturn(USER_UID);
        when(orderModel.getUser()).thenReturn(userModel);

        paymentTransactionModel = Mockito.mock(PaymentTransactionModel.class);
        paymentTransactionModels = new ArrayList<>();
        paymentTransactionModels.add(paymentTransactionModel);
        entryModel = Mockito.mock(PaymentTransactionEntryModel.class);
        when(brainTreeTransactionService
            .createPaymentTransaction(paymentTransactionModel, settlementTransactionResult, AMOUNT,
                PaymentTransactionType.CAPTURE, TRANSACTION_ID)).thenReturn(entryModel);
    }

    @Test(expected = BraintreeErrorException.class)
    public void submitForSettlementShouldThrowExceptionForOrderWithoutPaymentTransactions()
        throws BraintreeErrorException {
        unit.submitForSettlement(orderModel, AMOUNT, TRANSACTION_ID);
    }

    @Test
    public void submitForSettlementShouldJustReturnExceptedTransactionEntryIfSettlementTransactionResultNotSuccess()
        throws BraintreeErrorException {
        when(orderModel.getPaymentTransactions()).thenReturn(paymentTransactionModels);
        Assert.assertEquals(entryModel, unit.submitForSettlement(orderModel, AMOUNT, TRANSACTION_ID));
    }

    @Test
    public void submitForSettlementShouldSetOrderStatusAndContinueOrderProcessIfOrderFullyCaptured()
        throws BraintreeErrorException {
        when(orderModel.getPaymentTransactions()).thenReturn(paymentTransactionModels);
        when(settlementTransactionResult.isSuccess()).thenReturn(true);
        when(brainTreePaymentTransactionService.isOrderFullyCaptured(orderModel)).thenReturn(true);
        Assert.assertEquals(entryModel, unit.submitForSettlement(orderModel, AMOUNT, TRANSACTION_ID));
        verify(brainTreePaymentTransactionService).setOrderStatus(orderModel, OrderStatus.PAYMENT_CAPTURED);
        verify(brainTreePaymentTransactionService).continueOrderProcess(orderModel);
    }

    @Test
    public void isSubmitForSettlementAvailableWithNotAvailableOrder() {
        Assert.assertFalse(unit.isSubmitForSettlementAvailable(orderModel));
    }

    @Test
    public void isSubmitForSettlementAvailableWithAvailableOrder() {
        when(orderModel.getPaymentTransactions()).thenReturn(paymentTransactionModels);
        Assert.assertTrue(unit.isSubmitForSettlementAvailable(orderModel));
    }
}
