package com.braintree.order.service.impl;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import com.braintree.enums.BraintreePaymentTransactionStatus;
import com.braintreegateway.CreditCard.CardType;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

@UnitTest
public class DefaultBraintreeOrderBackofficeUtilServiceTest {

    @InjectMocks
    private DefaultBraintreeOrderBackofficeUtilService unit;

    @Mock
    private OrderModel orderModel;
    @Mock
    private PaymentTransactionModel transaction;

    @Mock
    private PaymentTransactionEntryModel transactionEntry1;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
    }

    @Test
    public void isOrderAmountCapturedShouldReturnFalseIfNotCapturedAllAmount() {
        List<PaymentTransactionModel> transactions = new ArrayList<>();
        transactions.add(transaction);
        List<PaymentTransactionEntryModel> entries = new ArrayList<>();
        PaymentTransactionEntryModel entry = mock(PaymentTransactionEntryModel.class);
        entries.add(entry);

        when(orderModel.getPaymentTransactions()).thenReturn(transactions);
        when(transaction.getEntries()).thenReturn(entries);
        when(entry.getType()).thenReturn(PaymentTransactionType.PARTIAL_CAPTURE);
        when(entry.getAmount()).thenReturn(BigDecimal.ONE);
        when(orderModel.getTotalPrice()).thenReturn(BigDecimal.TEN.doubleValue());

        assertFalse(unit.isOrderAmountCaptured(orderModel));
    }

    @Test
    public void isOrderAmountCapturedShouldReturnTrueWhenOrderCaptured() {
        List<PaymentTransactionModel> transactions = new ArrayList<>();
        transactions.add(transaction);
        List<PaymentTransactionEntryModel> entries = new ArrayList<>();
        PaymentTransactionEntryModel entry = mock(PaymentTransactionEntryModel.class);
        entries.add(entry);

        when(orderModel.getPaymentTransactions()).thenReturn(transactions);
        when(transaction.getEntries()).thenReturn(entries);
        when(entry.getType()).thenReturn(PaymentTransactionType.CAPTURE);
        when(entry.getAmount()).thenReturn(BigDecimal.TEN);
        when(orderModel.getTotalPrice()).thenReturn(BigDecimal.TEN.doubleValue());

        assertTrue(unit.isOrderAmountCaptured(orderModel));
    }

    @Test
    public void isOrderVoidedShouldReturnTrueIfVoidedTransactionExist() {
        when(orderModel.getPaymentTransactions()).thenReturn(List.of(transaction));
        when(transaction.getEntries()).thenReturn(List.of(transactionEntry1));
        when(transactionEntry1.getTransactionStatus()).thenReturn(BraintreePaymentTransactionStatus.VOIDED.getCode());

        assertTrue(unit.isOrderVoided(orderModel));
    }

    @Test
    public void isOrderVoidedShouldReturnFalseIfVoidedTransactionNotExist() {
        when(orderModel.getPaymentTransactions()).thenReturn(List.of(transaction));
        when(transaction.getEntries()).thenReturn(List.of(transactionEntry1));
        when(transactionEntry1.getTransactionStatus()).thenReturn(BraintreePaymentTransactionStatus.SUBMITTED_FOR_SETTLEMENT.getCode());

        assertFalse(unit.isOrderVoided(orderModel));
    }

    @Test
    public void paidByCardShouldReturnFalseIfCardTypePropertyNull() {
       PaymentInfoModel paymentInfo = mock(PaymentInfoModel.class);

        when(orderModel.getPaymentInfo()).thenReturn(paymentInfo);
        when(paymentInfo.getProperty("cardType")).thenReturn(null);

        assertFalse(unit.paidByCard(orderModel));
    }

    @Test
    public void paidByCardShouldReturnTrueIfCardTypePropertyNotNull() {
        PaymentInfoModel paymentInfo = mock(PaymentInfoModel.class);

        when(orderModel.getPaymentInfo()).thenReturn(paymentInfo);
        when(paymentInfo.getProperty("cardType")).thenReturn(CardType.MAESTRO);

        assertTrue(unit.paidByCard(orderModel));
    }

}
