/**
 *
 */
package com.braintree.order.refund.partial.services.impl;

import static org.mockito.Matchers.any;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import com.braintree.command.result.BrainTreeRefundTransactionResult;
import com.braintree.exceptions.BraintreeErrorException;
import com.braintree.method.BrainTreePaymentService;
import com.braintree.transaction.service.BrainTreeTransactionService;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.user.UserModel;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.user.UserService;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;

@UnitTest
public class DefaultBraintreePartialRefundServiceImplTest {

    private static final BigDecimal AMOUNT = BigDecimal.TEN;
    private static final String REQUEST_ID = "testId";
    private static final String USER_UID = "testUserUid";

    @Mock
    private ModelService modelService;

    @Mock
    private BrainTreeTransactionService brainTreeTransactionService;

    @Mock
    private BrainTreePaymentService brainTreePaymentService;

    @Mock
    private UserService userService;

    @InjectMocks
    private DefaultBraintreePartialRefundServiceImpl unit;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
        UserModel userModel = Mockito.mock(UserModel.class);
        when(userModel.getUid()).thenReturn(USER_UID);
        when(userService.getCurrentUser()).thenReturn(userModel);
    }

    @Test
    public void partialRefundTransactionShouldSaveEntryModelIfRefundTransactionSuccess() throws BraintreeErrorException {
        OrderModel orderModel = Mockito.mock(OrderModel.class);
        PaymentTransactionEntryModel entryModel = Mockito.mock(PaymentTransactionEntryModel.class);
        prepareMocks(orderModel, entryModel);

        BrainTreeRefundTransactionResult refundTransactionResult = Mockito.mock(BrainTreeRefundTransactionResult.class);
        when(refundTransactionResult.isSuccess()).thenReturn(true);
        when(brainTreePaymentService.refundTransaction(any())).thenReturn(refundTransactionResult);

        unit.partialRefundTransaction(orderModel, entryModel, AMOUNT);
        verify(modelService).save(entryModel);
    }

    @Test
    public void partialRefundTransactionShouldNotSaveEntryModelIfRefundTransactionNotSuccess()
        throws BraintreeErrorException {
        OrderModel orderModel = Mockito.mock(OrderModel.class);
        PaymentTransactionEntryModel entryModel = Mockito.mock(PaymentTransactionEntryModel.class);
        prepareMocks(orderModel, entryModel);

        BrainTreeRefundTransactionResult refundTransactionResult = Mockito.mock(BrainTreeRefundTransactionResult.class);
        when(refundTransactionResult.isSuccess()).thenReturn(false);
        when(brainTreePaymentService.refundTransaction(any())).thenReturn(refundTransactionResult);

        unit.partialRefundTransaction(orderModel, entryModel, AMOUNT);
        verify(modelService, never()).save(entryModel);
    }

    private void prepareMocks(OrderModel orderModel, PaymentTransactionEntryModel entryModel) {
        PaymentTransactionModel transactionModel = Mockito.mock(PaymentTransactionModel.class);
        List<PaymentTransactionModel> transactionModels = new ArrayList<>();
        transactionModels.add(transactionModel);

        PaymentTransactionEntryModel transactionEntryModel = Mockito.mock(PaymentTransactionEntryModel.class);
        List<PaymentTransactionEntryModel> transactionEntryModels = new ArrayList<>();
        transactionEntryModels.add(transactionEntryModel);
        when(transactionEntryModel.getType()).thenReturn(PaymentTransactionType.CAPTURE);
        when(transactionEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(transactionEntryModel.getRequestId()).thenReturn(REQUEST_ID);

        when(transactionModel.getEntries()).thenReturn(transactionEntryModels);
        when(orderModel.getPaymentTransactions()).thenReturn(transactionModels);
        when(entryModel.getType()).thenReturn(PaymentTransactionType.CAPTURE);
        when(entryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(entryModel.getTransactionStatusDetails()).thenReturn(TransactionStatusDetails.SUCCESFULL.name());
        when(entryModel.getRequestId()).thenReturn(REQUEST_ID);
    }

    @Test(expected = IllegalStateException.class)
    public void partialRefundShouldThrowExceptionIfRefundEdibleButTransactionIdNotValid()
        throws BraintreeErrorException {
        OrderModel orderModel = Mockito.mock(OrderModel.class);
        PaymentTransactionEntryModel entryModel = Mockito.mock(PaymentTransactionEntryModel.class);
        when(entryModel.getType()).thenReturn(PaymentTransactionType.CAPTURE);
        when(entryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(entryModel.getTransactionStatusDetails()).thenReturn(TransactionStatusDetails.SUCCESFULL.name());

        unit.partialRefundTransaction(orderModel, entryModel, AMOUNT);
    }

    @Test(expected = IllegalStateException.class)
    public void partialRefundShouldThrowExceptionIfRefundNotEdible() throws BraintreeErrorException {
        OrderModel orderModel = Mockito.mock(OrderModel.class);
        PaymentTransactionEntryModel entryModel = Mockito.mock(PaymentTransactionEntryModel.class);

        unit.partialRefundTransaction(orderModel, entryModel, AMOUNT);
    }
}
