/**
 *
 */
package com.braintree.graphql.commands.impl;

import static org.mockito.Matchers.any;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import com.braintree.command.request.BrainTreeAuthorizationRequest;
import com.braintree.command.result.BrainTreeAuthorizationResult;
import com.braintree.command.result.BrainTreePaymentDetailsResult;
import com.braintree.commands.impl.DefaultBrainTreeLoggingHandler;
import com.braintree.commands.impl.DefaultBraintreeCodeTranslator;
import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.customer.service.BrainTreeCustomerAccountService;
import com.braintree.graphql.commands.request.BrainTreeAddressInput;
import com.braintree.graphql.commands.request.BrainTreeChargeCreditCardInput;
import com.braintree.graphql.commands.request.BrainTreeChargePaymentMethodInput;
import com.braintree.graphql.commands.request.BrainTreeChargeVenmoAccountInput;
import com.braintree.graphql.commands.request.BrainTreeRiskDataInput;
import com.braintree.graphql.commands.request.BrainTreeTransactionDescriptorInput;
import com.braintree.graphql.commands.request.BrainTreeTransactionInput;
import com.braintree.graphql.commands.request.BrainTreeTransactionShippingInput;
import com.braintree.graphql.commands.response.BrainTreePaymentMethod;
import com.braintree.method.BrainTreePaymentService;
import com.braintree.payment.info.service.BraintreePaymentInfoService;
import com.braintree.util.BrainTreeUtils;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.payment.commands.result.AuthorizationResult;
import de.hybris.platform.payment.dto.AvsStatus;
import de.hybris.platform.payment.dto.BillingInfo;
import de.hybris.platform.payment.dto.CvnStatus;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import de.hybris.platform.servicelayer.config.ConfigurationService;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import java.io.IOException;
import java.math.BigDecimal;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Currency;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.TimeZone;
import org.apache.commons.configuration.Configuration;
import org.apache.commons.lang.StringUtils;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

@UnitTest
public class DefaultBrainTreeGraphQLAuthorizationCommandTest {

    private static final String CUSTOMER_ID = "123456";
    private static final String CREDIT_CARD_STATEMENT_NAME = "statement_name";
    private static final String PURCHASE_ORDER_NUMBER = "111111";
    private static final String SHIPS_FROM_POSTAL_CODE = "ships_from_postal_code";
    private static final String SHIPPING_POSTAL_CODE = "78072";
    private static final String METHOD_NONCE = "method_nonce";
    private static final String VENMO_PROFILE_ID = "696996";
    private static final String CREDIT_CARD = "CreditCard";
    private static final String PAYPAL_ACCOUNT = "PayPalAccount";
    private static final String BILLING_LEGACY_ID = "5pyb6wm";
    private static final String TRANSACTION_LEGACY_ID = "1re22aqj";
    private static final String BILLING_ID = "cF5bWVudG1ldGhvZF9wcF81cHliNndt";
    private static final String TRANSACTION_ID = "dHJhbnNhY3Rpb25fMXJlMjJhcWo";
    private static final String AMOUNT = "12.44";
    private static final String CURRENCY_CODE = "USD";
    private static final String PAYPAL_TRANSACTION_DETAILS = "PayPalTransactionDetails";
    private static final String MERCHANT_ACCOUNT_ID = "epam";
    private static final String STATUS_SETTLING = "SETTLING";
    private static final String ERROR_MESSAGE = "Unknown or expired single-use payment method.";
    private static final String ERROR_FULL_MESSAGE = "[BrainTree Validation Error] [code = 91565]";
    private static final String ERROR_LEGACY_CODE = "91565";
    private final Calendar CREATED_AT = new GregorianCalendar();

    private static final String DEFINITION_FILE_NAME_CHARGE_VENMO = "chargeVenmoAccount";
    private static final String DEFINITION_FILE_NAME_CHARGE = "chargePaymentMethod";
    private static final String DEFINITION_FILE_NAME_AUTHORIZATION_VENMO = "authorizeVenmoAccount";
    private static final String DEFINITION_FILE_NAME_AUTHORIZATION = "authorizePaymentMethod";


    @Mock
    private Configuration configuration;

    @Mock
    private BrainTreeCustomerAccountService brainTreeCustomerAccountService;

    @Mock
    private BrainTreeConfigService brainTreeConfigService;

    @Mock
    private BraintreePaymentInfoService paymentInfoService;

    @Mock
    protected ConfigurationService configurationService;

    @Mock
    private BrainTreePaymentService brainTreePaymentService;

    @Mock
    private DefaultBrainTreeLoggingHandler loggingHandler;

    @Mock
    private DefaultBraintreeCodeTranslator codeTranslator;

    @Mock
    private Converter<BillingInfo, BrainTreeAddressInput> braintreeBillingInfoToGraphQLAddressInputConverter;

    @Mock
    private Converter<BrainTreePaymentMethod, BrainTreePaymentDetailsResult> graphQLPaymentMethodToPaymentDetailsConverter;

    @Spy
    @InjectMocks
    DefaultBrainTreeGraphQLAuthorizationCommand command;

    @Before
    public void setUp() throws IOException {

        MockitoAnnotations.initMocks(this);
        command.initMethod();

        Map<String, Object> result = new HashMap<>();
        result.put("test", "test");

        when(configurationService.getConfiguration()).thenReturn(configuration);

        doReturn(result).when(command).makeGraphQlCall(any(), any());

        doReturn(StringUtils.EMPTY).when(command).parseToJson(any());
    }

    @Test
    public void perform() throws IOException {

        BrainTreeAuthorizationRequest request = Mockito.mock(BrainTreeAuthorizationRequest.class);

        doReturn(StringUtils.EMPTY).when(command).parseToJson(any());

        command.perform(request);

    }

    @Test
    public void useChargeVenmoAccountFile() throws IOException {

        BrainTreeAuthorizationRequest request = Mockito.mock(BrainTreeAuthorizationRequest.class);

        when(request.getPaymentType()).thenReturn(com.braintree.enums.BrainTreePaymentMethod.VENMOACCOUNT.getCode());
        when(request.getSubmitForSettlement()).thenReturn(true);

        command.perform(request);

        verify(command).makeGraphQlCall(eq(DEFINITION_FILE_NAME_CHARGE_VENMO), any());

    }

    @Test
    public void useAuthorizeVenmoAccountFile() throws IOException {

        BrainTreeAuthorizationRequest request = Mockito.mock(BrainTreeAuthorizationRequest.class);

        when(request.getPaymentType()).thenReturn(com.braintree.enums.BrainTreePaymentMethod.VENMOACCOUNT.getCode());
        when(request.getSubmitForSettlement()).thenReturn(false);

        command.perform(request);

        verify(command).makeGraphQlCall(eq(DEFINITION_FILE_NAME_AUTHORIZATION_VENMO), any());

    }

    @Test
    public void useChargePayPalAccountFile() throws IOException {

        BrainTreeAuthorizationRequest request = Mockito.mock(BrainTreeAuthorizationRequest.class);

        when(request.getSubmitForSettlement()).thenReturn(true);

        command.perform(request);

        verify(command).makeGraphQlCall(eq(DEFINITION_FILE_NAME_CHARGE), any());

    }

    @Test
    public void useAuthorizePayPalAccountFile() throws IOException {

        BrainTreeAuthorizationRequest request = Mockito.mock(BrainTreeAuthorizationRequest.class);

        when(request.getSubmitForSettlement()).thenReturn(false);

        command.perform(request);

        verify(command).makeGraphQlCall(eq(DEFINITION_FILE_NAME_AUTHORIZATION), any());

    }

    @Test
    public void checkVariablesMapForVenmo() throws IOException {

        BrainTreeAuthorizationRequest request = Mockito.mock(BrainTreeAuthorizationRequest.class);

        Map<String, Object> variables = new HashMap<>();

        when(request.getCustomerId()).thenReturn(CUSTOMER_ID);
        when(request.getTotalAmount()).thenReturn(BigDecimal.valueOf(10));
        when(request.getTaxAmountAuthorize()).thenReturn(10.0);
        when(request.getPurchaseOrderNumber()).thenReturn(PURCHASE_ORDER_NUMBER);
        when(request.getShippingAmount()).thenReturn(2.5);
        when(request.getDiscountAmount()).thenReturn(0.5);
        when(request.getShipsFromPostalCode()).thenReturn(SHIPS_FROM_POSTAL_CODE);
        when(request.getShippingPostalCode()).thenReturn(SHIPPING_POSTAL_CODE);

        when(request.getUsePaymentMethodToken()).thenReturn(false);
        when(request.getMethodNonce()).thenReturn(METHOD_NONCE);
        when(request.getPaymentType()).thenReturn(com.braintree.enums.BrainTreePaymentMethod.VENMOACCOUNT.getCode());
        when(request.getVenmoProfileId()).thenReturn(VENMO_PROFILE_ID);

        BrainTreeChargePaymentMethodInput input = createBaseInput(new BrainTreeChargeVenmoAccountInput(), request);

        variables.put("input", input);

        command.perform(request);

        verify(command).makeGraphQlCall(any(), eq(variables));

    }

    @Test
    public void checkVariablesMapForPayPal() throws IOException {

        BrainTreeAuthorizationRequest request = Mockito.mock(BrainTreeAuthorizationRequest.class);

        Map<String, Object> variables = new HashMap<>();

        when(request.getCustomerId()).thenReturn(CUSTOMER_ID);
        when(request.getTotalAmount()).thenReturn(BigDecimal.valueOf(10));
        when(request.getTaxAmountAuthorize()).thenReturn(10.0);
        when(request.getPurchaseOrderNumber()).thenReturn(PURCHASE_ORDER_NUMBER);
        when(request.getShippingAmount()).thenReturn(2.5);
        when(request.getDiscountAmount()).thenReturn(0.5);
        when(request.getShipsFromPostalCode()).thenReturn(SHIPS_FROM_POSTAL_CODE);
        when(request.getShippingPostalCode()).thenReturn(SHIPPING_POSTAL_CODE);

        when(request.getUsePaymentMethodToken()).thenReturn(false);
        when(request.getMethodNonce()).thenReturn(METHOD_NONCE);
        when(request.getPaymentType())
            .thenReturn(com.braintree.enums.BrainTreePaymentMethod.BRAINTREEPAYPALEXPRESS.getCode());
        when(request.isStoreInVault()).thenReturn(false);
        when(request.getAdvancedFraudTools()).thenReturn(true);

        BrainTreeChargePaymentMethodInput input = createBaseInput(new BrainTreeChargePaymentMethodInput(), request);

        variables.put("input", input);

        command.perform(request);

        verify(command).makeGraphQlCall(any(), eq(variables));

    }

    @Test
    public void checkVariablesMapForCreditCard() throws IOException {

        BrainTreeAuthorizationRequest request = Mockito.mock(BrainTreeAuthorizationRequest.class);

        Map<String, Object> variables = new HashMap<>();

        when(request.getCustomerId()).thenReturn(CUSTOMER_ID);
        when(request.getTotalAmount()).thenReturn(BigDecimal.valueOf(10));
        when(request.getTaxAmountAuthorize()).thenReturn(10.0);
        when(request.getPurchaseOrderNumber()).thenReturn(PURCHASE_ORDER_NUMBER);
        when(request.getShippingAmount()).thenReturn(2.5);
        when(request.getDiscountAmount()).thenReturn(0.5);
        when(request.getShipsFromPostalCode()).thenReturn(SHIPS_FROM_POSTAL_CODE);
        when(request.getShippingPostalCode()).thenReturn(SHIPPING_POSTAL_CODE);

        when(request.getUsePaymentMethodToken()).thenReturn(false);
        when(request.getMethodNonce()).thenReturn(METHOD_NONCE);
        when(request.getPaymentType()).thenReturn(CREDIT_CARD);
        when(request.getAdvancedFraudTools()).thenReturn(true);
        when(request.getCreditCardStatementName()).thenReturn(CREDIT_CARD_STATEMENT_NAME);

        BrainTreeChargePaymentMethodInput input = createBaseInput(new BrainTreeChargeCreditCardInput(), request);

        variables.put("input", input);

        command.perform(request);

        verify(command).makeGraphQlCall(any(), eq(variables));

    }

    @Test
    public void checkSuccessResult() throws IOException {
        BrainTreeAuthorizationRequest request = Mockito.mock(BrainTreeAuthorizationRequest.class);
        BrainTreePaymentDetailsResult paymentDetailsResult = new BrainTreePaymentDetailsResult();

        when(request.getUsePaymentMethodToken()).thenReturn(false);
        when(request.getMethodNonce()).thenReturn(METHOD_NONCE);
        when(request.getPaymentType())
            .thenReturn(com.braintree.enums.BrainTreePaymentMethod.BRAINTREEPAYPALEXPRESS.getCode());
        when(request.isStoreInVault()).thenReturn(false);
        when(request.getAdvancedFraudTools()).thenReturn(true);
        when(request.getSubmitForSettlement()).thenReturn(true);

        Map<String, Object> value = createSuccessMap();

        Map<String, Object> result = new HashMap<>();
        result.put("data", value);

        when(configurationService.getConfiguration()).thenReturn(configuration);

        doReturn(result).when(command).makeGraphQlCall(any(), any());
        doReturn(paymentDetailsResult).when(graphQLPaymentMethodToPaymentDetailsConverter).convert(any());

        AuthorizationResult expectedAuthorizationResult = createSuccessResult(paymentDetailsResult, request);

        AuthorizationResult actualAuthorizationResult = command.perform(request);

        Assert.assertEquals(expectedAuthorizationResult, actualAuthorizationResult);

    }

    @Test
    public void checkFailedResult() throws IOException {
        BrainTreeAuthorizationRequest request = Mockito.mock(BrainTreeAuthorizationRequest.class);

        when(request.getUsePaymentMethodToken()).thenReturn(false);
        when(request.getMethodNonce()).thenReturn(METHOD_NONCE);
        when(request.getPaymentType())
            .thenReturn(com.braintree.enums.BrainTreePaymentMethod.BRAINTREEPAYPALEXPRESS.getCode());
        when(request.isStoreInVault()).thenReturn(false);
        when(request.getAdvancedFraudTools()).thenReturn(true);
        when(request.getSubmitForSettlement()).thenReturn(true);

        ArrayList<Map<String, Object>> mapErrors = createFailedMap();
        Map<String, Object> chargePayPalAccount = new LinkedHashMap<>();
        chargePayPalAccount.put("chargePayPalAccount", null);

        Map<String, Object> result = new HashMap<>();
        result.put("errors", mapErrors);
        result.put("data", chargePayPalAccount);

        when(configurationService.getConfiguration()).thenReturn(configuration);

        doReturn(result).when(command).makeGraphQlCall(any(), any());
        doReturn(TransactionStatusDetails.UNKNOWN_CODE).when(codeTranslator).translateReasonCode(ERROR_LEGACY_CODE);
        doReturn(ERROR_FULL_MESSAGE).when(loggingHandler).handleErrors(any(), eq(true));

        AuthorizationResult expectedAuthorizationResult = createFailedResult();

        AuthorizationResult actualAuthorizationResult = command.perform(request);

        Assert.assertEquals(expectedAuthorizationResult, actualAuthorizationResult);

    }

    private BrainTreeChargePaymentMethodInput createBaseInput(BrainTreeChargePaymentMethodInput input,
        BrainTreeAuthorizationRequest request) {
        BrainTreeTransactionInput transactionInput = new BrainTreeTransactionInput();
        BrainTreeTransactionShippingInput shippingInput = new BrainTreeTransactionShippingInput();
        BrainTreeRiskDataInput riskData = new BrainTreeRiskDataInput();
        BrainTreeTransactionDescriptorInput descriptor = new BrainTreeTransactionDescriptorInput();

        transactionInput.setAmount(request.getTotalAmount());
        transactionInput.setShipping(shippingInput);

        input.setPaymentMethodId(request.getMethodNonce());
        input.setTransaction(transactionInput);

        if (BrainTreeUtils.isPayPalPayment(request.getPaymentType())) {
            riskData.setDeviceData(request.getDeviceData());
            input.getTransaction().setRiskData(riskData);
        }

        if (CREDIT_CARD.equals(request.getPaymentType())) {
            riskData.setDeviceData(request.getDeviceData());
            descriptor.setName(request.getCreditCardStatementName());
            input.getTransaction().setRiskData(riskData);
            input.getTransaction().setDescriptor(descriptor);
        }

        return input;
    }

    private Map<String, Object> createSuccessMap() {
        Map<String, Object> value = new LinkedHashMap<>();
        Map<String, Object> chargePayPalAccount = new LinkedHashMap<>();
        Map<String, Object> transaction = new LinkedHashMap<>();
        Map<String, Object> amount = new LinkedHashMap<>();
        Map<String, Object> billingAgreementWithPurchasePaymentMethod = new LinkedHashMap<>();
        Map<String, Object> details = new LinkedHashMap<>();
        Map<String, Object> paymentMethodSnapshot = new LinkedHashMap<>();

        details.put("__typename", PAYPAL_ACCOUNT);
        billingAgreementWithPurchasePaymentMethod.put("details", details);
        billingAgreementWithPurchasePaymentMethod.put("legacyId", BILLING_LEGACY_ID);
        billingAgreementWithPurchasePaymentMethod.put("id", BILLING_ID);

        amount.put("value", AMOUNT);
        amount.put("currencyCode", CURRENCY_CODE);
        paymentMethodSnapshot.put("__typename", PAYPAL_TRANSACTION_DETAILS);
        transaction.put("id", TRANSACTION_ID);
        transaction.put("legacyId", TRANSACTION_LEGACY_ID);
        transaction.put("merchantAccountId", MERCHANT_ACCOUNT_ID);
        transaction.put("createdAt", toISO8601UTC(CREATED_AT.getTime()));
        transaction.put("status", STATUS_SETTLING);
        transaction.put("paymentMethodSnapshot", paymentMethodSnapshot);
        transaction.put("paymentMethod", null);
        transaction.put("amount", amount);

        chargePayPalAccount.put("transaction", transaction);
        chargePayPalAccount.put("billingAgreementWithPurchasePaymentMethod", billingAgreementWithPurchasePaymentMethod);

        value.put("chargePayPalAccount", chargePayPalAccount);

        return value;
    }

    private ArrayList<Map<String, Object>> createFailedMap() {
        ArrayList<Map<String, Object>> mapErrors = new ArrayList<>();
        Map<String, Object> value = new LinkedHashMap<>();
        Map<String, Object> extensions = new LinkedHashMap<>();

        value.put("message", ERROR_MESSAGE);
        extensions.put("legacyCode", ERROR_LEGACY_CODE);
        value.put("extensions", extensions);

        mapErrors.add(value);

        return mapErrors;
    }

    private BrainTreeAuthorizationResult createSuccessResult(BrainTreePaymentDetailsResult paymentDetailsResult,
        BrainTreeAuthorizationRequest request) {
        BrainTreeAuthorizationResult authorizationResult = new BrainTreeAuthorizationResult();

        authorizationResult.setSuccess(true);
        authorizationResult.setPaymentDetails(paymentDetailsResult);
        authorizationResult.setRequestGraphQLId(TRANSACTION_ID);
        authorizationResult.setShouldGetMethodTokenFromBraintree(request.getUsePaymentMethodToken());
        authorizationResult.setRequestBody("");
        authorizationResult.setResponseBody("");
        authorizationResult.setCurrency(Currency.getInstance(CURRENCY_CODE));
        authorizationResult.setTotalAmount(new BigDecimal(AMOUNT));
        authorizationResult.setAvsStatus(AvsStatus.MATCHED);
        authorizationResult.setCvnStatus(CvnStatus.MATCHED);
        authorizationResult.setAuthorizationTime(CREATED_AT.getTime());
        authorizationResult.setMerchantTransactionCode(MERCHANT_ACCOUNT_ID);
        authorizationResult.setRequestId(TRANSACTION_LEGACY_ID);
        authorizationResult.setRequestToken(TRANSACTION_LEGACY_ID);
        authorizationResult.setTransactionStatus(TransactionStatus.ACCEPTED);
        authorizationResult.setTransactionStatusDetails(TransactionStatusDetails.SUCCESFULL);

        return authorizationResult;
    }

    private BrainTreeAuthorizationResult createFailedResult() {
        BrainTreeAuthorizationResult authorizationResult = new BrainTreeAuthorizationResult();

        authorizationResult.setSuccess(false);
        authorizationResult.setShouldGetMethodTokenFromBraintree(false);
        authorizationResult.setRequestBody("");
        authorizationResult.setResponseBody("");
        authorizationResult.setMessage(ERROR_MESSAGE);
        authorizationResult.setAuthorizationTime(CREATED_AT.getTime());
        authorizationResult.setRequestId("N/A");
        authorizationResult.setTransactionStatus(TransactionStatus.REJECTED);
        authorizationResult.setTransactionStatusDetails(TransactionStatusDetails.UNKNOWN_CODE);

        return authorizationResult;
    }

    private static String toISO8601UTC(Date date) {
        TimeZone tz = TimeZone.getTimeZone("UTC");
        DateFormat df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm'Z'");
        df.setTimeZone(tz);
        return df.format(date);
    }

}
