/**
 *
 */
package com.braintree.customer.service;

import static com.braintree.constants.BraintreeConstants.PayPalIntent.PAYPAL_INTENT_SALE;
import static org.junit.Assert.assertEquals;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.inOrder;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import com.braintree.command.result.BrainTreeAddressResult;
import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.customer.dao.BrainTreeCustomerAccountDao;
import com.braintree.method.BrainTreePaymentService;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintreegateway.Address;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commerceservices.customer.DuplicateUidException;
import de.hybris.platform.commerceservices.customer.dao.CustomerAccountDao;
import de.hybris.platform.commerceservices.enums.CustomerType;
import de.hybris.platform.commerceservices.event.DeletedAddressEvent;
import de.hybris.platform.commerceservices.event.SavedAddressEvent;
import de.hybris.platform.core.model.c2l.CountryModel;
import de.hybris.platform.core.model.c2l.RegionModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.servicelayer.event.EventService;
import de.hybris.platform.servicelayer.i18n.CommonI18NService;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.user.UserService;
import de.hybris.platform.site.BaseSiteService;
import de.hybris.platform.store.BaseStoreModel;
import de.hybris.platform.store.services.BaseStoreService;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InOrder;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;


@UnitTest
public class BrainTreeCustomerAccountServiceTest {

    private static final String MERCHANT_ACCOUNT_ID = "ACCOUNT";
    private static final String BRAINTREE_LEGACY_CUSTOMER_ID = "BraintreeCustomerId";
    private static final String BRAINTREE_GRAPHQL_CUSTOMER_ID = "BraintreeCustomerId";
    private static final String ADDRESS_ID = "AddressId";
    private static final String PWD = "pwd";
    private static final String ORDER_GUID = "orderGUID";

    private static final String CODE = "123456";

    @Mock
    private BrainTreeConfigService brainTreeConfigService;

    @Mock
    private ModelService modelService;

    @Mock
    private BrainTreeCustomerAccountDao brainTreeCustomerAccountDao;

    @Mock
    private UserService userService;

    @Mock
    private CustomerAccountDao customerAccountDao;

    @Mock
    private BrainTreePaymentService brainTreePaymentService;

    @Mock
    private BaseStoreService baseStoreService;

    @Mock
    private CommonI18NService commonI18NService;

    @Mock
    private BaseSiteService baseSiteService;

    @Mock
    private EventService eventService;

    @InjectMocks
    private final BrainTreeCustomerAccountService unit = new BrainTreeCustomerAccountService();

    @Before
    public void init() {
        MockitoAnnotations.initMocks(this);
    }

    @Test
    public void shouldGetBrainTreePaymentInfoForCode() {
        final CustomerModel customerModel = mock(CustomerModel.class);
        final BrainTreePaymentInfoModel brainTreePaymentInfoModelExpected = mock(BrainTreePaymentInfoModel.class);

        when(brainTreeCustomerAccountDao.findBrainTreePaymentInfoByCustomer(customerModel, CODE))
            .thenReturn(brainTreePaymentInfoModelExpected);

        final BrainTreePaymentInfoModel brainTreePaymentInfoModelActual = unit
            .getBrainTreePaymentInfoForCode(customerModel, CODE);

        assertEquals(brainTreePaymentInfoModelExpected, brainTreePaymentInfoModelActual);
    }

    @Test
    public void shouldGetCorrectPaymentInfoForCode() {
        final CustomerModel customerModel = mock(CustomerModel.class);
        final BrainTreePaymentInfoModel brainTreePaymentInfoModelExpected = mock(BrainTreePaymentInfoModel.class);

        when(brainTreeCustomerAccountDao.findPaymentInfoByCustomer(customerModel, CODE))
            .thenReturn(brainTreePaymentInfoModelExpected);

        PaymentInfoModel paymentInfoForCode = unit.getPaymentInfoForCode(customerModel, CODE);

        assertEquals(brainTreePaymentInfoModelExpected, paymentInfoForCode);
    }

    @Test
    public void shouldGetBrainTreePaymentInfos() {
        final CustomerModel customerModel = mock(CustomerModel.class);
        final List<BrainTreePaymentInfoModel> brainTreePaymentInfoModelListExpected = new ArrayList<BrainTreePaymentInfoModel>();
        final boolean saved = true;

        when(brainTreeCustomerAccountDao.findBrainTreePaymentInfosByCustomer(customerModel, saved))
            .thenReturn(brainTreePaymentInfoModelListExpected);
        when(brainTreeConfigService.getMerchantAccountIdForCurrentSiteAndCurrency()).thenReturn(MERCHANT_ACCOUNT_ID);

        final List<BrainTreePaymentInfoModel> brainTreePaymentInfoModelListActual = unit
            .getBrainTreePaymentInfos(customerModel, saved);

        assertEquals(brainTreePaymentInfoModelListExpected, brainTreePaymentInfoModelListActual);
    }

    @Test
    public void shouldUnlinkCCPaymentInfo() {
        final CustomerModel customerModel = mock(CustomerModel.class);
        final BrainTreePaymentInfoModel brainTreePaymentInfoModel = mock(BrainTreePaymentInfoModel.class);
        final Collection<PaymentInfoModel> paymentInfoList = new ArrayList<>();
        paymentInfoList.add(brainTreePaymentInfoModel);
        final Collection<PaymentInfoModel> paymentInfoListWithoutBTPaymentModel = new ArrayList<>(paymentInfoList);
        paymentInfoListWithoutBTPaymentModel.remove(brainTreePaymentInfoModel);

        when(customerModel.getPaymentInfos()).thenReturn(paymentInfoList);

        unit.unlinkCCPaymentInfo(customerModel, brainTreePaymentInfoModel);

        final InOrder inOrder = inOrder(customerModel, modelService);
        inOrder.verify(customerModel).setPaymentInfos(paymentInfoListWithoutBTPaymentModel);
        inOrder.verify(modelService).save(customerModel);
        inOrder.verify(modelService).refresh(customerModel);
    }

    @Test(expected = IllegalArgumentException.class)
    public void shouldThrowExceptionWhenNoPaymentInfo() {
        final CustomerModel customerModel = mock(CustomerModel.class);
        final BrainTreePaymentInfoModel brainTreePaymentInfoModel = mock(BrainTreePaymentInfoModel.class);
        final Collection<PaymentInfoModel> paymentInfoList = new ArrayList<>();

        when(customerModel.getPaymentInfos()).thenReturn(paymentInfoList);

        unit.unlinkCCPaymentInfo(customerModel, brainTreePaymentInfoModel);
    }

    @Test
    public void shouldDeleteAddressEntryShouldCallRemoveMethodAndPublishEvent() {
        final CustomerModel customerModel = mock(CustomerModel.class);
        final AddressModel addressModel = mock(AddressModel.class);
        List<AddressModel> addressModels = new ArrayList<>();
        addressModels.add(addressModel);

        when(customerModel.getAddresses()).thenReturn(addressModels);

        CountryModel countryModel = mock(CountryModel.class);
        RegionModel regionModel = mock(RegionModel.class);
        when(addressModel.getBrainTreeAddressId()).thenReturn(BRAINTREE_LEGACY_CUSTOMER_ID);
        when(addressModel.getCountry()).thenReturn(countryModel);
        when(addressModel.getRegion()).thenReturn(regionModel);

        unit.deleteAddressEntry(customerModel, addressModel);
        verify(brainTreePaymentService).removeAddress(any());
        verify(eventService).publishEvent(any(DeletedAddressEvent.class));
    }

    @Test
    public void saveAddressEntryShouldOnlyPublishEvent() {
        final CustomerModel customerModel = mock(CustomerModel.class);
        final AddressModel addressModel = mock(AddressModel.class);

        unit.saveAddressEntry(customerModel, addressModel);
        verify(eventService).publishEvent(any(SavedAddressEvent.class));
        verify(brainTreePaymentService, never()).createAddress(any(), any());
    }


    @Test
    public void saveAddressEntryShouldOnlyPublishEventAndSaveOnBrainTreeSide() {
        final CustomerModel customerModel = mock(CustomerModel.class);
        final AddressModel addressModel = mock(AddressModel.class);
        when(addressModel.getVisibleInAddressBook()).thenReturn(true);
        when(addressModel.getShippingAddress()).thenReturn(true);
        when(brainTreeConfigService.isStoreInVaultForCurrentUser()).thenReturn(Boolean.TRUE);
        BrainTreeAddressResult addressResult = mock(BrainTreeAddressResult.class);
        Address address = mock(Address.class);
        when(address.getId()).thenReturn(ADDRESS_ID);
        when(addressResult.getAddress()).thenReturn(address);
        when(brainTreePaymentService.createAddress(any(), any())).thenReturn(addressResult);

        unit.saveAddressEntry(customerModel, addressModel);
        verify(addressModel).setBrainTreeAddressId(ADDRESS_ID);
        verify(eventService).publishEvent(any(SavedAddressEvent.class));
        verify(brainTreePaymentService).createAddress(any(), any());
    }

    @Test(expected = IllegalArgumentException.class)
    public void deleteBrainTreePaymentInfoShouldThrowExceptionIfPaymentInfoDoNotBelongCustomer() {
        final CustomerModel customerModel = mock(CustomerModel.class);
        final BrainTreePaymentInfoModel paymentInfoModel = mock(BrainTreePaymentInfoModel.class);
        unit.deleteBrainTreePaymentInfo(customerModel, paymentInfoModel);
    }

    @Test
    public void deleteBrainTreePaymentInfo() {
        final CustomerModel customerModel = mock(CustomerModel.class);
        final BrainTreePaymentInfoModel paymentInfoModel = mock(BrainTreePaymentInfoModel.class);
        List<PaymentInfoModel> infoModels = new ArrayList<>();
        infoModels.add(paymentInfoModel);
        when(customerModel.getPaymentInfos()).thenReturn(infoModels);

        unit.deleteBrainTreePaymentInfo(customerModel, paymentInfoModel);
        verify(paymentInfoModel).setSaved(false);
        verify(modelService).save(paymentInfoModel);
    }

    @Test
    public void convertGuestToCustomer() throws DuplicateUidException {

        BaseStoreModel storeModel = mock(BaseStoreModel.class);
        when(baseStoreService.getCurrentBaseStore()).thenReturn(storeModel);

        OrderModel orderModel = mock(OrderModel.class);
        AddressModel deliveryAddress = mock(AddressModel.class);

        CustomerModel customerModel = mock(CustomerModel.class);
        when(customerModel.getBraintreeCustomerId()).thenReturn(BRAINTREE_LEGACY_CUSTOMER_ID);
        when(customerModel.getType()).thenReturn(CustomerType.GUEST);
        when(userService.getCurrentUser()).thenReturn(customerModel);

        when(orderModel.getUser()).thenReturn(customerModel);
        when(orderModel.getDeliveryAddress()).thenReturn(deliveryAddress);
        when(customerAccountDao.findOrderByGUIDAndStore(ORDER_GUID, storeModel, null)).thenReturn(orderModel);

        when(brainTreeConfigService.getIntent()).thenReturn(PAYPAL_INTENT_SALE);
        when(brainTreeConfigService.isStoreInVault()).thenReturn(Boolean.TRUE);

        BrainTreeAddressResult addressResult = mock(BrainTreeAddressResult.class);
        Address address = mock(Address.class);
        when(address.getId()).thenReturn(ADDRESS_ID);
        when(addressResult.getAddress()).thenReturn(address);

        when(brainTreePaymentService.createAddress(any(), any())).thenReturn(addressResult);

        unit.convertGuestToCustomer(PWD, ORDER_GUID);

        verify(deliveryAddress).setBrainTreeAddressId(ADDRESS_ID);
        verify(brainTreePaymentService).createAddress(any(), any());
    }

    @Test
    public void getGraphQLIdForCustomerShouldExchangeAndNotSaveIfCustomerWithThisIdNotExits() {
        CustomerModel customerModel = mock(CustomerModel.class);
        unit.getGraphQLIdForCustomer(BRAINTREE_LEGACY_CUSTOMER_ID);
        verify(brainTreePaymentService).exchangeLegacyIdToGraphQLId(any());
        verify(modelService, never()).save(customerModel);

    }

    @Test
    public void getGraphQLIdForCustomer() {
        CustomerModel customerModel = mock(CustomerModel.class);
        when(brainTreeCustomerAccountDao.findCustomerByBrainTreeCustomerId(BRAINTREE_LEGACY_CUSTOMER_ID))
            .thenReturn(customerModel);
        when(brainTreePaymentService.exchangeLegacyIdToGraphQLId(any())).thenReturn(BRAINTREE_GRAPHQL_CUSTOMER_ID);

        unit.getGraphQLIdForCustomer(BRAINTREE_LEGACY_CUSTOMER_ID);
        verify(brainTreePaymentService).exchangeLegacyIdToGraphQLId(any());
        verify(customerModel).setBraintreeCustomerGraphQLId(BRAINTREE_GRAPHQL_CUSTOMER_ID);
        verify(modelService).save(customerModel);
    }

}
