package com.braintree.commands.impl;

import com.braintree.command.result.BrainTreeRefundTransactionResult;
import com.braintree.command.result.BrainTreeTransactionResult;
import com.braintreegateway.BraintreeGateway;
import com.braintreegateway.Transaction;
import com.braintreegateway.TransactionGateway;
import com.braintreegateway.exceptions.NotFoundException;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import org.assertj.core.util.Lists;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.ArgumentCaptor;
import org.mockito.Captor;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Spy;
import org.mockito.runners.MockitoJUnitRunner;

import java.util.List;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
@RunWith(MockitoJUnitRunner.class)
public class DefaultBrainTreeFindTransactionByIdCommandTest {

    private static final String TRANSACTION_ID = "transactionId";
    private static final String FIRST_ID = "first_id";
    private static final String SECOND_ID = "second_id";
    private static final String UNEXISTING_TRANSACTION_ID = "unexisting_transaction_id";
    @Mock
    private Converter<Transaction, BrainTreeRefundTransactionResult> transactionToBraintreeRefundResultConverter;
    @Mock
    private Converter<Transaction, BrainTreeTransactionResult> transactionToBraintreeTransactionResultConverter;
    @Mock
    private BraintreeGateway gateway;
    @Mock
    private TransactionGateway transactionGateway;
    @Mock
    private Transaction transaction;
    @Mock
    private BrainTreeTransactionResult transactionResult;
    @Mock
    private Transaction firstTransaction;
    @Mock
    private BrainTreeRefundTransactionResult firstTransactionRefundResult;
    @Mock
    private Transaction secondTransaction;
    @Mock
    private BrainTreeRefundTransactionResult secondTransactionRefundResult;
    @Captor
    private ArgumentCaptor<List<BrainTreeRefundTransactionResult>> refundCaptor;
    @InjectMocks
    @Spy
    private DefaultBrainTreeFindTransactionByIdCommand findTransactionByIdCommand;

    @Before
    public void initSetup() {
        doReturn(gateway).when((AbstractCommand) findTransactionByIdCommand).getBraintreeGateway();
        when(gateway.transaction()).thenReturn(transactionGateway);
    }

    @Test
    public void shouldFindTransactionByIdWhenItIsPresent() {
        when(transactionGateway.find(TRANSACTION_ID)).thenReturn(transaction);
        when(transactionToBraintreeTransactionResultConverter.convert(transaction)).thenReturn(transactionResult);
        when(transaction.getRefundIds()).thenReturn(List.of(FIRST_ID, SECOND_ID));
        when(transactionGateway.find(FIRST_ID)).thenReturn(firstTransaction);
        when(transactionGateway.find(SECOND_ID)).thenReturn(secondTransaction);
        when(transactionToBraintreeRefundResultConverter.convert(firstTransaction)).thenReturn(firstTransactionRefundResult);
        when(transactionToBraintreeRefundResultConverter.convert(secondTransaction)).thenReturn(secondTransactionRefundResult);

        BrainTreeTransactionResult actual = findTransactionByIdCommand.perform(TRANSACTION_ID);

        verify(transactionResult).setRefunds(refundCaptor.capture());
        assertTrue(refundCaptor.getValue().containsAll(List.of(firstTransactionRefundResult, secondTransactionRefundResult)));
        assertEquals(transactionResult, actual);
    }

    @Test
    public void shouldNotPopulateRefundsIfThereIsNoOne() {
        when(transactionGateway.find(TRANSACTION_ID)).thenReturn(transaction);
        when(transactionToBraintreeTransactionResultConverter.convert(transaction)).thenReturn(transactionResult);
        when(transaction.getRefundIds()).thenReturn(Lists.emptyList());

        BrainTreeTransactionResult actual = findTransactionByIdCommand.perform(TRANSACTION_ID);

        verify(transactionResult).setRefunds(refundCaptor.capture());
        assertTrue(refundCaptor.getValue().isEmpty());
        assertEquals(transactionResult, actual);
    }

    @Test(expected = NotFoundException.class)
    public void shouldThrowExceptionWhenThereISNoTransactionById() {
        when(transactionGateway.find(UNEXISTING_TRANSACTION_ID)).thenThrow(NotFoundException.class);

        findTransactionByIdCommand.perform(UNEXISTING_TRANSACTION_ID);
    }

}
