/**
 *
 */
package com.braintree.populators;

import com.braintree.converters.utils.BraintreeTransactionConverterUtils;
import com.braintree.enums.BraintreePaymentTransactionStatus;
import com.braintree.graphql.commands.response.BrainTreeCustomer;
import com.braintree.graphql.commands.response.BrainTreeTransaction;
import com.braintree.graphql.commands.response.BrainTreeTransactionConnection;
import com.braintree.graphql.commands.response.BrainTreeTransactionConnectionEdge;
import com.braintree.graphql.commands.response.BrainTreeRefund;
import com.braintree.graphql.commands.response.BrainTreePaymentStatusEvent;
import com.braintree.hybris.data.BrainTreeGraphQLTransactionData;
import com.braintree.hybris.data.BraintreeTransactionData;
import com.braintree.hybris.data.BraintreeTransactionEntryData;
import com.braintree.hybris.data.BraintreeRefundData;
import com.braintree.hybris.data.BraintreeStatusEventData;
import de.hybris.platform.converters.Populator;
import de.hybris.platform.servicelayer.dto.converter.ConversionException;
import javax.xml.bind.DatatypeConverter;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Calendar;

/**
 * Populates BrainTreeGraphQLCustomerData from BrainTreeCustomerConnection.
 */
public class BraintreeGraphQLTransactionPopulator implements
    Populator<BrainTreeTransactionConnection, BrainTreeGraphQLTransactionData> {

    @Override
    public void populate(BrainTreeTransactionConnection source, BrainTreeGraphQLTransactionData target)
        throws ConversionException {

        target.setHasNextPage(source.getPageInfo().isHasNextPage());
        target.setEndCursor(source.getPageInfo().getEndCursor());

        BraintreeTransactionData transactionData = new BraintreeTransactionData();
        List<BraintreeTransactionEntryData> entryDataList = new ArrayList<>();

        for (BrainTreeTransactionConnectionEdge edge : source.getEdges()) {
            BrainTreeTransaction transaction = edge.getNode();
            if (transaction != null) {
                final BraintreeTransactionEntryData entryData = new BraintreeTransactionEntryData();
                entryData.setId(transaction.getLegacyId());
                entryData.setIdGraphQL(transaction.getId());
                entryData.setDate(BraintreeTransactionConverterUtils.formedDate(transaction.getCreatedAt()));
                entryData.setPaymentInfo(
                    BraintreeTransactionConverterUtils.formedPaymentInfo(transaction.getPaymentMethodSnapshot()));
                entryData.setAmount(BraintreeTransactionConverterUtils
                    .formedAmount(new BigDecimal(transaction.getAmount().getValue()),
                        transaction.getAmount().getCurrencyCode()));
                entryData.setCurrencyIsoCode(transaction.getAmount().getCurrencyCode());
                entryData.setTotal(new BigDecimal(transaction.getAmount().getValue()));

                BrainTreeCustomer customer = transaction.getCustomer();
                if (customer != null) {
                    entryData.setCustomer(BraintreeTransactionConverterUtils
                            .formedName(customer.getFirstName(), customer.getLastName()));
                }

                if (transaction.getStatus() != null) {
                    entryData.setStatus(transaction.getStatus().name());
                }

                if (transaction.getRiskData() != null) {
                    entryData.setRiskDecision(transaction.getRiskData().getDecision());
                }
                entryData.setDetails(BraintreeTransactionConverterUtils.convertDetails(transaction));

                if (transaction.getRefunds() != null && !transaction.getRefunds().isEmpty()) {
                    List<BraintreeRefundData> refundeDataList = new ArrayList<>();
                    for (BrainTreeRefund refund : transaction.getRefunds()) {
                        BraintreeRefundData refundData = new BraintreeRefundData();
                        refundData.setRequestIdGraphQL(refund.getId());
                        refundData.setRequestId(refund.getLegacyId());
                        refundeDataList.add(refundData);
                    }
                    entryData.setRefundIds(refundeDataList);
                }

                List<BraintreeStatusEventData> statusEventDataList = new ArrayList<>();
                if (transaction.getStatusHistory() != null) {
                    for (BrainTreePaymentStatusEvent statusHistory: transaction.getStatusHistory()) {
                        BraintreeStatusEventData statusEventData = new BraintreeStatusEventData();
                        statusEventData.setStatus(BraintreePaymentTransactionStatus.valueOf(statusHistory.getStatus().name()));
                        Calendar calendar = DatatypeConverter.parseDateTime(statusHistory.getTimestamp());
                        statusEventData.setTimestamp(calendar.getTime());
                        statusEventDataList.add(statusEventData);
                    }
                }
                entryData.setStatusHistory(statusEventDataList);

                entryDataList.add(entryData);
            }
        }
        transactionData.setTransactionEntries(entryDataList);
        target.setTransactionData(transactionData);
    }
}
