package com.braintree.payment.info.service.impl;

import static com.braintree.constants.BraintreeConstants.APPLE_PAY_ORIGIN_TYPE;
import static com.braintree.constants.BraintreeConstants.CREDIT_CARD_DETAILS;
import static com.braintree.constants.BraintreeConstants.CREDIT_CARD_TRANSACTION_DETAILS;
import static com.braintree.constants.BraintreeConstants.GOOGLE_PAY_ORIGIN_TYPE;
import static com.braintree.constants.BraintreeConstants.InstrumentType.ANDROID_PAY_CARD_PAYMENT_INSTRUMENT_TYPE;
import static com.braintree.constants.BraintreeConstants.InstrumentType.APPLE_PAY_PAYMENT_INSTRUMENT_TYPE;
import static com.braintree.constants.BraintreeConstants.InstrumentType.CREDIT_CARD_PAYMENT_INSTRUMENT_TYPE;
import static com.braintree.constants.BraintreeConstants.InstrumentType.LOCAL_PAYMENT_INSTRUMENT_TYPE;
import static com.braintree.constants.BraintreeConstants.InstrumentType.PAYPAL_PAYMENT_INSTRUMENT_TYPE;
import static com.braintree.constants.BraintreeConstants.InstrumentType.VENMO_PAYMENT_INSTRUMENT_TYPE;
import static com.braintree.constants.BraintreeConstants.InstrumentType.US_BANK_ACCOUNT_PAYMENT_INSTRUMENT_TYPE;
import static com.braintree.constants.BraintreeConstants.InstrumentType.VISA_CHECKOUT_CARD_PAYMENT_INSTRUMENT_TYPE;
import static com.braintree.constants.BraintreeConstants.LOCAL_PAYMENT_DETAILS;
import static com.braintree.constants.BraintreeConstants.PAYPAL_ACCOUNT_DETAILS;
import static com.braintree.constants.BraintreeConstants.US_BANK_ACCOUNT_DETAIL;
import static com.braintree.constants.BraintreeConstants.PAYPAL_TRANSACTION_DETAILS;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_ACCEPTED_PAYMENT_METHODS;
import static com.braintree.constants.BraintreeConstants.TYPE_PAYMENT_METHOD_PARAMETER;
import static com.braintree.constants.BraintreeConstants.VENMO_ACCOUNT_DETAILS;
import static com.braintree.constants.BraintreeConstants.VISA_CHECKOUT_ORIGIN_TYPE;

import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.customer.dao.BrainTreeCustomerAccountDao;
import com.braintree.customer.service.BrainTreeCustomerAccountService;
import com.braintree.enums.BrainTreePaymentMethod;
import com.braintree.enums.ExpirationStatus;
import com.braintree.graphql.commands.request.BrainTreeExchangeLegacyIdRequest;
import com.braintree.graphql.commands.response.BrainTreePaymentMethodSnapshot;
import com.braintree.method.BrainTreePaymentService;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintree.payment.info.dao.BrainTreePaymentInfoDao;
import com.braintree.payment.info.service.BraintreePaymentInfoService;
import com.braintree.util.BrainTreeDuplicateCheckUtils;
import com.braintree.util.BrainTreeUtils;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.core.model.user.TitleModel;
import de.hybris.platform.core.servicelayer.data.SearchPageData;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.search.FlexibleSearchService;
import de.hybris.platform.servicelayer.util.ServicesUtil;

import java.util.List;
import java.util.Optional;
import java.util.UUID;

import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

/**
 * This class is a default implementation of the BraintreePaymentInfoService interface
 */
public class DefaultBraintreePaymentInfoService implements BraintreePaymentInfoService {

    private static final Logger LOG = Logger.getLogger(DefaultBraintreePaymentInfoService.class);

    private BrainTreePaymentInfoDao brainTreePaymentInfoDao;
    private BrainTreeCustomerAccountDao brainTreeCustomerAccountDao;
    private ModelService modelService;
    private BrainTreeConfigService brainTreeConfigService;
    private FlexibleSearchService flexibleSearchService;
    private BrainTreePaymentService brainTreePaymentService;
    private BrainTreeCustomerAccountService brainTreeCustomerAccountService;

    @Override
    public Optional<BrainTreePaymentInfoModel> findByCustomerIdAndCode(String customerId, String code) {
        return brainTreePaymentInfoDao.findByCustomerIdAndCode(customerId, code);
    }

    @Override
    public void remove(final String customerId, final String paymentMethodToken) {
        final BrainTreePaymentInfoModel brainTreePaymentInfoModel = getBrainTreePaymentInfoDao().find(customerId,
            paymentMethodToken);
        if (brainTreePaymentInfoModel != null) {
            getModelService().remove(brainTreePaymentInfoModel);
        }
    }

    @Override
    public void disable(final String customerId, final String paymentMethodToken) {
        final BrainTreePaymentInfoModel brainTreePaymentInfoModel = getBrainTreePaymentInfoDao().find(customerId,
            paymentMethodToken);
        final CustomerModel customerModel = getBrainTreeCustomerAccountDao()
            .findCustomerByBrainTreeCustomerId(customerId);
        if (customerModel != null && brainTreePaymentInfoModel != null
            && customerModel.getPaymentInfos().contains(brainTreePaymentInfoModel)) {
            brainTreePaymentInfoModel.setDuplicate(true);
            getModelService().save(brainTreePaymentInfoModel);
            getModelService().refresh(customerModel);
        } else {
            LOG.info("Payment Info " + brainTreePaymentInfoModel
                + " does not belong to the customer " + customerModel + " and will not be removed.");
        }
    }

    @Override
    public void addToCustomer(final BrainTreePaymentInfoModel paymentInfo) {
        ServicesUtil.validateParameterNotNull(paymentInfo, "paymentInfo must not be null");
        final CustomerModel customerModel = getBrainTreeCustomerAccountDao().findCustomerByBrainTreeCustomerId(
            paymentInfo.getCustomerId());
        if (customerModel != null) {
            paymentInfo.setUser(customerModel);
            paymentInfo.setCode(customerModel.getUid() + "_" + UUID.randomUUID());
            paymentInfo.setSaved(Boolean.TRUE.booleanValue());
            paymentInfo.setUsePaymentMethodToken(Boolean.TRUE);
            paymentInfo.setThreeDSecureConfiguration(getBrainTreeConfigService().is3dSecureConfiguration());
            paymentInfo.setAdvancedFraudTools(getBrainTreeConfigService().isAdvancedFraudTools());
            paymentInfo
                .setIsSkip3dSecureLiabilityResult(getBrainTreeConfigService().isSkip3DSecureLiabilityResult());
            paymentInfo.setCreditCardStatementName(getBrainTreeConfigService().getCreditCardStatementName());
            paymentInfo.setMerchantAccountIdForCurrentSite(getBrainTreeConfigService()
                .getMerchantAccountIdForCurrentSiteAndCurrency());
            paymentInfo.setBrainTreeChannel(getBrainTreeConfigService().getBrainTreeChannel());

            if (paymentInfo.getBillingAddress() != null) {
                final AddressModel billingAddress = paymentInfo.getBillingAddress();
                billingAddress.setOwner(paymentInfo);
                getModelService().save(billingAddress);
            }
            getModelService().save(paymentInfo);
            getModelService().refresh(customerModel);
        }
    }

    @Override
    public void update(final String paymentMethodToken, final BrainTreePaymentInfoModel paymentInfo) {
        ServicesUtil.validateParameterNotNull(paymentInfo, "paymentInfo must not be null");

        update(paymentInfo, null, paymentMethodToken);
    }

    @Override
    public void update(BrainTreePaymentInfoModel paymentInfo, AddressData addressData) {
        ServicesUtil.validateParameterNotNull(paymentInfo, "paymentInfo must not be null");
        final String paymentMethodToken = paymentInfo.getPaymentMethodToken();

        update(paymentInfo, addressData, paymentMethodToken);
    }

    @Override
    public String getGraphQLTokenForPaymentMethod(String legacyToken) {
        final BrainTreePaymentInfoModel paymentInfoModel = getBrainTreePaymentInfoDao().find(legacyToken);
        if (paymentInfoModel != null) {
            String methodGraphQLToken = paymentInfoModel.getPaymentMethodGraphQLToken();
            if (methodGraphQLToken == null) {
                final BrainTreeExchangeLegacyIdRequest request = new BrainTreeExchangeLegacyIdRequest(legacyToken,
                    TYPE_PAYMENT_METHOD_PARAMETER);
                methodGraphQLToken = brainTreePaymentService.exchangeLegacyIdToGraphQLId(request);
                paymentInfoModel.setPaymentMethodGraphQLToken(methodGraphQLToken);
                modelService.save(paymentInfoModel);
            }
            return methodGraphQLToken;
        } else {
            final BrainTreeExchangeLegacyIdRequest request = new BrainTreeExchangeLegacyIdRequest(legacyToken,
                TYPE_PAYMENT_METHOD_PARAMETER);
            return brainTreePaymentService.exchangeLegacyIdToGraphQLId(request);
        }
    }

    @Override
    public String getPaymentInstrumentTypeBySnapshotAndOrigin(BrainTreePaymentMethodSnapshot paymentMethodSnapshot) {
        final String paymentDetailsSnapshot = paymentMethodSnapshot.get__typename();
        String originType = null;
        if (paymentMethodSnapshot.getOrigin() != null) {
            originType = paymentMethodSnapshot.getOrigin().getType();
        } else if (paymentMethodSnapshot.getCreditCard() != null
            && paymentMethodSnapshot.getCreditCard().getOrigin() != null) {
            originType = paymentMethodSnapshot.getCreditCard().getOrigin().getType();
        }
        return setPaymentInstrumentType(paymentDetailsSnapshot, originType);
    }
    @Override
    public void updateAllExpiredStatuses(List<BrainTreePaymentInfoModel> paymentInfoModels){
        paymentInfoModels.forEach(this::updateExpiredStatus);
        modelService.saveAll(paymentInfoModels);
    }

    @Override
    public void updateExpiredStatus(BrainTreePaymentInfoModel paymentInfoModel) {
        final int expireBetween = brainTreeConfigService.getBrainTreeCreditCartExpireBetween();
        final long reminderMonthToExpire = BrainTreeUtils.getReminderMonthToExpire(paymentInfoModel);

        if(reminderMonthToExpire < 0){
            paymentInfoModel.setExpirationStatus(ExpirationStatus.EXPIRED);
        }else if (reminderMonthToExpire < expireBetween) {
            paymentInfoModel.setExpirationStatus(ExpirationStatus.EXPIRE_SOON);
            CustomerModel customerModel = getBrainTreeCustomerAccountDao().findCustomerByBrainTreeCustomerId(paymentInfoModel.getCustomerId());
            customerModel.setCustomerHasNewExpireSoonCard(true);
            getModelService().save(customerModel);
        }else {
            paymentInfoModel.setExpirationStatus(ExpirationStatus.NOT_EXPIRED);
        }
    }

    private String setPaymentInstrumentType(String paymentDetailsSnapshot, String originType) {
        String paymentInstrumentType;
        if (PAYPAL_TRANSACTION_DETAILS.equals(paymentDetailsSnapshot)) {
            paymentInstrumentType = PAYPAL_PAYMENT_INSTRUMENT_TYPE;
        } else if ((CREDIT_CARD_DETAILS.equals(paymentDetailsSnapshot) || (CREDIT_CARD_TRANSACTION_DETAILS
            .equals(paymentDetailsSnapshot)))
            && (originType == null)) {
            paymentInstrumentType = CREDIT_CARD_PAYMENT_INSTRUMENT_TYPE;
        } else if (VENMO_ACCOUNT_DETAILS.equals(paymentDetailsSnapshot)) {
            paymentInstrumentType = VENMO_PAYMENT_INSTRUMENT_TYPE;
        } else if (US_BANK_ACCOUNT_DETAIL.equals(paymentDetailsSnapshot)) {
            paymentInstrumentType = US_BANK_ACCOUNT_PAYMENT_INSTRUMENT_TYPE;
        } else if (GOOGLE_PAY_ORIGIN_TYPE.equals(originType)) {
            paymentInstrumentType = ANDROID_PAY_CARD_PAYMENT_INSTRUMENT_TYPE;
        } else if (VISA_CHECKOUT_ORIGIN_TYPE.equals(originType)) {
            paymentInstrumentType = VISA_CHECKOUT_CARD_PAYMENT_INSTRUMENT_TYPE;
        } else if (APPLE_PAY_ORIGIN_TYPE.equals(originType)) {
            paymentInstrumentType = APPLE_PAY_PAYMENT_INSTRUMENT_TYPE;
        } else if (LOCAL_PAYMENT_DETAILS.equals(paymentDetailsSnapshot)) {
            paymentInstrumentType = LOCAL_PAYMENT_INSTRUMENT_TYPE;
        } else {
            throw new IllegalArgumentException(
                "Can't find payment instrument type for paymentDetailsSnapshot and originType");
        }
        return paymentInstrumentType;
    }

    @Override
    public String getPaymentProviderByDetailsAndOrigin(String detailsType, String originType) {
        String paymentProvider;
        if (PAYPAL_ACCOUNT_DETAILS.equalsIgnoreCase(detailsType)) {
            paymentProvider = BrainTreePaymentMethod.PAYPALACCOUNT.getCode();
        } else if (VENMO_ACCOUNT_DETAILS.equalsIgnoreCase(detailsType)) {
            paymentProvider = BrainTreePaymentMethod.VENMOACCOUNT.getCode();
        } else if (GOOGLE_PAY_ORIGIN_TYPE.equalsIgnoreCase(originType)) {
            paymentProvider = BrainTreePaymentMethod.ANDROIDPAYCARD.getCode();
        } else if (VISA_CHECKOUT_ORIGIN_TYPE.equalsIgnoreCase(originType)) {
            paymentProvider = BrainTreePaymentMethod.VISACHECKOUTCARD.getCode();
        } else if (APPLE_PAY_ORIGIN_TYPE.equalsIgnoreCase(originType)) {
            paymentProvider = BrainTreePaymentMethod.APPLEPAYCARD.getCode();
        } else if (US_BANK_ACCOUNT_DETAIL.equalsIgnoreCase(detailsType)) {
            paymentProvider = BrainTreePaymentMethod.USBANKACCOUNT.getCode();
        } else {
            paymentProvider = BrainTreePaymentMethod.CREDITCARD.getCode();
        }
        return paymentProvider;
    }

    @Override
    public String getImageSourceByPaymentProviderAndCardType(String paymentProvider, String cardType) {
        String imageSource = "";
        if (BrainTreeUtils.isPayPalPayment(paymentProvider)) {
            imageSource = brainTreePaymentService
                .getImageSourceForPaymentMethod(BrainTreePaymentMethod.PAYPAL.getCode(),
                    BRAINTREE_ACCEPTED_PAYMENT_METHODS);
        } else if (BrainTreeUtils.isVenmoPayment(paymentProvider)) {
            imageSource = brainTreePaymentService
                .getImageSourceForPaymentMethod(BrainTreePaymentMethod.VENMOACCOUNT.getCode(),
                    BRAINTREE_ACCEPTED_PAYMENT_METHODS);
        } else if (BrainTreeUtils.isGooglePayment(paymentProvider)) {
            imageSource = brainTreePaymentService
                .getImageSourceForPaymentMethod(BrainTreePaymentMethod.ANDROIDPAYCARD.getCode(),
                    BRAINTREE_ACCEPTED_PAYMENT_METHODS);
        } else if (BrainTreeUtils.isApplePayPayment(paymentProvider)) {
            imageSource = brainTreePaymentService
                .getImageSourceForPaymentMethod(BrainTreePaymentMethod.APPLEPAYCARD.getCode(),
                    BRAINTREE_ACCEPTED_PAYMENT_METHODS);
        } else if (BrainTreeUtils.isUsBankAccountPayment(paymentProvider)) {
            imageSource = brainTreePaymentService
                    .getImageSourceForPaymentMethod(BrainTreePaymentMethod.USBANKACCOUNT.getCode(),
                            BRAINTREE_ACCEPTED_PAYMENT_METHODS);
        } else if (StringUtils.isNotEmpty(cardType)) {
            imageSource = brainTreePaymentService.getImageSourceForPaymentMethod(cardType,
                BRAINTREE_ACCEPTED_PAYMENT_METHODS);
        }
        return imageSource;
    }

    @Override
    public SearchPageData<BrainTreePaymentInfoModel> getAllSavedCreditCard(SearchPageData<BrainTreePaymentInfoModel> searchPageData) {
        return brainTreePaymentInfoDao.findAllSavedCreditCard(searchPageData);
    }

    @Override
    public List<BrainTreePaymentInfoModel> getAllSavedCreditCardByCustomer(String customerId) {
        return brainTreePaymentInfoDao.findAllSavedCreditCardByCustomer(customerId);
    }

    private void update(BrainTreePaymentInfoModel paymentInfo, AddressData addressData, String paymentMethodToken) {
        final BrainTreePaymentInfoModel brainTreePaymentInfoModel = populateBraintreePaymentInfo(paymentInfo,
            paymentMethodToken);
        final AddressModel billingAddress = paymentInfo.getBillingAddress();
        if (brainTreePaymentInfoModel != null && billingAddress != null) {
            updateBillingAddress(brainTreePaymentInfoModel, billingAddress, addressData);
            getModelService().save(brainTreePaymentInfoModel);
            getModelService().refresh(brainTreePaymentInfoModel);
        }
    }

    private BrainTreePaymentInfoModel populateBraintreePaymentInfo(final BrainTreePaymentInfoModel paymentInfo,
        final String originalPaymentMethodToken) {
        final BrainTreePaymentInfoModel brainTreePaymentInfoModel = getBrainTreePaymentInfoDao()
            .find(paymentInfo.getCustomerId(),
                originalPaymentMethodToken);
        if (brainTreePaymentInfoModel != null) {
            brainTreePaymentInfoModel.setPaymentMethodToken(paymentInfo.getPaymentMethodToken());
            brainTreePaymentInfoModel.setCardholderName(paymentInfo.getCardholderName());
            brainTreePaymentInfoModel.setCardNumber(paymentInfo.getCardNumber());
            brainTreePaymentInfoModel.setExpirationMonth(paymentInfo.getExpirationMonth());
            brainTreePaymentInfoModel.setExpirationYear(paymentInfo.getExpirationYear());
            brainTreePaymentInfoModel.setIsDefault(paymentInfo.isIsDefault());
            return brainTreePaymentInfoModel;
        }
        return null;
    }

    private void updateBillingAddress(final BrainTreePaymentInfoModel brainTreePaymentInfoModel,
        final AddressModel billingAddress, final AddressData addressData) {
        if (addressData != null && addressData.getTitleCode() != null) {
            final TitleModel title = new TitleModel();
            title.setCode(addressData.getTitleCode());
            billingAddress.setTitle(getFlexibleSearchService().getModelByExample(title));
        }
        billingAddress.setOwner(brainTreePaymentInfoModel);
        brainTreePaymentInfoModel.setBillingAddress(billingAddress);
        getModelService().save(billingAddress);
    }

    @Override
    public void checkOnDuplicatePaymentMethod(BrainTreePaymentInfoModel paymentInfo, CustomerModel customer, AddressModel address) {
        getDuplicatePaymentMethod(paymentInfo, customer, paymentInfo.getBillingAddress())
                .ifPresent(duplicatePaymentInfo -> {
                    if (BrainTreeUtils.isPayPalPayment(duplicatePaymentInfo.getPaymentProvider()) && paymentInfo.isSaved()) {
                        duplicatePaymentInfo.setDuplicate(true);
                        modelService.save(duplicatePaymentInfo);
                    } else {
                        paymentInfo.setDuplicate(true);
                        modelService.save(paymentInfo);
                    }
                });
    }

    private Optional <BrainTreePaymentInfoModel> getDuplicatePaymentMethod(BrainTreePaymentInfoModel paymentInfo,
                                                                           CustomerModel customer, AddressModel billingAddress) {
        final String paymentProvider = paymentInfo.getPaymentProvider();
        return  brainTreeCustomerAccountService.getBrainTreePaymentInfos(customer, true)
                .stream()
                .filter(p -> p.getPaymentProvider().equals(paymentProvider) && !(p.getPk().equals(paymentInfo.getPk())))
                .filter(payment -> {
                    if (BrainTreeUtils.isPayPalPayment(paymentProvider)
                            || BrainTreeUtils.isVenmoPayment(paymentProvider)) {
                        return BrainTreeDuplicateCheckUtils.emailBasedCheck(payment, paymentInfo);
                    } else if (BrainTreeUtils.isSrcPayment(paymentProvider)
                            || BrainTreeUtils.isGooglePayment(paymentProvider)) {
                        return BrainTreeDuplicateCheckUtils.checkForGoogleAndSrc(payment, paymentInfo);
                    } else if (BrainTreeUtils.isCreditCardPayment(paymentProvider)) {
                        return BrainTreeDuplicateCheckUtils.checkForCreditCard(payment, paymentInfo, billingAddress);
                    } else if (BrainTreeUtils.isUsBankAccountPayment(paymentProvider)) {
                        return BrainTreeDuplicateCheckUtils.checkForUsBankAccount(payment, paymentInfo);
                    } else {
                        return payment.getCardholderName().equals(paymentInfo.getCardholderName());
                    }
                }).findFirst();
    }

    public BrainTreePaymentInfoDao getBrainTreePaymentInfoDao() {
        return brainTreePaymentInfoDao;
    }

    public void setBrainTreePaymentInfoDao(final BrainTreePaymentInfoDao brainTreePaymentInfoDao) {
        this.brainTreePaymentInfoDao = brainTreePaymentInfoDao;
    }

    public ModelService getModelService() {
        return modelService;
    }

    public void setModelService(final ModelService modelService) {
        this.modelService = modelService;
    }

    public BrainTreeCustomerAccountDao getBrainTreeCustomerAccountDao() {
        return brainTreeCustomerAccountDao;
    }

    public void setBrainTreeCustomerAccountDao(final BrainTreeCustomerAccountDao brainTreeCustomerAccountDao) {
        this.brainTreeCustomerAccountDao = brainTreeCustomerAccountDao;
    }

    public BrainTreeConfigService getBrainTreeConfigService() {
        return brainTreeConfigService;
    }

    public void setBrainTreeConfigService(final BrainTreeConfigService brainTreeConfigService) {
        this.brainTreeConfigService = brainTreeConfigService;
    }

    public FlexibleSearchService getFlexibleSearchService() {
        return flexibleSearchService;
    }

    public void setFlexibleSearchService(FlexibleSearchService flexibleSearchService) {
        this.flexibleSearchService = flexibleSearchService;
    }

    public BrainTreePaymentService getBrainTreePaymentService() {
        return brainTreePaymentService;
    }

    public void setBrainTreePaymentService(BrainTreePaymentService brainTreePaymentService) {
        this.brainTreePaymentService = brainTreePaymentService;
    }

    public BrainTreeCustomerAccountService getBrainTreeCustomerAccountService() {
        return brainTreeCustomerAccountService;
    }

    public void setBrainTreeCustomerAccountService(
        BrainTreeCustomerAccountService brainTreeCustomerAccountService) {
        this.brainTreeCustomerAccountService = brainTreeCustomerAccountService;
    }
}
