package com.braintree.order.submitForSettlement.service.impl;

import com.braintree.command.request.BrainTreeSubmitForSettlementTransactionRequest;
import com.braintree.command.result.BrainTreeSubmitForSettlementTransactionResult;
import com.braintree.exceptions.BraintreeErrorException;
import com.braintree.method.BrainTreePaymentService;
import com.braintree.order.submitForSettlement.service.BraintreeSubmitForSettlementService;
import com.braintree.transaction.service.BrainTreePaymentTransactionService;
import com.braintree.transaction.service.BrainTreeTransactionService;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.model.ModelService;
import org.apache.commons.collections.CollectionUtils;

import java.math.BigDecimal;
import java.util.List;

import static de.hybris.platform.servicelayer.util.ServicesUtil.validateParameterNotNull;

/**
 * This class is a default implementation of the BraintreeSubmitForSettlementService interface
 */
public class DefaultBraintreeSubmitForSettlementService implements BraintreeSubmitForSettlementService {

    private static final String ORDER_MODEL_CAN_NOT_BE_NULL_MESSAGE = "orderModel can not be null";

    private BrainTreePaymentService brainTreePaymentService;
    private BrainTreePaymentTransactionService brainTreePaymentTransactionService;
    private BrainTreeTransactionService brainTreeTransactionService;
    private ModelService modelService;

    @Override
    public PaymentTransactionEntryModel submitForSettlement(OrderModel orderModel, BigDecimal amount,
        String authorizeTransactionID)
        throws BraintreeErrorException {
        validateParameterNotNull(orderModel, ORDER_MODEL_CAN_NOT_BE_NULL_MESSAGE);

        BrainTreeSubmitForSettlementTransactionRequest request =
            new BrainTreeSubmitForSettlementTransactionRequest(orderModel.getUser().getUid());
        request.setAmount(amount);
        request.setTransactionId(authorizeTransactionID);
        request.setOrderId(orderModel.getBrainTreeOrderId());

        BrainTreeSubmitForSettlementTransactionResult brainTreeSubmitForSettlementTransactionResult = getBrainTreePaymentService()
            .submitForSettlementTransaction(request);

        PaymentTransactionEntryModel transactionEntry = createTransaction(orderModel,
            brainTreeSubmitForSettlementTransactionResult, amount, authorizeTransactionID);

        if (brainTreeSubmitForSettlementTransactionResult.isSuccess()) {
            if (getBrainTreePaymentTransactionService().isOrderFullyCaptured(orderModel)) {
                getBrainTreePaymentTransactionService().setOrderStatus(orderModel, OrderStatus.PAYMENT_CAPTURED);
                getBrainTreePaymentTransactionService().continueOrderProcess(orderModel);
            } else {
                getBrainTreePaymentTransactionService().setOrderStatus(orderModel, OrderStatus.PARTIAL_CAPTURE);
            }
        }
        brainTreeTransactionService.saveBraintreeTransactionStatus(
                brainTreeSubmitForSettlementTransactionResult.getStatus(), transactionEntry.getPaymentTransaction());
        return transactionEntry;
    }

    @Override
    public boolean isSubmitForSettlementAvailable(OrderModel orderModel) {
        getModelService().refresh(orderModel);
        final List<PaymentTransactionModel> paymentTransactions = orderModel.getPaymentTransactions();
        if (CollectionUtils.isNotEmpty(paymentTransactions)) {
            return paymentTransactions.stream()
                .flatMap(transaction -> transaction.getEntries()
                    .stream())
                .noneMatch(entries -> entries.getType().equals(PaymentTransactionType.CAPTURE));
        }
        return Boolean.FALSE;
    }

    private PaymentTransactionEntryModel createTransaction(OrderModel orderModel,
        BrainTreeSubmitForSettlementTransactionResult brainTreeSaleTransactionResult, final BigDecimal amount,
        final String authorizeTransactionID) throws BraintreeErrorException {
        final List<PaymentTransactionModel> paymentTransactions = orderModel.getPaymentTransactions();
        final PaymentTransactionType transactionType = PaymentTransactionType.CAPTURE;
        if (CollectionUtils.isNotEmpty(paymentTransactions)) {
            PaymentTransactionModel transactionModel = paymentTransactions.iterator().next();
            return getBrainTreeTransactionService().createPaymentTransaction(transactionModel,
                brainTreeSaleTransactionResult, amount, transactionType, authorizeTransactionID);
        }
        throw new BraintreeErrorException("Order doesn't have payment transactions");
    }

    public BrainTreePaymentService getBrainTreePaymentService() {
        return brainTreePaymentService;
    }

    public void setBrainTreePaymentService(BrainTreePaymentService brainTreePaymentService) {
        this.brainTreePaymentService = brainTreePaymentService;
    }

    public BrainTreeTransactionService getBrainTreeTransactionService() {
        return brainTreeTransactionService;
    }

    public void setBrainTreeTransactionService(BrainTreeTransactionService brainTreeTransactionService) {
        this.brainTreeTransactionService = brainTreeTransactionService;
    }

    public BrainTreePaymentTransactionService getBrainTreePaymentTransactionService() {
        return brainTreePaymentTransactionService;
    }

    public void setBrainTreePaymentTransactionService(
        BrainTreePaymentTransactionService brainTreePaymentTransactionService) {
        this.brainTreePaymentTransactionService = brainTreePaymentTransactionService;
    }

    public ModelService getModelService() {
        return modelService;
    }

    public void setModelService(ModelService modelService) {
        this.modelService = modelService;
    }
}
