package com.braintree.graphql.commands.impl;

import static com.braintree.constants.BraintreeConstants.INPUT_PARAMETER;
import static com.braintree.constants.BraintreeConstants.RESULT_DATA;
import static com.braintree.constants.BraintreeConstants.RESULT_ERRORS;
import static de.hybris.platform.servicelayer.util.ServicesUtil.validateParameterNotNullStandardMessage;

import com.braintree.command.request.BrainTreeUpdateCreditCardRequest;
import com.braintree.command.result.BrainTreeUpdateCreditCardBillingAddressResult;
import com.braintree.commands.BrainTreeVerifyCreditCardCommand;
import com.braintree.commands.impl.AbstractCommand;
import com.braintree.graphql.commands.request.BrainTreeCreditCardVerificationOptionsInput;
import com.braintree.graphql.commands.request.BrainTreeVerifyCreditCardInput;
import com.braintree.graphql.commands.response.BrainTreeErrorDefinition;
import com.braintree.graphql.commands.response.BrainTreeVerification;
import com.braintree.graphql.commands.response.BrainTreeVerifyPaymentMethodPayload;
import com.braintree.payment.info.service.BraintreePaymentInfoService;
import com.braintreegateway.exceptions.NotFoundException;
import de.hybris.platform.payment.AdapterException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.log4j.Logger;
import org.apache.commons.lang.StringUtils;

public class DefaultBrainTreeGraphQLVerifyCreditCardCommand extends AbstractCommand implements
        BrainTreeVerifyCreditCardCommand {

    private static final Logger LOG = Logger.getLogger(DefaultBrainTreeGraphQLVerifyCreditCardCommand.class);
    private static final String VERIFIED = "VERIFIED";
    private static final String DEFINITION_FILE_NAME = "verifyCreditCard";
    private static final String MUTATION_NAME = "verifyCreditCard";

    private BraintreePaymentInfoService paymentInfoService;

    @Override
    public BrainTreeUpdateCreditCardBillingAddressResult perform(final BrainTreeUpdateCreditCardRequest request) {
        try {
            validateParameterNotNullStandardMessage("Verify Credit Card Request", request);
            final String graphQLTokenForPaymentMethod = paymentInfoService.getGraphQLTokenForPaymentMethod(
                    request.getToken());
            Map<String, Object> result = makeGraphQlCall(DEFINITION_FILE_NAME,
                    createVariablesMap(request, graphQLTokenForPaymentMethod));
            ArrayList<Map<String, Object>> mapErrors = (ArrayList<Map<String, Object>>) result.get(RESULT_ERRORS);

            if (mapErrors == null) {
                return translateResponse((Map<String, Object>) result.get(RESULT_DATA));
            } else {
                List<BrainTreeErrorDefinition> errors = mapErrors.stream()
                        .map(stringObjectMap -> objectMapper.convertValue(stringObjectMap, BrainTreeErrorDefinition.class))
                        .collect(Collectors.toList());
                getLoggingHandler().handleErrors(errors, true);
                return translateErrorResponse(request.getToken(), errors);
            }
        } catch (final NotFoundException notFoundException) {
            return translateNotFoundResponse(request, notFoundException);
        } catch (final Exception exception) {
            LOG.error("[BT Verify Credit Card] Error during verifying credit card!");
            throw new AdapterException(exception.getMessage(), exception);
        }
    }

    private Map<String, Object> createVariablesMap(final BrainTreeUpdateCreditCardRequest request,
                                                   final String graphQLTokenForPaymentMethod) {
        Map<String, Object> map = new HashMap<>();
        BrainTreeVerifyCreditCardInput input = new BrainTreeVerifyCreditCardInput();
        input.setPaymentMethodId(graphQLTokenForPaymentMethod);

        if (StringUtils.isNotBlank(request.getCvvNonce())) {
            BrainTreeCreditCardVerificationOptionsInput options = new BrainTreeCreditCardVerificationOptionsInput();
            options.setTokenizedCvv(request.getCvvNonce());
            input.setOptions(options);
        }
        map.put(INPUT_PARAMETER, input);
        return map;
    }

    private BrainTreeUpdateCreditCardBillingAddressResult translateResponse(Map<String, Object> data) {
        final BrainTreeUpdateCreditCardBillingAddressResult result = new BrainTreeUpdateCreditCardBillingAddressResult();
        BrainTreeVerifyPaymentMethodPayload payload =
                objectMapper.convertValue(data.get(MUTATION_NAME), BrainTreeVerifyPaymentMethodPayload.class);
        result.setSuccess(false);

        if (payload != null) {
            BrainTreeVerification verification = payload.getVerification();
            if (verification != null && verification.getStatus().equalsIgnoreCase(VERIFIED)) {
                result.setSuccess(true);
            }
        }
        return result;
    }

    private BrainTreeUpdateCreditCardBillingAddressResult translateErrorResponse(final String token,
                                                                                 List<BrainTreeErrorDefinition> errors) {
        final BrainTreeUpdateCreditCardBillingAddressResult brainTreeUpdateBillingAddressResult =
                new BrainTreeUpdateCreditCardBillingAddressResult();
        brainTreeUpdateBillingAddressResult.setSuccess(false);

        if (errors != null && errors.size() > 0) {
            final BrainTreeErrorDefinition validationError = errors.get(0);
            getLoggingHandler().getLogger().info(
                    String.format("BT credit card(credit card token id=%s) verified with error: %s", token,
                            validationError.getMessage()));

            brainTreeUpdateBillingAddressResult.setErrorMessage(validationError.getMessage());
        }

        return brainTreeUpdateBillingAddressResult;
    }

    private BrainTreeUpdateCreditCardBillingAddressResult translateNotFoundResponse(final BrainTreeUpdateCreditCardRequest request,
                                                                                    final NotFoundException notFoundException) {
        getLoggingHandler().getLogger()
                .info(String.format("Credit card with token=%s not Found! Error %s", request.getToken(),
                        notFoundException.getMessage()));

        final BrainTreeUpdateCreditCardBillingAddressResult brainTreeUpdateBillingAddressResult = new BrainTreeUpdateCreditCardBillingAddressResult();
        brainTreeUpdateBillingAddressResult.setSuccess(false);
        brainTreeUpdateBillingAddressResult
                .setErrorMessage(String.format("Credit card with token id=%s not Found!", request.getToken()));
        return brainTreeUpdateBillingAddressResult;
    }

    public void setPaymentInfoService(BraintreePaymentInfoService paymentInfoService) {
        this.paymentInfoService = paymentInfoService;
    }
}