/**
 *
 */
package com.braintree.graphql.commands.impl;

import static com.braintree.constants.BraintreeConstants.INPUT_PARAMETER;
import static com.braintree.constants.BraintreeConstants.RESULT_DATA;
import static com.braintree.constants.BraintreeConstants.RESULT_ERRORS;
import static de.hybris.platform.servicelayer.util.ServicesUtil.validateParameterNotNullStandardMessage;

import com.braintree.command.request.BrainTreeSubmitForSettlementTransactionRequest;
import com.braintree.command.result.BrainTreeSubmitForSettlementTransactionResult;
import com.braintree.commands.BrainTreeSubmitForSettlementCommand;
import com.braintree.commands.impl.AbstractCommand;
import com.braintree.graphql.commands.request.BrainTreeCaptureTransactionInput;
import com.braintree.graphql.commands.request.BrainTreeCaptureTransactionOptionsInput;
import com.braintree.graphql.commands.response.BrainTreeErrorDefinition;
import com.braintree.graphql.commands.response.BrainTreePaymentMethodSnapshot;
import com.braintree.graphql.commands.response.BrainTreeTransaction;
import com.braintree.graphql.commands.response.BrainTreeTransactionPayload;
import com.braintree.payment.info.service.BraintreePaymentInfoService;
import com.braintree.transaction.service.BrainTreeTransactionService;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * This class extends AbstractCommand, implements BrainTreeSubmitForSettlementCommand and is used in GraphQL API.
 */
public class DefaultBrainTreeGraphQLSubmitForSettlementTransactionCommand extends AbstractCommand implements
    BrainTreeSubmitForSettlementCommand {

    private static final String DEFINITION_FILE_NAME = "captureTransaction";
    private static final String MUTATION_NAME = "captureTransaction";

    private BrainTreeTransactionService brainTreeTransactionService;
    private BraintreePaymentInfoService paymentInfoService;

    @Override
    public BrainTreeSubmitForSettlementTransactionResult perform(
        BrainTreeSubmitForSettlementTransactionRequest request) {
        validateParameterNotNullStandardMessage("Submit For Settlement transaction Request", request);
        try {
            Map<String, Object> result = makeGraphQlCall(DEFINITION_FILE_NAME, createVariablesMap(request));
            ArrayList<Map<String, Object>> mapErrors = (ArrayList<Map<String, Object>>) result.get(RESULT_ERRORS);

            BrainTreeSubmitForSettlementTransactionResult braintreeResult;
            if (mapErrors == null) {
                braintreeResult = translateResponse((Map<String, Object>) result.get(RESULT_DATA));
            } else {
                List<BrainTreeErrorDefinition> errors = mapErrors.stream()
                    .map(stringObjectMap -> objectMapper.convertValue(stringObjectMap, BrainTreeErrorDefinition.class))
                    .collect(Collectors.toList());
                braintreeResult = translateErrorResponse(request.getTransactionId(), errors);
            }

            braintreeResult.setRequestBody(parseToJson(createVariablesMap(request)));
            braintreeResult.setResponseBody(parseToJson(result));
            return braintreeResult;
        } catch (final Exception exception) {
            throw new AdapterException(exception.getMessage(), exception);
        }
    }

    private BrainTreeSubmitForSettlementTransactionResult translateResponse(Map<String, Object> data) {
        final BrainTreeSubmitForSettlementTransactionResult result = new BrainTreeSubmitForSettlementTransactionResult(
            true);
        BrainTreeTransactionPayload payload = objectMapper
            .convertValue(data.get(MUTATION_NAME), BrainTreeTransactionPayload.class);
        if (payload != null && payload.getTransaction() != null) {
            BrainTreeTransaction transaction = payload.getTransaction();
            if (transaction.getStatus() != null && !transaction.getStatus().isSuccess()) {
                result.setSuccess(false);
                result.setTransactionStatus(TransactionStatus.REJECTED);
                result.setErrorMessage(getMessageByStatus(transaction.getStatusHistory(), transaction.getStatus()));
                result.setErrorCode(getErrorCodeFromStatusHistory(transaction.getStatusHistory()));
                return result;
            }
            result.setStatus(transaction.getStatus().name());
            result.setTransactionId(transaction.getLegacyId());
            result.setTransactionGraphQLId(transaction.getId());
            result.setCurrencyIsoCode(transaction.getAmount().getCurrencyCode());
            result.setAmount(new BigDecimal(transaction.getAmount().getValue()));
            result.setTransactionStatus(TransactionStatus.ACCEPTED);
            result.setTransactionStatusDetails(TransactionStatusDetails.SUCCESFULL);
            result.setCreatedAt(transaction.getCreatedAt().getTime());

            BrainTreePaymentMethodSnapshot paymentMethodSnapshot = transaction.getPaymentMethodSnapshot();
            result.setPaymentInstrumentType(
                paymentInfoService.getPaymentInstrumentTypeBySnapshotAndOrigin(paymentMethodSnapshot));

            getLoggingHandler().handleResult("[SUBMIT FOR SETTLEMENT TRANSACTION] ", transaction);
        }

        return result;
    }

    private BrainTreeSubmitForSettlementTransactionResult translateErrorResponse(final String transactionId,
        List<BrainTreeErrorDefinition> errors) {
        final BrainTreeSubmitForSettlementTransactionResult response = new BrainTreeSubmitForSettlementTransactionResult(
            false);

        response.setTransactionStatus(TransactionStatus.REJECTED);

        if (errors != null && errors.size() > 0) {
            BrainTreeErrorDefinition errorDefinition = errors.get(0);
            getLoggingHandler().getLogger().info(
                String.format("BT transaction id(%s) submit for settlement with error: %s %s", transactionId,
                    errorDefinition.getExtensions().getLegacyCode(),
                    errorDefinition.getMessage()));

            if (errorDefinition.getExtensions().getLegacyCode() != null) {
                response.setErrorCode(errorDefinition.getExtensions().getLegacyCode());
            }
            response.setErrorMessage(errorDefinition.getMessage());

        }

        return response;
    }

    private Map<String, Object> createVariablesMap(final BrainTreeSubmitForSettlementTransactionRequest request) {
        Map<String, Object> map = new HashMap<>();
        final BrainTreeCaptureTransactionInput input = new BrainTreeCaptureTransactionInput();
        input.setTransactionId(brainTreeTransactionService.getGraphQLIdFromTransaction(request.getTransactionId()));
        final BrainTreeCaptureTransactionOptionsInput transactionOptionsInput = new BrainTreeCaptureTransactionOptionsInput();

        if (request.getOrderId() != null) {
            transactionOptionsInput.setOrderId(request.getOrderId());
        }
        if (request.getAmount() != null) {
            transactionOptionsInput.setAmount(request.getAmount());
        }
        input.setTransaction(transactionOptionsInput);
        map.put(INPUT_PARAMETER, input);
        return map;
    }

    public BrainTreeTransactionService getBrainTreeTransactionService() {
        return brainTreeTransactionService;
    }

    public void setBrainTreeTransactionService(
        BrainTreeTransactionService brainTreeTransactionService) {
        this.brainTreeTransactionService = brainTreeTransactionService;
    }

    public BraintreePaymentInfoService getPaymentInfoService() {
        return paymentInfoService;
    }

    public void setPaymentInfoService(BraintreePaymentInfoService paymentInfoService) {
        this.paymentInfoService = paymentInfoService;
    }
}
