/**
 *
 */
package com.braintree.graphql.commands.impl;

import static com.braintree.constants.BraintreeConstants.INPUT_PARAMETER;
import static com.braintree.constants.BraintreeConstants.RESULT_DATA;
import static com.braintree.constants.BraintreeConstants.RESULT_ERRORS;

import com.braintree.command.request.BrainTreeGenerateClientTokenRequest;
import com.braintree.command.result.BrainTreeGenerateClientTokenResult;
import com.braintree.commands.BrainTreeGenerateClientTokenCommand;
import com.braintree.commands.impl.AbstractCommand;
import com.braintree.commands.impl.DefaultBrainTreeGenerateClientTokenCommand;
import com.braintree.customer.service.BrainTreeCustomerAccountService;
import com.braintree.graphql.commands.request.BrainTreeClientTokenInput;
import com.braintree.graphql.commands.request.BrainTreeCreateClientTokenInput;
import com.braintree.graphql.commands.response.BrainTreeCreateClientTokenPayload;
import de.hybris.platform.payment.AdapterException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

/**
 * This class extends AbstractCommand, implements BrainTreeGenerateClientTokenCommand and is used in GraphQL API.
 */
public class DefaultBrainTreeGraphQLGenerateClientTokenCommand extends AbstractCommand implements
    BrainTreeGenerateClientTokenCommand {

    private static final Logger LOG = Logger.getLogger(DefaultBrainTreeGenerateClientTokenCommand.class);

    private static final String DEFINITION_FILE_NAME = "createClientToken";
    private static final String MUTATION_NAME = "createClientToken";

    private BrainTreeCustomerAccountService brainTreeCustomerAccountService;

    @Override
    public BrainTreeGenerateClientTokenResult perform(
        BrainTreeGenerateClientTokenRequest brainTreeGenerateClientTokenRequest) {
        try {
            Map<String, Object> result = makeGraphQlCall(DEFINITION_FILE_NAME,
                createVariablesMap(brainTreeGenerateClientTokenRequest));
            ArrayList<Map<String, Object>> mapErrors = (ArrayList<Map<String, Object>>) result.get(RESULT_ERRORS);

            if (mapErrors == null) {
                return translateResponse((Map<String, Object>) result.get(RESULT_DATA));
            } else {
                final String errorMessage =
                    "[Generate Token] Error due to creation token. " + getFistErrorMessage(mapErrors);
                LOG.error(errorMessage);
                throw new IllegalArgumentException(errorMessage);
            }

        } catch (final Exception exception) {
            throw new AdapterException(exception.getMessage(), exception);
        }
    }

    private BrainTreeGenerateClientTokenResult translateResponse(Map<String, Object> data) {

        LOG.debug(String.format("[Generate Token] Generated token: %s", data));

        BrainTreeCreateClientTokenPayload payload = objectMapper
            .convertValue(data.get(MUTATION_NAME), BrainTreeCreateClientTokenPayload.class);

        return new BrainTreeGenerateClientTokenResult(payload.getClientToken());
    }

    private Map<String, Object> createVariablesMap(final BrainTreeGenerateClientTokenRequest request) {

        Map<String, Object> map = new HashMap<>();

        final BrainTreeCreateClientTokenInput input = new BrainTreeCreateClientTokenInput();
        final BrainTreeClientTokenInput clientTokenInput = new BrainTreeClientTokenInput();

        if (StringUtils.isNotEmpty(request.getCustomerId())) {
            clientTokenInput
                .setCustomerId(brainTreeCustomerAccountService.getGraphQLIdForCustomer(request.getCustomerId()));
        }
        if (request.getMerchantAccount() != null) {
            clientTokenInput.setMerchantAccountId(request.getMerchantAccount());
        }

        input.setClientToken(clientTokenInput);
        map.put(INPUT_PARAMETER, input);

        LOG.info("clientTokenRequest: " + map);

        return map;
    }

    public BrainTreeCustomerAccountService getBrainTreeCustomerAccountService() {
        return brainTreeCustomerAccountService;
    }

    public void setBrainTreeCustomerAccountService(BrainTreeCustomerAccountService brainTreeCustomerAccountService) {
        this.brainTreeCustomerAccountService = brainTreeCustomerAccountService;
    }
}
