package com.braintree.graphql.commands.impl;

import com.braintree.command.request.BrainTreeCreatePaymentMethodRequest;
import com.braintree.command.result.BrainTreeCreatePaymentMethodResult;
import com.braintree.commands.BrainTreeCreateUsBankAccountPaymentMethodCommand;
import com.braintree.commands.impl.AbstractCommand;
import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.customer.service.BrainTreeCustomerAccountService;
import com.braintree.graphql.commands.request.BrainTreeVaultUsBankAccountInput;
import com.braintree.graphql.commands.response.BrainTreePaymentMethod;
import com.braintree.graphql.commands.response.BrainTreeVaultPaymentMethodPayload;
import com.braintree.graphql.commands.response.BrainTreePaymentMethodOrigin;
import com.braintree.method.BrainTreePaymentService;
import com.braintree.payment.info.service.BraintreePaymentInfoService;
import de.hybris.platform.payment.AdapterException;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import static com.braintree.constants.BraintreeConstants.INPUT_PARAMETER;
import static com.braintree.constants.BraintreeConstants.RESULT_ERRORS;
import static com.braintree.constants.BraintreeConstants.RESULT_DATA;

public class DefaultBrainTreeGraphQLCreateUsBankAccountPaymentMethodCommand extends AbstractCommand implements
        BrainTreeCreateUsBankAccountPaymentMethodCommand {

    private static final Logger LOG = Logger.getLogger(DefaultBrainTreeGraphQLCreateUsBankAccountPaymentMethodCommand.class);

    private static final String VAULT_US_BANK_ACCOUNT = "vaultUsBankAccount";

    private BrainTreeCustomerAccountService brainTreeCustomerAccountService;

    private BrainTreePaymentService brainTreePaymentService;

    private BrainTreeConfigService brainTreeConfigService;

    private BraintreePaymentInfoService braintreePaymentInfoService;

    @Override
    public BrainTreeCreatePaymentMethodResult perform(BrainTreeCreatePaymentMethodRequest brainTreeCreatePaymentMethodRequest) {
        try {
            Map<String, Object> result = makeGraphQlCall(VAULT_US_BANK_ACCOUNT,
                    createVariablesMap(brainTreeCreatePaymentMethodRequest));
            ArrayList<Map<String, Object>> mapErrors = (ArrayList<Map<String, Object>>) result.get(RESULT_ERRORS);

            if (mapErrors == null) {
                return translateResult((Map<String, Object>) result.get(RESULT_DATA));
            } else {
                final String errorMessage =
                        "[Create payment method] Error due to creation of payment method. " + getFistErrorMessage(
                                mapErrors);
                LOG.error(errorMessage);
                throw new IllegalArgumentException();
            }

        } catch (final Exception exception) {
            throw new AdapterException(exception.getMessage(), exception);
        }
    }
    private Map<String, Object> createVariablesMap(final BrainTreeCreatePaymentMethodRequest request) {
        getLoggingHandler().handleCreatePaymentMethodRequest(request);
        Map<String, Object> map = new HashMap<>();

        BrainTreeVaultUsBankAccountInput input = new BrainTreeVaultUsBankAccountInput();
        input.setPaymentMethodId(request.getMethodNonce());
        input.setVerificationMethod(request.getOptions().getUsBankAccountVerificationMethod());
        if (StringUtils.isNotEmpty(request.getCustomerId())) {
            input.setCustomerId(brainTreeCustomerAccountService.getGraphQLIdForCustomer(request.getCustomerId()));
        }

        map.put(INPUT_PARAMETER, input);
        return map;
    }

    private BrainTreeCreatePaymentMethodResult translateResult(Map<String, Object> data) {

        final BrainTreeCreatePaymentMethodResult result = new BrainTreeCreatePaymentMethodResult();

        BrainTreeVaultPaymentMethodPayload payload = objectMapper
                .convertValue(data.get(VAULT_US_BANK_ACCOUNT), BrainTreeVaultPaymentMethodPayload.class);

        BrainTreePaymentMethod paymentMethod = payload.getPaymentMethod();

        result.setSuccess(true);
        result.setVerified(paymentMethod.getDetails().getVerified());
        result.setPaymentMethodToken(paymentMethod.getLegacyId());
        result.setPaymentMethodGraphQLToken(paymentMethod.getId());
        result.setLast4(paymentMethod.getDetails().getLast4());
        result.setRoutingNumber(paymentMethod.getDetails().getRoutingNumber());
        String originType = Optional.ofNullable(paymentMethod.getDetails().getOrigin()).map(BrainTreePaymentMethodOrigin::getType)
                .orElse(null);
        String paymentProvider = braintreePaymentInfoService
                .getPaymentProviderByDetailsAndOrigin(paymentMethod.getDetails().get__typename(), originType);
        result.setPaymentProvider(paymentProvider);
        result.setImageSource(braintreePaymentInfoService
                .getImageSourceByPaymentProviderAndCardType(paymentProvider, paymentMethod.getDetails().getBrandCode()));

        getLoggingHandler().handleResult(payload);
        return result;
    }

    public void setBrainTreeCustomerAccountService(
            BrainTreeCustomerAccountService brainTreeCustomerAccountService) {
        this.brainTreeCustomerAccountService = brainTreeCustomerAccountService;
    }

    public BrainTreePaymentService getBrainTreePaymentService() {
        return brainTreePaymentService;
    }

    public void setBrainTreePaymentService(BrainTreePaymentService brainTreePaymentService) {
        this.brainTreePaymentService = brainTreePaymentService;
    }

    @Override
    public BrainTreeConfigService getBrainTreeConfigService() {
        return brainTreeConfigService;
    }

    @Override
    public void setBrainTreeConfigService(BrainTreeConfigService brainTreeConfigService) {
        this.brainTreeConfigService = brainTreeConfigService;
    }

    public BraintreePaymentInfoService getBraintreePaymentInfoService() {
        return braintreePaymentInfoService;
    }

    public void setBraintreePaymentInfoService(
            BraintreePaymentInfoService braintreePaymentInfoService) {
        this.braintreePaymentInfoService = braintreePaymentInfoService;
    }
}
