package com.braintree.configuration.service;

import static com.braintree.constants.BraintreeConstants.PAYPAL_CONNECT_SCOPES;
import static com.braintree.constants.BraintreeConstants.PAYPAL_CONNECT_SCRIPT;
import static com.braintree.constants.BraintreeConstants.PAYPAL_CONNECT_RESPONSE_TYPE;
import static com.braintree.constants.BraintreeConstants.PAYPAL_CONNECT_RETURN_URL;
import static com.braintree.constants.BraintreeConstants.PAYPAL_CONNECT_CHECKOUT_RETURN_URL;
import static com.braintree.constants.BraintreeConstants.CLIENT_ID;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.B2C_ENDPOINT_URL_PROPERTY;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BILLING_AGREEMENT_DESCRIPTION_KEY;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_3D_SECURE;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_3D_SECURE_FALLBACK;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_3D_SECURE_VERIFICATION_MY_ACCOUNT;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_ADVANCED_FRAUD_TOOLS_ENABLED;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_APPLE_PAY_ENABLE;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_APPLE_PAY_ON_EXPRESS_CHECKOUT_ENABLE;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_CHANGE_PAYMENT_METHOD_ENABLED;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_CHANNEL_NAME;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_CREDIT_CARD_EXPIRE_BETWEEN;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_CREDIT_CARD_EXPIRE_ON_ADDING_VALIDATION_ENABLED;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_CREDIT_CARD_STATEMENT_NAME;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_CREDIT_ENABLE;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_DROPIN_ENABLE;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_ENVIRONMENT;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_GOOGLE_PAY_COUNTRY_CODE;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_GOOGLE_PAY_ENABLE;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_GOOGLE_PAY_MERCHANT_ID;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_GOOGLE_PAY_ON_EXPRESS_CHECKOUT_ENABLE;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_GRAPHQL_ENABLE;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_IMAGES_PREFIX;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_KEY;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_LOCALE;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_LOCAL_PAYMENTS_ENABLE;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_MERCHANT_ACCOUNT_PREFIX;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_PAYLATER_FOR_NON_US_COUNTRIES;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_PAYPAL_INTENT;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_SEARCH_LIMIT;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_SRC_ENABLE;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_SRC_IMG_URL;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_SRC_SDK_URL;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_SUBMIT_FOR_SETTLEMENT;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_USER_ACTION;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_VENMO_ENABLE;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_US_BANK_ACCOUNT_ENABLE;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_US_BANK_ACCOUNT_MERCHANT_NAME;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_VENMO_PROFILE_ID;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.CURRENCY_MERCHANT_ACCOUNT_ID;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.HOSTED_FIELDS_ENABLED;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.HYBRIS_BUILD_API_VERSION;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.IS_SKIP_3D_SECURE_LIABILITY_RESULT;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.MULTICAPTURE_ENABLED;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.PAYPAL_CONNECT_ADD_PAYMENT_METHOD_FLOW;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.PAYPAL_DISABLE_FUNDING;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.PAYPAL_EXPRESS_ENABLED;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.PAYPAL_LIPP_LINK_USERS_ACCOUNT;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.PAYPAL_STANDARD_ENABLED;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.STORE_IN_VAULT;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.VERIFY_CARD;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_VERIFY_FLOW;
import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_RESTRICTION_PM_REMOVAL_ENABLED;
import static com.braintree.constants.BraintreeConstants.SECRET_KEY;
import static com.braintree.constants.BraintreeConstants.PAYPAL_API_URL;
import static com.braintree.constants.BraintreeConstants.ENVIRONMENT_SANDBOX;
import static com.braintree.constants.BraintreeConstants.ENVIRONMENT_PRODUCTION;
import static com.braintree.constants.BraintreeConstants.COMMA_SEPARATOR;
import static com.braintree.constants.BraintreeConstants.BRAINTREE_ACCEPTED_PAYMENT_METHODS_DELIMETER;
import static com.braintree.constants.BraintreeConstants.BRAINTREE_DISABLE_FUNDING_ON_MY_ACCOUNT;
import static com.braintree.constants.BraintreeConstants.CARD_FUNDING_SOURCE;
import static com.braintree.constants.BraintreeConstants.CONFIGURATION_PROPERTY_DELIMETER;

import com.braintree.enums.BraintreePageType;
import com.braintree.util.BrainTreeUtils;
import com.braintreegateway.Environment;
import com.braintreegateway.util.BraintreeCrypto;
import de.hybris.platform.servicelayer.config.ConfigurationService;
import de.hybris.platform.servicelayer.i18n.CommonI18NService;
import de.hybris.platform.servicelayer.user.UserService;
import de.hybris.platform.site.BaseSiteService;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import org.apache.commons.configuration.Configuration;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

public class BrainTreeConfigService {

    private static final Logger LOG = Logger.getLogger(BrainTreeConfigService.class);
    private final Map<String, Environment> ENV_TYPE_MAP = new HashMap<>();
    private BaseSiteService baseSiteService;
    private CommonI18NService commonI18NService;
    private ConfigurationService configurationService;
    private UserService userService;

    public BrainTreeConfigService() {
        ENV_TYPE_MAP.put(ENVIRONMENT_SANDBOX, Environment.SANDBOX);
        ENV_TYPE_MAP.put(ENVIRONMENT_PRODUCTION, Environment.PRODUCTION);
    }

    private String addDisabledFunding(String disableFunding, String add) {
        if (StringUtils.isEmpty(disableFunding)) {
            return add;
        }
        return disableFunding + COMMA_SEPARATOR + add;
    }

    public boolean is3dSecureFallbackEnabled() {
        return getConfiguration().getBoolean(BRAINTREE_3D_SECURE_FALLBACK, true);
    }

    public boolean is3dSecureVerificationOnMyAccountEnabled() {
        return getConfiguration().getBoolean(BRAINTREE_3D_SECURE_VERIFICATION_MY_ACCOUNT, false);
    }

    public boolean isAdvancedFraudTools() {
        return getConfiguration().getBoolean(BRAINTREE_ADVANCED_FRAUD_TOOLS_ENABLED, true);
    }

    public boolean isApplePayEnabled() {
        return getConfiguration().getBoolean(BRAINTREE_APPLE_PAY_ENABLE, false);
    }

    public boolean isApplePayOnExpressCheckoutEnabled() {
        return getConfiguration().getBoolean(BRAINTREE_APPLE_PAY_ON_EXPRESS_CHECKOUT_ENABLE, false);
    }

    public boolean isBrainTreeGraphQLEnabled() {
        return getConfiguration().getBoolean(BRAINTREE_GRAPHQL_ENABLE, true);
    }

    public boolean isChangePaymentMethodButtonEnabled() {
        return getConfiguration().getBoolean(BRAINTREE_CHANGE_PAYMENT_METHOD_ENABLED, true);
    }

    public boolean isCreditEnabled() {
        return getConfiguration().getBoolean(BRAINTREE_CREDIT_ENABLE, true);
    }

    public boolean isDropInEnabled() {
        return getConfiguration().getBoolean(BRAINTREE_DROPIN_ENABLE, false);
    }

    public boolean isGooglePayEnabled() {
        return getConfiguration().getBoolean(BRAINTREE_GOOGLE_PAY_ENABLE, true);
    }

    public boolean isGooglePayOnExpressCheckoutEnabled() {
        return getConfiguration().getBoolean(BRAINTREE_GOOGLE_PAY_ON_EXPRESS_CHECKOUT_ENABLE, true);
    }

    public boolean isHostedFieldEnabled() {
        return getConfiguration().getBoolean(HOSTED_FIELDS_ENABLED, true);
    }

    public boolean isLocalPaymentsEnabled() {
        return getConfiguration().getBoolean(BRAINTREE_LOCAL_PAYMENTS_ENABLE, true);
    }

    public boolean isMultiCaptureEnabled() {
        return getConfiguration().getBoolean(MULTICAPTURE_ENABLED, true);
    }

    public boolean isOneOfPaymentMethodsEnabled() {
        return isPayPalStandardEnabled() || isHostedFieldEnabled() || isVenmoEnabled() || isLocalPaymentsEnabled()
            || isApplePayEnabled() || isGooglePayEnabled() || isSrcEnabled();
    }

    public boolean isPayLaterEnabledForNonUsCountries() {
        return getConfiguration().getBoolean(BRAINTREE_PAYLATER_FOR_NON_US_COUNTRIES, true);
    }

    public boolean isPayPalExpressEnabled() {
        return isPayPalStandardEnabled() && getConfiguration().getBoolean(PAYPAL_EXPRESS_ENABLED, true);
    }

    public boolean isPayPalStandardEnabled() {
        return getConfiguration().getBoolean(PAYPAL_STANDARD_ENABLED, true);
    }

    public boolean isSkip3dSecureLiabilityResult() {
        return getConfiguration().getBoolean(IS_SKIP_3D_SECURE_LIABILITY_RESULT, false);
    }

    public boolean isSkip3DSecureLiabilityResult() {
        return isSkip3dSecureLiabilityResult() && !isBrainTreeGraphQLEnabled();
    }

    public boolean is3dSecureConfiguration() {
        return getConfiguration().getBoolean(BRAINTREE_3D_SECURE, false);
    }

    public boolean isSrcEnabled() {
        return getConfiguration().getBoolean(BRAINTREE_SRC_ENABLE, true);
    }

    public boolean isStoreInVault() {
        return getConfiguration().getBoolean(STORE_IN_VAULT, true);
    }

    public boolean isStoreInVaultForCurrentUser() {
        if (userService.isAnonymousUser(userService.getCurrentUser()) || BrainTreeUtils.isIntentOrder(getIntent())) {
            return Boolean.FALSE;
        }
        return isStoreInVault();
    }

    public boolean isStoreInVaultForCardVaulting(String braintreeCustomerId) {
        return !(StringUtils.isEmpty(braintreeCustomerId)) && isStoreInVault();
    }

    public boolean isSubmitForSettlement() {
        return getConfiguration().getBoolean(BRAINTREE_SUBMIT_FOR_SETTLEMENT, true);
    }

    public boolean isVaultingForCurrentUser(String paymentProvider) {
        boolean isPayPalCheckout = BrainTreeUtils.isPayPalPayment(paymentProvider);

        if (BrainTreeUtils.isIntentOrder(getIntent()) && isPayPalCheckout) {
            return false;
        }
        return isStoreInVault();
    }

    public int getBrainTreeCreditCartExpireBetween(){
        return getConfiguration().getInt(BRAINTREE_CREDIT_CARD_EXPIRE_BETWEEN, 1);
    }

    public boolean isBrainTreeCreditExpireOnAddingValidation() {
        return getConfiguration().getBoolean(BRAINTREE_CREDIT_CARD_EXPIRE_ON_ADDING_VALIDATION_ENABLED, true);
    }

    public boolean isVenmoEnabled() {
        return getConfiguration().getBoolean(BRAINTREE_VENMO_ENABLE, true);
    }

    public boolean isUsBankAccountEnabled() {
        return getConfiguration().getBoolean(BRAINTREE_US_BANK_ACCOUNT_ENABLE, true);
    }

    public String getUsBankAccountMerchantName() {
        return getConfiguration().getString(BRAINTREE_US_BANK_ACCOUNT_MERCHANT_NAME);
    }

    public boolean isVerifyCard() {
        return getConfiguration().getBoolean(VERIFY_CARD, false);
    }

    public Map<String, String> getAcceptedPaymentMethodImages(final String paymentMethodsString) {
        final Map<String, String> acceptedPaymentMethodImages = new HashMap<>();
        final String acceptedPaymentMethods = getConfiguration().getString(paymentMethodsString);

        if (StringUtils.isNotBlank(acceptedPaymentMethods)) {
            final String paymentMethods = StringUtils.deleteWhitespace(acceptedPaymentMethods);
            final String[] paymentMethodList = StringUtils.split(paymentMethods, BRAINTREE_ACCEPTED_PAYMENT_METHODS_DELIMETER);

            for (final String paymentMethod : paymentMethodList) {
                final String imageLink = getConfiguration().getString(BRAINTREE_IMAGES_PREFIX + paymentMethod);
                if (StringUtils.isNotBlank(imageLink)) {
                    acceptedPaymentMethodImages.put(paymentMethod, imageLink);
                }
            }
        }
        return acceptedPaymentMethodImages;
    }

    public int getBackofficeSearchLimit() {
        return getConfiguration().getInt(BRAINTREE_SEARCH_LIMIT, 300);
    }

    public String getBillingAgreementDescription() {
        return getConfiguration().getString(BILLING_AGREEMENT_DESCRIPTION_KEY, StringUtils.EMPTY);
    }

    public String getBrainTreeChannel() {
        return BraintreeCrypto
            .decrypt(getConfiguration().getString(BRAINTREE_KEY), getConfiguration().getString(BRAINTREE_CHANNEL_NAME));
    }

    public String getBrainTreeLocale() {
        return getConfiguration().getString(BRAINTREE_LOCALE);
    }

    private Configuration getConfiguration() {
        return getConfigurationService().getConfiguration();
    }

    public String getCreditCardStatementName() {
        return getConfiguration().getString(BRAINTREE_CREDIT_CARD_STATEMENT_NAME);
    }

    public String getCurrencyMerchantAccountId() {
        return getConfiguration().getString(CURRENCY_MERCHANT_ACCOUNT_ID);
    }

    public String getDisableFunding(final BraintreePageType pageType) {
        String disableFunding = getConfiguration().getString(PAYPAL_DISABLE_FUNDING);
        if (BraintreePageType.ACCOUNT.equals(pageType)) {
            disableFunding = addDisabledFunding(disableFunding, BRAINTREE_DISABLE_FUNDING_ON_MY_ACCOUNT);
        }
        if (isChangePaymentMethodButtonEnabled()) {
            disableFunding = addDisabledFunding(disableFunding, CARD_FUNDING_SOURCE);
        }
        return disableFunding;
    }

    public String getEndpointURL() {
        return getConfiguration().getString(B2C_ENDPOINT_URL_PROPERTY);
    }

    public Environment getEnvironmentType() {
        if (ENV_TYPE_MAP.containsKey(getConfiguration().getString(BRAINTREE_ENVIRONMENT))) {
            return ENV_TYPE_MAP.get(getConfiguration().getString(BRAINTREE_ENVIRONMENT));
        }
        LOG.warn("Configuration environment property name is incorrect! Use environment: " + Environment.DEVELOPMENT);
        return Environment.DEVELOPMENT;
    }

    public String getEnvironmentTypeName() {
        return getEnvironmentType().getEnvironmentName();
    }

    public String getGooglePayMerchantId() {
        return getConfiguration().getString(BRAINTREE_GOOGLE_PAY_MERCHANT_ID);
    }

    public String getGooglePayCountryCode() {
        return getConfiguration().getString(BRAINTREE_GOOGLE_PAY_COUNTRY_CODE);
    }

    public Integer getHybrisBuildApiVersion() {
        String apiVersion = getConfiguration().getString(HYBRIS_BUILD_API_VERSION);
        if (StringUtils.isNotEmpty(apiVersion)) {
            String[] version = apiVersion.split("\\.");
            if (version.length > 0 && StringUtils.isNumeric(version[0])) {
                return Integer.valueOf(version[0]);
            }
        }
        return null;
    }

    public String getIntent() {
        return getConfiguration().getString(BRAINTREE_PAYPAL_INTENT).toLowerCase();
    }

    public String getMerchantAccountIdForCurrentSiteAndCurrency() {
        return getMerchantAccountIdByCurrentSiteNameAndCurrency(getBaseSiteService().getCurrentBaseSite().getUid(),
            getCommonI18NService().getCurrentCurrency().getIsocode().toLowerCase());
    }

    public String getMerchantAccountIdForCurrentSiteAndCurrencyIsoCode(final String currencyIsoCode) {
        return getMerchantAccountIdByCurrentSiteNameAndCurrency(getBaseSiteService().getCurrentBaseSite().getUid(),
            currencyIsoCode.toLowerCase(Locale.US));
    }

    public String getMerchantAccountIdForSiteAndCurrencyIsoCode(final String baseSiteUid, final String currencyIsoCode) {
        return getMerchantAccountIdByCurrentSiteNameAndCurrency(baseSiteUid, currencyIsoCode.toLowerCase(Locale.US));
    }

    public String getMerchantAccountIdByCurrentSiteNameAndCurrency(final String currentSiteName, final String currency) {
        return getConfiguration()
            .getString(BRAINTREE_MERCHANT_ACCOUNT_PREFIX + currentSiteName + CONFIGURATION_PROPERTY_DELIMETER + currency);
    }

    public String getUserAction() {
        return getConfiguration().getString(BRAINTREE_USER_ACTION);
    }

    public String getSrcSdkUrl() {
        return getConfiguration().getString(BRAINTREE_SRC_SDK_URL);
    }

    public String getSrcImgUrl() {
        return getConfiguration().getString(BRAINTREE_SRC_IMG_URL);
    }

    public String getVenmoProfileId() {
        return getConfiguration().getString(BRAINTREE_VENMO_PROFILE_ID, StringUtils.EMPTY);
    }

    public boolean isPayPalConnectAddPaymentMethodFlow() {
        return getConfiguration().getBoolean(PAYPAL_CONNECT_ADD_PAYMENT_METHOD_FLOW);
    }

    public boolean isEnableLoginViaPayPalToTheExistingUser() {
        return getConfiguration().getBoolean(PAYPAL_LIPP_LINK_USERS_ACCOUNT);
    }

    public boolean isBraintreeVerifyFlow() {
        return getConfiguration().getBoolean(BRAINTREE_VERIFY_FLOW);
    }

    public boolean isRestrictionPaymentMethodRemovalEnable() {
        return getConfiguration().getBoolean(BRAINTREE_RESTRICTION_PM_REMOVAL_ENABLED);
    }

    public BaseSiteService getBaseSiteService() {
        return baseSiteService;
    }

    public void setBaseSiteService(final BaseSiteService baseSiteService) {
        this.baseSiteService = baseSiteService;
    }

    public CommonI18NService getCommonI18NService() {
        return commonI18NService;
    }

    public void setCommonI18NService(final CommonI18NService commonI18NService) {
        this.commonI18NService = commonI18NService;
    }

    public ConfigurationService getConfigurationService() {
        return configurationService;
    }

    public void setConfigurationService(final ConfigurationService configurationService) {
        this.configurationService = configurationService;
    }

    public UserService getUserService() {
        return userService;
    }

    public void setUserService(UserService userService) {
        this.userService = userService;
    }

    public String getPayPalConnectScript() {
        return configurationService.getConfiguration().getString(PAYPAL_CONNECT_SCRIPT);
    }

    public String getPayPalConnectScopes() {
        return configurationService.getConfiguration().getString(PAYPAL_CONNECT_SCOPES);
    }

    public String getPayPalConnectResponseType() {
        return configurationService.getConfiguration().getString(PAYPAL_CONNECT_RESPONSE_TYPE);
    }

    public String getPayPalConnectReturnUrl() {
        return configurationService.getConfiguration().getString(PAYPAL_CONNECT_RETURN_URL);
    }

    public Object getPayPalConnectCheckoutReturnUrl() {
        return configurationService.getConfiguration().getString(PAYPAL_CONNECT_CHECKOUT_RETURN_URL);
    }

    public String getClientID() {
        return configurationService.getConfiguration().getString(CLIENT_ID);
    }

    public String getSecretKey() {
        return getConfigurationService().getConfiguration().getString(SECRET_KEY);
    }

    public String getPayPalApiUrl() {
        return configurationService.getConfiguration().getString(PAYPAL_API_URL);
    }
}
