package com.braintree.commands.impl;

import com.braintree.command.request.BrainTreeUpdateCreditCardRequest;
import com.braintree.command.result.BrainTreeBillingAddressResult;
import com.braintree.command.result.BrainTreeUpdateCreditCardBillingAddressResult;
import com.braintree.commands.BrainTreeUpdateCreditCardBillingAddressCommand;
import com.braintree.commands.BrainTreeVerifyCreditCardCommand;
import com.braintree.util.BrainTreeUtils;
import com.braintreegateway.Address;
import com.braintreegateway.CreditCard;
import com.braintreegateway.PaymentMethod;
import com.braintreegateway.PaymentMethodRequest;
import com.braintreegateway.Result;
import com.braintreegateway.ValidationError;
import com.braintreegateway.exceptions.NotFoundException;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.dto.BillingInfo;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import org.apache.commons.lang.StringUtils;


import java.util.List;

import static de.hybris.platform.servicelayer.util.ServicesUtil.validateParameterNotNullStandardMessage;


public class DefaultBrainTreeUpdateCreditCardCommand extends AbstractCommand implements
    BrainTreeUpdateCreditCardBillingAddressCommand, BrainTreeVerifyCreditCardCommand {

    @Override
    public BrainTreeUpdateCreditCardBillingAddressResult perform(final BrainTreeUpdateCreditCardRequest request) {
        validateParameterNotNullStandardMessage("Update Billing Address Request", request);
        try {
            final PaymentMethodRequest paymentMethodRequest = translateRequest(request);
            final Result<? extends PaymentMethod> result = getBraintreeGateway().paymentMethod()
                .update(request.getToken(),
                    paymentMethodRequest);
            if (result.isSuccess()) {
                return translateResponse(result.getTarget(), result.isSuccess());
            } else {
                return translateErrorResponse(request.getToken(), result);
            }
        } catch (final NotFoundException notFoundException) {
            return translateNotFoundResponse(request, notFoundException);
        } catch (final Exception exception) {
            throw new AdapterException(exception.getMessage(), exception);
        }
    }

    private PaymentMethodRequest translateRequest(final BrainTreeUpdateCreditCardRequest request) {
        final PaymentMethodRequest paymentMethodRequest = new PaymentMethodRequest();

        if (request.getBillingInfo() != null) {
            final BillingInfo billingInfo = request.getBillingInfo();
            paymentMethodRequest.billingAddress()
                .countryCodeAlpha2(billingInfo.getCountry())
                .region(billingInfo.getState())
                .firstName(billingInfo.getFirstName())
                .lastName(billingInfo.getLastName())
                .streetAddress(billingInfo.getStreet1())
                .extendedAddress(billingInfo.getStreet2())
                .locality(billingInfo.getCity())
                .postalCode(billingInfo.getPostalCode());
        }

        if (StringUtils.isNotBlank(request.getCvvNonce())) {
            paymentMethodRequest.paymentMethodNonce(request.getCvvNonce());
        }

        if (request.getOptions() != null) {
            paymentMethodRequest.options().verifyCard(request.getOptions().getVerifyCard());
        }
        return paymentMethodRequest;
    }

    private BrainTreeUpdateCreditCardBillingAddressResult translateResponse(final PaymentMethod target, final boolean success) {
        final BrainTreeUpdateCreditCardBillingAddressResult updateCustomerResult = new BrainTreeUpdateCreditCardBillingAddressResult();

        if (target != null) {
            CreditCard creditCard = (CreditCard)target;
            Address billingAddress = creditCard.getBillingAddress();
            BrainTreeBillingAddressResult brainTreeBillingAddressResult = BrainTreeUtils.convertRestBrainTreeAddress(billingAddress);
            updateCustomerResult.setBillingAddressResult(brainTreeBillingAddressResult);
            updateCustomerResult.setPaymentMethodToken(target.getToken());
            updateCustomerResult.setSuccess(success);
            if (success) {
                updateCustomerResult.setTransactionStatus(TransactionStatus.ACCEPTED);
                updateCustomerResult.setTransactionStatusDetails(TransactionStatusDetails.SUCCESFULL);
                getLoggingHandler().handleResult(target.getToken());
            }
        }
        return updateCustomerResult;
    }

    private BrainTreeUpdateCreditCardBillingAddressResult translateErrorResponse(final String token,
        final Result<? extends PaymentMethod> result) {
        final BrainTreeUpdateCreditCardBillingAddressResult response = new BrainTreeUpdateCreditCardBillingAddressResult();
        response.setSuccess(result.isSuccess());
        if (result.getErrors() != null) {
            final List<ValidationError> allDeepValidationErrors = result.getErrors().getAllDeepValidationErrors();
            if (allDeepValidationErrors != null && allDeepValidationErrors.size() > 0) {
                final ValidationError validationError = allDeepValidationErrors.get(0);
                getLoggingHandler().getLogger().info(
                    String.format("BT Billing Address(payment method token id=%s) updated with error: %s %s", token,
                        validationError.getCode(),
                        validationError.getMessage()));

                if (validationError.getCode() != null) {
                    response.setErrorCode(validationError.getCode().toString());
                }
                response.setErrorMessage(validationError.getMessage());
            }

            getLoggingHandler().handleErrors(result.getErrors().getAllDeepValidationErrors());
            getLoggingHandler().handleErrors(result.getErrors().getAllValidationErrors());
        }
        return response;
    }


    private BrainTreeUpdateCreditCardBillingAddressResult translateNotFoundResponse(
        final BrainTreeUpdateCreditCardRequest request,
        final NotFoundException notFoundException) {
        getLoggingHandler().getLogger()
            .info(String.format("Payment Method with token=%s not Found! Error %s", request.getToken(),
                notFoundException.getMessage()));
        final BrainTreeUpdateCreditCardBillingAddressResult brainTreeCustomerResult = new BrainTreeUpdateCreditCardBillingAddressResult();
        brainTreeCustomerResult.setSuccess(false);
        brainTreeCustomerResult.setErrorMessage(String.format("Payment Method with token=%s not Found!", request.getToken()));
        return brainTreeCustomerResult;
    }


}
