package com.braintree.commands.impl;

import static de.hybris.platform.servicelayer.util.ServicesUtil.validateParameterNotNullStandardMessage;

import com.braintree.command.request.BrainTreeSubmitForSettlementTransactionRequest;
import com.braintree.command.result.BrainTreeSubmitForSettlementTransactionResult;
import com.braintree.commands.BrainTreeSubmitForSettlementCommand;
import com.braintreegateway.Result;
import com.braintreegateway.Transaction;
import com.braintreegateway.TransactionRequest;
import com.braintreegateway.ValidationError;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import java.util.List;


public class DefaultBrainTreeSubmitForSettlementTransactionCommand extends AbstractCommand implements
    BrainTreeSubmitForSettlementCommand {

    @Override
    public BrainTreeSubmitForSettlementTransactionResult perform(
        final BrainTreeSubmitForSettlementTransactionRequest request) {
        validateParameterNotNullStandardMessage("Submit For Settlement transaction Request", request);
        try {
            final TransactionRequest transactionRequest = translateRequest(request);
            final Result<Transaction> result = getBraintreeGateway().transaction()
                .submitForSettlement(request.getTransactionId(), transactionRequest);

            BrainTreeSubmitForSettlementTransactionResult braintreeResult;
            if (result.isSuccess()) {
                braintreeResult = translateResponse(result.getTarget(), result.isSuccess());
            } else {
                braintreeResult = translateErrorResponse(request.getTransactionId(), result);
            }
            braintreeResult.setRequestBody(parseToJson(request));
            braintreeResult.setResponseBody(parseToJson(result));
            return braintreeResult;
        } catch (final Exception exception) {
            throw new AdapterException(exception.getMessage(), exception);
        }
    }

    protected TransactionRequest translateRequest(final BrainTreeSubmitForSettlementTransactionRequest request) {
        final TransactionRequest transactionRequest = new TransactionRequest();
        if (request.getOrderId() != null) {
            transactionRequest.orderId(request.getOrderId());
        }
        if (request.getAmount() != null) {
            transactionRequest.amount(request.getAmount());
        }
        return transactionRequest;
    }

    private BrainTreeSubmitForSettlementTransactionResult translateErrorResponse(final String transactionId,
        final Result<Transaction> result) {
        final BrainTreeSubmitForSettlementTransactionResult response = new BrainTreeSubmitForSettlementTransactionResult(
            result.isSuccess());

        response.setTransactionStatus(TransactionStatus.REJECTED);
        if (result.getMessage() != null) {
            response.setErrorMessage(result.getMessage());
        }

        if (result.getErrors() != null) {

            final List<ValidationError> allDeepValidationErrors = result.getErrors().getAllDeepValidationErrors();
            if (allDeepValidationErrors != null && allDeepValidationErrors.size() > 0) {
                final ValidationError validationError = allDeepValidationErrors.get(0);
                getLoggingHandler().getLogger().info(
                    String.format("BT transaction id(%s) submit for settlement with error: %s %s", transactionId,
                        validationError.getCode(), validationError.getMessage()));

                if (validationError.getCode() != null) {
                    response.setErrorCode(validationError.getCode().toString());
                }
                response.setErrorMessage(validationError.getMessage());
            }
            getLoggingHandler().handleResult("[SUBMIT FOR SETTLEMENT TRANSACTION] ", result.getTarget());
        }
        return response;
    }

    private BrainTreeSubmitForSettlementTransactionResult translateResponse(final Transaction target,
        final boolean success) {
        final BrainTreeSubmitForSettlementTransactionResult result = new BrainTreeSubmitForSettlementTransactionResult(
            success);
        if (target != null) {
            result.setStatus(target.getStatus().name());
            result.setTransactionId(target.getId());
            result.setTransactionGraphQLId(target.getGraphQLId());
            result.setCurrencyIsoCode(target.getCurrencyIsoCode());
            result.setAmount(target.getAmount());
            result.setTransaction(target);
            result.setCreatedAt(target.getCreatedAt().getTime());
            result.setPaymentInstrumentType(target.getPaymentInstrumentType());
            if (success) {
                result.setTransactionStatus(TransactionStatus.ACCEPTED);
                result.setTransactionStatusDetails(TransactionStatusDetails.SUCCESFULL);
            }
            getLoggingHandler().handleResult("[SUBMIT FOR SETTLEMENT TRANSACTION] ", target);
        }
        return result;
    }
}
