package com.braintree.commands.impl;

import static de.hybris.platform.servicelayer.util.ServicesUtil.validateParameterNotNullStandardMessage;

import com.braintree.command.request.BrainTreeRefundTransactionRequest;
import com.braintree.command.result.BrainTreeRefundTransactionResult;
import com.braintree.commands.BrainTreeRefundCommand;
import com.braintreegateway.Result;
import com.braintreegateway.Transaction;
import com.braintreegateway.TransactionRefundRequest;
import com.braintreegateway.ValidationError;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import java.util.List;
import org.apache.log4j.Logger;


public class DefaultBrainTreeRefundTransactionCommand extends AbstractCommand implements BrainTreeRefundCommand {

    private static final Logger LOG = Logger.getLogger(DefaultBrainTreeRefundTransactionCommand.class);

    private static final String COLON = ":";

    @Override
    public BrainTreeRefundTransactionResult perform(final BrainTreeRefundTransactionRequest request) {
        validateParameterNotNullStandardMessage("Refund transaction Request", request);
        try {
            final TransactionRefundRequest transactionRequest = new TransactionRefundRequest();
            transactionRequest.amount(request.getAmount());
            transactionRequest.orderId(request.getOrderId());

            LOG.info("transactionRequest.query: " + transactionRequest.toQueryString());
            LOG.info("transactionRequest.xml: " + transactionRequest.toXML());

            final Result<Transaction> result = getBraintreeGateway().transaction().refund(request.getTransactionId(),
                transactionRequest);
            LOG.info("Message: " + result.getMessage());
            LOG.info("Parameters: " + result.getParameters());
            BrainTreeRefundTransactionResult braintreeResult;
            if (result.isSuccess()) {
                braintreeResult = translateResponse(result.getTarget(), result.isSuccess());
            } else {
                braintreeResult = translateErrorResponse(request.getTransactionId(), result);
            }
            braintreeResult.setRequestBody(parseToJson(request));
            braintreeResult.setResponseBody(parseToJson(result));
            return braintreeResult;
        } catch (final Exception exception) {
            throw new AdapterException(exception.getMessage(), exception);
        }
    }

    private BrainTreeRefundTransactionResult translateErrorResponse(final String transactionId,
        final Result<Transaction> result) {
        final BrainTreeRefundTransactionResult response = new BrainTreeRefundTransactionResult(result.isSuccess());
        if (result.getErrors() != null) {
            final List<ValidationError> allDeepValidationErrors = result.getErrors().getAllDeepValidationErrors();
            if (allDeepValidationErrors != null && allDeepValidationErrors.size() > 0) {
                final ValidationError validationError = allDeepValidationErrors.get(0);
                getLoggingHandler().getLogger().info(
                    String.format("BT transaction id(%s) refund with error: %s %s", transactionId,
                        validationError.getCode(),
                        validationError.getMessage()));

                if (validationError.getCode() != null) {
                    response.setErrorCode(validationError.getCode().toString());
                }
                response.setErrorMessage(validationError.getMessage());
            } else {
                final String[] errorMessage = result.getTransaction().getAdditionalProcessorResponse().split(COLON);
                response.setErrorMessage(errorMessage[errorMessage.length - 1]);
            }
        } else {
            response.setErrorMessage(result.getMessage());
        }
        response.setTransactionStatus(TransactionStatus.REJECTED);
        return response;
    }

    private BrainTreeRefundTransactionResult translateResponse(final Transaction target, final boolean success) {
        final BrainTreeRefundTransactionResult result = new BrainTreeRefundTransactionResult(success);
        if (target != null) {
            result.setStatus(target.getStatus().name());
            result.setAmount(target.getAmount());
            result.setCurrencyIsoCode(target.getCurrencyIsoCode());
            result.setTransactionId(target.getId());
            result.setTransactionGraphQLId(target.getGraphQLId());
            result.setTransaction(target);
            result.setOrderId(target.getOrderId());
            result.setCreatedAt(target.getCreatedAt().getTime());
            result.setPaymentInstrumentType(target.getPaymentInstrumentType());
            if (success) {
                result.setTransactionStatus(TransactionStatus.ACCEPTED);
                result.setTransactionStatusDetails(TransactionStatusDetails.SUCCESFULL);
            }
        }
        return result;
    }
}
