package com.braintree.commands.impl;

import static de.hybris.platform.servicelayer.util.ServicesUtil.validateParameterNotNullStandardMessage;

import com.braintree.command.request.BrainTreeFindTransactionRequest;
import com.braintree.command.result.BrainTreeFindTransactionResult;
import com.braintree.commands.BrainTreeFindRefundCommand;
import com.braintree.commands.BrainTreeFindTransactionCommand;
import com.braintree.constants.BrainTreeSearchTextOptions;
import com.braintree.hybris.data.BraintreeTransactionData;
import com.braintreegateway.ResourceCollection;
import com.braintreegateway.Transaction;
import com.braintreegateway.TransactionSearchRequest;
import com.braintreegateway.exceptions.NotFoundException;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;


public class DefaultBrainTreeFindTransactionCommand extends AbstractCommand
        implements BrainTreeFindTransactionCommand, BrainTreeFindRefundCommand {

    private static final Logger LOG = Logger.getLogger(DefaultBrainTreeFindTransactionCommand.class);

    private Converter<BrainTreeFindTransactionResult, BraintreeTransactionData> transactionSearchResultConverter;

    @Override
    public BrainTreeFindTransactionResult perform(final BrainTreeFindTransactionRequest request) {
        validateParameterNotNullStandardMessage("Find Transaction Request", request);
        try {
            final TransactionSearchRequest transactionSearchRequest = translateRequest(request);
            final ResourceCollection<Transaction> transactions = getBraintreeGateway().transaction()
                .search(transactionSearchRequest);
            final BrainTreeFindTransactionResult transactionResult = new BrainTreeFindTransactionResult(transactions);
            transactionResult.setExceedLimit(false);
            return translateResponse(transactionResult);
        } catch (final NotFoundException exception) {
            LOG.error("[BT Find Transaction] Can't find BrainTree customer with id: ", exception);
            return new BrainTreeFindTransactionResult();
        } catch (final Exception exception) {
            LOG.error("[BT Find Transaction] Error during try to find Transaction: ", exception);
            throw new IllegalArgumentException(exception.getMessage());
        }
    }

    private BrainTreeFindTransactionResult translateResponse(BrainTreeFindTransactionResult transactionResult) {
        final int searchLimit = getBrainTreeConfigService().getBackofficeSearchLimit();
        if (transactionResult.getTransactions().getMaximumSize() > searchLimit) {
            final String message = String.format("[BT Transaction search]Too many results! Limit is %s " +
                    "Please type transaction id or customer email.", searchLimit);
            LOG.warn(message);
            transactionResult.setExceedLimit(true);
            return transactionResult;
        }
        final BraintreeTransactionData transactionData = transactionSearchResultConverter.convert(transactionResult);
        transactionResult.setTransactionData(transactionData);
        return transactionResult;
    }

    private TransactionSearchRequest translateRequest(final BrainTreeFindTransactionRequest request) {
        final TransactionSearchRequest transactionSearchRequest = new TransactionSearchRequest();
        if (request != null) {
            if (request.getStartDate() != null && request.getEndDate() != null) {
                transactionSearchRequest.createdAt().between(request.getStartDate(), request.getEndDate());
            }
            if (StringUtils.isNotBlank(request.getTransactionId())) {
                transactionSearchRequest.id().is(request.getTransactionId());
            }
            if (StringUtils.isNotBlank(request.getCustomerId())) {
                transactionSearchRequest.customerId().is(request.getCustomerId());
            }
            if (StringUtils.isNotBlank(request.getCustomerEmail())) {
                customerEmailOperator(transactionSearchRequest, request);
            }
            if (StringUtils.isNotBlank(request.getTransactionStatus())) {
                transactionSearchRequest.status().is(Transaction.Status.valueOf(request.getTransactionStatus()));
            }
            if (request.getTransactionIds() != null) {
                transactionSearchRequest.ids().in(request.getTransactionIds());
            }
        }
        return transactionSearchRequest;
    }

    private void customerEmailOperator(TransactionSearchRequest transactionSearchRequest, BrainTreeFindTransactionRequest request) {
        switch (BrainTreeSearchTextOptions.valueOf(request.getCustomerEmailOperator())) {
            case CONTAINS:
                transactionSearchRequest.customerEmail().contains(request.getCustomerEmail());
                break;

            case STARTS_WITH:
                transactionSearchRequest.customerEmail().startsWith(request.getCustomerEmail());
                break;

            case ENDS_WITH:
                transactionSearchRequest.customerEmail().endsWith(request.getCustomerEmail());
                break;

            default:
                transactionSearchRequest.customerEmail().is(request.getCustomerEmail());
                break;
        }
    }

    public Converter<BrainTreeFindTransactionResult, BraintreeTransactionData> getTransactionSearchResultConverter() {
        return transactionSearchResultConverter;
    }

    public void setTransactionSearchResultConverter(
        Converter<BrainTreeFindTransactionResult, BraintreeTransactionData> transactionSearchResultConverter) {
        this.transactionSearchResultConverter = transactionSearchResultConverter;
    }
}
