package com.braintree.commands.impl;

import static com.braintree.constants.BraintreeConstants.BRAINTREE_3D_SECURE_ENVIRONMENT_PSD2;
import static com.braintree.constants.BraintreeConstants.BRAINTREE_3D_SECURE_ENVIRONMENT_UNAVAILABLE;

import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintreegateway.AuthenticationInsight;
import com.braintreegateway.PaymentMethodNonceRequest;
import java.util.List;

import com.braintree.command.result.BrainTreePaymentMethodNonceResult;
import com.braintree.util.GenericBuilder;
import java.util.Optional;
import org.apache.commons.collections4.CollectionUtils;

import com.braintree.commands.BrainTreeCreatePaymentMethodNonceCommand;
import com.braintreegateway.PaymentMethodNonce;
import com.braintreegateway.Result;
import com.braintreegateway.ValidationError;
import com.braintreegateway.ValidationErrors;


public class DefaultBrainTreeCreatePaymentMethodNonceCommand
        extends AbstractCommand<String, BrainTreePaymentMethodNonceResult>
        implements BrainTreeCreatePaymentMethodNonceCommand {

    private BrainTreeConfigService configService;

    @Override
    public BrainTreePaymentMethodNonceResult perform(final String methodToken) {
        Result<PaymentMethodNonce> brainTreeReply;
        if (configService.is3dSecureFallbackEnabled()) {
            brainTreeReply = getBraintreeGateway()
                    .paymentMethodNonce()
                    .create(new PaymentMethodNonceRequest()
                            .paymentMethodToken(methodToken)
                            .merchantAccountId(configService.getCurrencyMerchantAccountId())
                            .authenticationInsight(true));
        } else {
            brainTreeReply = getBraintreeGateway()
                    .paymentMethodNonce()
                    .create(methodToken);
        }

        validateErrors(methodToken, brainTreeReply);

        return GenericBuilder.of(BrainTreePaymentMethodNonceResult::new)
                .with(BrainTreePaymentMethodNonceResult::setNonce, brainTreeReply.getTarget().getNonce())
                .with(BrainTreePaymentMethodNonceResult::setBin, brainTreeReply.getTarget().getDetails().getBin())
                .with(BrainTreePaymentMethodNonceResult::setShouldPerform3dSecure,
                    Optional.ofNullable(brainTreeReply.getTarget())
                        .map(PaymentMethodNonce::getAuthenticationInsight)
                        .map(AuthenticationInsight::getRegulationEnvironment)
                        .map(this::isRegulatedEnvironment)
                        .orElse(Boolean.FALSE))
                .build();
    }

    private void validateErrors(String request, Result<PaymentMethodNonce> brainTreeReply) {
        final ValidationErrors errors = brainTreeReply.getErrors();
        if (errors != null) {
            final List<ValidationError> validationErrors = errors.getAllDeepValidationErrors();
            if (CollectionUtils.isNotEmpty(validationErrors)) {
                final ValidationError validationError = validationErrors.get(0);
                getLoggingHandler().getLogger().info(
                    String.format("BT payment method nonce for token(%s) is creating with error: %s %s", request,
                        validationError.getCode(),
                        validationError.getMessage()));
            }
        }
    }

    private boolean isRegulatedEnvironment(String environment) {
        return BRAINTREE_3D_SECURE_ENVIRONMENT_PSD2.equalsIgnoreCase(environment)
            || BRAINTREE_3D_SECURE_ENVIRONMENT_UNAVAILABLE.equalsIgnoreCase(environment);
    }


    public void setConfigService(BrainTreeConfigService configService) {
        this.configService = configService;
    }
}
