package com.braintree.auth;


import de.hybris.platform.core.Registry;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.core.model.user.UserModel;
import de.hybris.platform.jalo.JaloConnection;
import de.hybris.platform.jalo.JaloSession;
import de.hybris.platform.jalo.user.User;
import de.hybris.platform.jalo.user.UserManager;
import de.hybris.platform.servicelayer.exceptions.UnknownIdentifierException;
import de.hybris.platform.servicelayer.user.UserService;
import de.hybris.platform.spring.security.CoreAuthenticationProvider;
import de.hybris.platform.spring.security.CoreUserDetails;
import java.util.Collections;
import org.apache.commons.lang.StringUtils;
import org.springframework.security.authentication.AbstractAuthenticationToken;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.CredentialsExpiredException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UserDetailsChecker;
import org.springframework.security.core.userdetails.UsernameNotFoundException;

public class PayPalAuthProvider extends CoreAuthenticationProvider {

    private final UserDetailsChecker postAuthenticationChecks = new PayPalAuthProvider.DefaultPostAuthenticationChecks();
    private UserService userService;


    @Override
    public Authentication authenticate(Authentication authentication) throws AuthenticationException {
        if (Registry.hasCurrentTenant() && JaloConnection.getInstance().isSystemInitialized()) {
            try {
                UserDetails userDetails = null;
                String userName = authentication.getPrincipal() == null ? "NONE_PROVIDED" : authentication.getName();
                UserModel userForUID = userService.getUserForUID(userName);
                if (userForUID instanceof CustomerModel) {
                    CustomerModel customer = (CustomerModel) userForUID;
                    if (StringUtils.isNotBlank(customer.getPayPalAccessToken())) {
                        userDetails = this.retrieveUser(userName);
                        this.getPreAuthenticationChecks().check(userDetails);
                        User user = UserManager.getInstance().getUserByLogin(userDetails.getUsername());
                        Object credential = authentication.getCredentials();
                        if (credential instanceof String) {
                            if (!customer.getPayPalAccessToken().equals(credential)) {
                                throw new BadCredentialsException(
                                        this.messages.getMessage("CoreAuthenticationProvider.badCredentials", "Bad credentials"));
                            }
                        } else {
                            throw new BadCredentialsException(this.messages.getMessage("CoreAuthenticationProvider.badCredentials", "Bad credentials"));
                        }

                        this.additionalAuthenticationChecks(userDetails, (AbstractAuthenticationToken) authentication);
                        this.postAuthenticationChecks.check(userDetails);
                        JaloSession.getCurrentSession().setUser(user);
                        return this.createSuccessAuthentication(authentication, userDetails);
                    } else {
                        return super.authenticate(authentication);
                    }
                } else {
                    return super.authenticate(authentication);
                }
            } catch (UsernameNotFoundException | UnknownIdentifierException var6) {
                throw new BadCredentialsException(
                    this.messages.getMessage("CoreAuthenticationProvider.badCredentials", "Bad credentials"), var6);
            }
        } else {
            return this.createSuccessAuthentication(authentication, new CoreUserDetails("systemNotInitialized", "systemNotInitialized", true, false, true, true, Collections.emptyList(), (String)null));
        }

    }

    private class DefaultPostAuthenticationChecks implements UserDetailsChecker {
        private DefaultPostAuthenticationChecks() {
        }

        public void check(UserDetails user) {
            if (!user.isCredentialsNonExpired()) {
                throw new CredentialsExpiredException(PayPalAuthProvider.this.messages.getMessage("CoreAuthenticationProvider.credentialsExpired", "User credentials have expired"));
            }
        }
    }

    @Override
    protected Authentication createSuccessAuthentication(Authentication authentication, UserDetails user) {
        ConnectWithPayPalAuthenticationToken result = new ConnectWithPayPalAuthenticationToken(user.getUsername(),
            user.getAuthorities());
        result.setDetails(authentication.getDetails());
        return result;
    }

    public UserService getUserService() {
        return userService;
    }

    public void setUserService(UserService userService) {
        this.userService = userService;
    }

}
