package com.braintree.validators.impl;

import com.braintree.hybris.data.ws.BraintreeCreditCardPaymentDetailsWsDTO;
import com.braintree.validators.BrainTreeValidator;
import de.hybris.platform.commercewebservicescommons.dto.order.PaymentDetailsWsDTO;
import org.apache.commons.lang.StringUtils;
import org.springframework.util.Assert;
import org.springframework.validation.Errors;
import org.springframework.validation.ValidationUtils;
import org.springframework.validation.Validator;
import java.util.Calendar;

public class BraintreeCreditCardPaymentResponseValidator implements BrainTreeValidator {

    private static final String FIELD_REQUIRED_MESSAGE_ID = "field.required";

    private Validator creditCardDataValidator;
    private Validator paymentAddressValidator;

    @Override
    public boolean supports(Class aClass) {
        return BraintreeCreditCardPaymentDetailsWsDTO.class.isAssignableFrom(aClass);
    }

    @Override
    public void validate(Object target, Errors errors) {

        final BraintreeCreditCardPaymentDetailsWsDTO creditCardPaymentDetails = (BraintreeCreditCardPaymentDetailsWsDTO) target;

        validateCreditCardPaymentDetails(creditCardPaymentDetails, errors);

        Assert.notNull(creditCardPaymentDetails.getTokenizedCardData(), "Tokenized credit card data must not be null");
        validate(creditCardPaymentDetails.getTokenizedCardData(), "tokenizedCardData", creditCardDataValidator);
    }

    private void validateCreditCardPaymentDetails(final Object target, Errors errors) {

        final PaymentDetailsWsDTO paymentDetails = (PaymentDetailsWsDTO) target;

        if (StringUtils.isNotBlank(paymentDetails.getStartMonth()) && StringUtils.isNotBlank(paymentDetails.getStartYear())
                && StringUtils.isNotBlank(paymentDetails.getExpiryMonth()) && StringUtils.isNotBlank(paymentDetails.getExpiryYear())) {
            final Calendar start = Calendar.getInstance();
            start.set(Calendar.DAY_OF_MONTH, 0);
            start.set(Calendar.MONTH, Integer.parseInt(paymentDetails.getStartMonth()) - 1);
            start.set(Calendar.YEAR, Integer.parseInt(paymentDetails.getStartYear()) - 1);

            final Calendar expiration = Calendar.getInstance();
            expiration.set(Calendar.DAY_OF_MONTH, 0);
            expiration.set(Calendar.MONTH, Integer.parseInt(paymentDetails.getExpiryMonth()) - 1);
            expiration.set(Calendar.YEAR, Integer.parseInt(paymentDetails.getExpiryYear()) - 1);

            if (start.after(expiration)) {
                errors.rejectValue("startMonth", "payment.startDate.invalid");
            }
        }

        ValidationUtils.rejectIfEmptyOrWhitespace(errors, "accountHolderName", FIELD_REQUIRED_MESSAGE_ID);
        ValidationUtils.rejectIfEmptyOrWhitespace(errors, "cardType.code", FIELD_REQUIRED_MESSAGE_ID);
        ValidationUtils.rejectIfEmptyOrWhitespace(errors, "cardNumber", FIELD_REQUIRED_MESSAGE_ID);
        ValidationUtils.rejectIfEmptyOrWhitespace(errors, "expiryMonth", FIELD_REQUIRED_MESSAGE_ID);
        ValidationUtils.rejectIfEmptyOrWhitespace(errors, "expiryYear", FIELD_REQUIRED_MESSAGE_ID);

        paymentAddressValidator.validate(paymentDetails, errors);
    }

    public void setCreditCardDataValidator(Validator creditCardDataValidator) {
        this.creditCardDataValidator = creditCardDataValidator;
    }

    public void setPaymentAddressValidator(final Validator paymentAddressValidator) {
        this.paymentAddressValidator = paymentAddressValidator;
    }
}
