package com.braintree.controllers;

import com.braintree.facade.BrainTreeSetUpPasswordFacade;
import de.hybris.platform.commercefacades.customer.CustomerFacade;
import de.hybris.platform.commercewebservicescommons.dto.user.UserSignUpWsDTO;
import de.hybris.platform.webservicescommons.cache.CacheControl;
import de.hybris.platform.webservicescommons.cache.CacheControlDirective;
import de.hybris.platform.webservicescommons.swagger.ApiBaseSiteIdAndUserIdParam;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import org.springframework.http.HttpStatus;
import org.springframework.security.access.annotation.Secured;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Controller;
import org.springframework.validation.Validator;
import org.springframework.web.bind.annotation.*;
import com.braintree.hybris.data.ws.BraintreeB2BSetUpPasswordFormWsDTO;

import javax.annotation.Resource;


@Controller
@RequestMapping(value = "/{baseSiteId}/users")
@CacheControl(directive = CacheControlDirective.PRIVATE)
@Tag(name = "Braintree Users")
public class BraintreeB2BUsersController extends BraintreeBaseCommerceController {

    @Resource(name = "brainTreeSetUpPasswordFacade")
    private BrainTreeSetUpPasswordFacade braintreeSetUpPasswordFacade;

    @Resource(name = "wsCustomerFacade")
    private CustomerFacade customerFacade;

    @Resource(name = "passwordStrengthValidator")
    private Validator passwordStrengthValidator;

    @Secured({ "ROLE_CUSTOMERGROUP", "ROLE_TRUSTED_CLIENT", "ROLE_CUSTOMERMANAGERGROUP" })
    @RequestMapping(value = "/{userId}/getSetUpPasswordFormData", method = RequestMethod.GET)
    @ResponseStatus(value = HttpStatus.OK)
    @ResponseBody
    @Operation(summary = "Get information about ability for setting single authentication password")
    @ApiBaseSiteIdAndUserIdParam
    public BraintreeB2BSetUpPasswordFormWsDTO getSetUpPasswordFormData(
            @Parameter(description = "User identifier.", required = true) @PathVariable final String userId) {
        BraintreeB2BSetUpPasswordFormWsDTO setUpPasswordWsDTO = new BraintreeB2BSetUpPasswordFormWsDTO();
        final Authentication auth = SecurityContextHolder.getContext().getAuthentication();
        if (containsRole(auth, "ROLE_TRUSTED_CLIENT") || containsRole(auth, "ROLE_CUSTOMERMANAGERGROUP")) {
            setUpPasswordWsDTO.setIsFormForSettingSingleAuthenticationPasswordActive(
                    braintreeSetUpPasswordFacade.isSetUpPasswordForm(userId));
        } else {
            setUpPasswordWsDTO.setIsFormForSettingSingleAuthenticationPasswordActive(
                    braintreeSetUpPasswordFacade.isSetUpPasswordForm());
        }
        return setUpPasswordWsDTO;
    }

    @Secured({"ROLE_CUSTOMERGROUP", "ROLE_TRUSTED_CLIENT", "ROLE_CUSTOMERMANAGERGROUP"})
    @RequestMapping(value = "/{userId}/setPassword", method = RequestMethod.POST)
    @ResponseStatus(value = HttpStatus.ACCEPTED)
    @Operation(operationId = "setUpPassword", summary = "Set up a password for Single Auth", description = "Set up a password for Single Auth")
    @ApiBaseSiteIdAndUserIdParam
    public void setPasswordForSingleAuth(
            @Parameter(description = "User identifier.", required = true) @PathVariable final String userId,
            @Parameter(description = "New password.", required = true) @RequestParam final String newPassword) {
        final Authentication auth = SecurityContextHolder.getContext().getAuthentication();
        final UserSignUpWsDTO customer = new UserSignUpWsDTO();
        customer.setPassword(newPassword);
        validate(customer, "password", passwordStrengthValidator);
        if (containsRole(auth, "ROLE_TRUSTED_CLIENT") || containsRole(auth, "ROLE_CUSTOMERMANAGERGROUP")) {
            customerFacade.setPassword(userId, newPassword);
        } else {
            braintreeSetUpPasswordFacade.setUpPassword(newPassword);
        }
    }

    protected boolean containsRole(final Authentication auth, final String role) {
        for (final GrantedAuthority ga : auth.getAuthorities()) {
            if (ga.getAuthority().equals(role)) {
                return true;
            }
        }
        return false;
    }
}
