/**
 *
 */
package com.braintree.facade;

import static org.mockito.Matchers.any;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.constants.BraintreeConstants.PayPalIntent;
import com.braintree.facade.impl.DefaultBrainTreeCheckoutFacade;
import com.braintree.model.BrainTreePaymentInfoModel;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.b2bacceleratorfacades.checkout.data.PlaceOrderData;
import de.hybris.platform.b2bacceleratorfacades.order.data.B2BPaymentTypeData;
import de.hybris.platform.b2bacceleratorfacades.order.data.B2BReplenishmentRecurrenceEnum;
import de.hybris.platform.b2b.enums.CheckoutPaymentType;
import de.hybris.platform.commercefacades.order.CartFacade;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commercefacades.order.data.CartData;
import de.hybris.platform.commercefacades.order.data.DeliveryModeData;
import de.hybris.platform.commercefacades.order.data.OrderData;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.commerceservices.order.CommerceCheckoutService;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.order.payment.InvoicePaymentInfoModel;
import de.hybris.platform.order.CartService;
import de.hybris.platform.order.InvalidCartException;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.i18n.I18NService;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.site.BaseSiteService;
import java.util.Date;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;
import org.springframework.test.util.ReflectionTestUtils;

@UnitTest
public class BrainTreeB2BCheckoutFacadeTest {
    private static final String SHIPS_FROM_POSTAL_CODE = "ships";
    private static final String SECURITY_CODE = "securityCode";
    private static final String DESCRIPTION = "description";
    private static final Date DATE = new Date();

    @Mock
    private I18NService i18NService;

    @Mock
    private CartFacade cartFacade;

    @Mock
    private DefaultBrainTreeCheckoutFacade brainTreeCheckoutFacade;

    @Mock
    private CartService cartService;

    @Mock
    private Converter<OrderModel, OrderData> orderConverter;

    @Mock
    private BaseSiteService baseSiteService;

    @Mock
    private BrainTreeConfigService brainTreeConfigService;

    @Mock
    private ModelService modelService;

    @Mock
    private CommerceCheckoutService commerceCheckoutService;

    @InjectMocks
    private BrainTreeB2BCheckoutFacade unit;


    private CartModel cartModel;
    private CartData cartData;
    private BrainTreePaymentInfoModel brainTreePaymentInfoModel;
    private InvoicePaymentInfoModel invoicePaymentInfoModel;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
        ReflectionTestUtils
            .setField(unit, BrainTreeB2BCheckoutFacade.class, "cartService", cartService, CartService.class);
        cartModel = Mockito.mock(CartModel.class);
        brainTreePaymentInfoModel = Mockito.mock(BrainTreePaymentInfoModel.class);
        invoicePaymentInfoModel = Mockito.mock(InvoicePaymentInfoModel.class);
        cartData = Mockito.mock(CartData.class);
        when(cartService.getSessionCart()).thenReturn(cartModel);
        when(brainTreeCheckoutFacade.getCartService()).thenReturn(cartService);
        when(brainTreeCheckoutFacade.getBrainTreeConfigService()).thenReturn(brainTreeConfigService);
        when(cartFacade.getSessionCart()).thenReturn(cartData);
    }

    @Test
    public void storeIntentToCartShouldSetPaypalIntentAndSaveModelForBrainTreePaymentInfoModel() {
        final CartModel cartModel = Mockito.mock(CartModel.class);
        when(cartModel.getPaymentInfo()).thenReturn(brainTreePaymentInfoModel);
        when(cartService.getSessionCart()).thenReturn(cartModel);
        unit.storeIntentToCart();
        verify(brainTreePaymentInfoModel).setPayPalIntent(any());
        verify(modelService).save(brainTreePaymentInfoModel);
    }

    @Test
    public void storeIntentToCartShouldSaveModelForInvoicePaymentInfoModel() {
        final CartModel cartModel = Mockito.mock(CartModel.class);
        when(cartModel.getPaymentInfo()).thenReturn(invoicePaymentInfoModel);
        when(cartService.getSessionCart()).thenReturn(cartModel);
        unit.storeIntentToCart();
        verify(modelService).save(invoicePaymentInfoModel);
    }

    @Test
    public void storeShipsFromPostalCodeToCartShouldSetShipsAndSaveBraintreePaymentInfoModel() {
        final CartModel cartModel = Mockito.mock(CartModel.class);
        when(cartModel.getPaymentInfo()).thenReturn(brainTreePaymentInfoModel);
        when(cartService.getSessionCart()).thenReturn(cartModel);
        unit.storeShipsFromPostalCodeToCart(SHIPS_FROM_POSTAL_CODE);
        verify(brainTreePaymentInfoModel).setShipsFromPostalCode(SHIPS_FROM_POSTAL_CODE);
        verify(modelService).save(brainTreePaymentInfoModel);
    }

    @Test
    public void storeShipsFromPostalCodeToCartShouldSaveInvoicePaymentInfoModel() {
        final CartModel cartModel = Mockito.mock(CartModel.class);
        when(cartModel.getPaymentInfo()).thenReturn(invoicePaymentInfoModel);
        when(cartService.getSessionCart()).thenReturn(cartModel);
        unit.storeShipsFromPostalCodeToCart(SHIPS_FROM_POSTAL_CODE);
        verify(modelService).save(invoicePaymentInfoModel);
    }


    @Test
    public void authorizePaymentShouldDelegateToCheckoutFacadeIfPaymentTypeNotAccount() {
        unit.authorizePayment(SECURITY_CODE);
        verify(brainTreeCheckoutFacade).authorizePayment(SECURITY_CODE);
    }

    @Test
    public void authorizePaymentShouldNotDelegateToCheckoutFacadeIfPaymentTypeAccount() {
        when(cartModel.getPaymentType()).thenReturn(CheckoutPaymentType.ACCOUNT);
        unit.authorizePayment(SECURITY_CODE);
        verify(brainTreeCheckoutFacade, never()).authorizePayment(SECURITY_CODE);
    }

    @Test(expected = IllegalStateException.class)
    public void placeOrderShouldThrowExceptionIfOrderDataFalseTermsCheck() throws InvalidCartException {
        final PlaceOrderData orderData = Mockito.mock(PlaceOrderData.class);
        final CartModel cartModel = Mockito.mock(CartModel.class);
        when(cartModel.getPaymentInfo()).thenReturn(brainTreePaymentInfoModel);
        when(brainTreePaymentInfoModel.getPayPalIntent()).thenReturn(PayPalIntent.PAYPAL_INTENT_ORDER);
        when(orderData.getTermsCheck()).thenReturn(false);
        when(cartService.getSessionCart()).thenReturn(cartModel);

        unit.placeOrder(orderData);
    }

    @Test(expected = IllegalStateException.class)
    public void placeOrderShouldThrowExceptionWhenReplenishmentRecurrenceWeeklyButNDaysOfWeekEmpty()
        throws InvalidCartException {
        final PlaceOrderData orderData = Mockito.mock(PlaceOrderData.class);
        when(cartModel.getPaymentInfo()).thenReturn(brainTreePaymentInfoModel);
        when(brainTreePaymentInfoModel.getPayPalIntent()).thenReturn(PayPalIntent.PAYPAL_INTENT_ORDER);
        when(orderData.getTermsCheck()).thenReturn(true);
        when(cartData.isCalculated()).thenReturn(true);
        final AddressData deliveryAddress = Mockito.mock(AddressData.class);
        when(cartData.getDeliveryAddress()).thenReturn(deliveryAddress);
        final DeliveryModeData modeData = Mockito.mock(DeliveryModeData.class);
        when(cartData.getDeliveryMode()).thenReturn(modeData);
        final B2BPaymentTypeData paymentTypeData = Mockito.mock(B2BPaymentTypeData.class);
        when(cartData.getPaymentType()).thenReturn(paymentTypeData);
        final CCPaymentInfoData paymentInfoData = Mockito.mock(CCPaymentInfoData.class);
        when(cartData.getPaymentInfo()).thenReturn(paymentInfoData);
        when(orderData.getNegotiateQuote()).thenReturn(true);
        when(orderData.getQuoteRequestDescription()).thenReturn(DESCRIPTION);
        when(orderData.getReplenishmentOrder()).thenReturn(true);
        when(orderData.getReplenishmentStartDate()).thenReturn(DATE);
        when(orderData.getReplenishmentRecurrence()).thenReturn(B2BReplenishmentRecurrenceEnum.WEEKLY);

        unit.placeOrder(orderData);
    }

}
