package com.braintree.controllers.pages;

import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.controllers.handler.BraintreePayPalResponseExpressCheckoutHandler;
import com.braintree.enums.BrainTreePaymentMethod;
import com.braintree.enums.BraintreePageType;
import com.braintree.exceptions.BrainTreeUnverifiedAccountPayPalException;
import com.braintree.facade.*;
import com.braintree.facade.impl.DefaultBrainTreeCheckoutFacade;
import com.braintree.facade.impl.DefaultBrainTreePaymentFacade;
import com.braintree.hybris.data.*;
import com.braintree.method.BrainTreePaymentService;
import com.braintree.security.PayPalAutoLoginStrategy;
import com.braintree.util.BrainTreeUtils;
import com.braintree.util.GenericBuilder;
import de.hybris.platform.acceleratorfacades.flow.CheckoutFlowFacade;
import de.hybris.platform.acceleratorstorefrontcommons.breadcrumb.Breadcrumb;
import de.hybris.platform.acceleratorstorefrontcommons.controllers.pages.AbstractLoginPageController;
import de.hybris.platform.acceleratorstorefrontcommons.controllers.util.GlobalMessages;
import de.hybris.platform.acceleratorstorefrontcommons.forms.GuestForm;
import de.hybris.platform.acceleratorstorefrontcommons.forms.LoginForm;
import de.hybris.platform.acceleratorstorefrontcommons.forms.RegisterForm;
import de.hybris.platform.b2bacceleratoraddon.constants.B2bacceleratoraddonWebConstants;
import de.hybris.platform.b2bacceleratoraddon.forms.RegistrationForm;
import de.hybris.platform.b2bacceleratorfacades.exception.CustomerAlreadyExistsException;
import de.hybris.platform.b2bacceleratorfacades.exception.RegistrationNotEnabledException;
import de.hybris.platform.b2bacceleratorfacades.registration.B2BRegistrationFacade;
import de.hybris.platform.b2bcommercefacades.data.B2BRegistrationData;
import de.hybris.platform.cms2.exceptions.CMSItemNotFoundException;
import de.hybris.platform.cms2.model.pages.AbstractPageModel;
import de.hybris.platform.cms2.model.pages.ContentPageModel;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.commercefacades.user.data.CountryData;
import de.hybris.platform.commercefacades.user.data.CustomerData;
import de.hybris.platform.converters.Converters;
import de.hybris.platform.core.model.c2l.CountryModel;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.i18n.CommonI18NService;
import de.hybris.platform.servicelayer.session.SessionService;
import de.hybris.platform.util.localization.Localization;
import org.apache.log4j.Logger;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.validation.Validator;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;
import org.springframework.web.servlet.support.RequestContextUtils;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_ACCEPTED_CREDIT_CARD_PAYMENT_METHODS;
import static com.braintree.controllers.Braintreeb2baddonControllerConstants.*;
import static com.braintree.controllers.Braintreeb2baddonControllerConstants.Views.Pages.Account.AccountLoginPage;
import static com.braintree.controllers.Braintreeb2baddonControllerConstants.Views.Pages.Checkout.CheckoutLoginPage;
import static de.hybris.platform.util.localization.Localization.getLocalizedString;

@Controller
@RequestMapping(value = "/paypal/connect/noauthentication")
public class BrainTreeConnectWithPayPalNoAuthenticationController extends AbstractLoginPageController {

    private static final Logger LOG = Logger.getLogger(BrainTreeConnectWithPayPalNoAuthenticationController.class);

    public static final String CHECKOUT = "checkout";
    private static final String IS_USER_APPROVED = "isUserApproved";
    private static final String PAYER_ID = "payerId";
    private static final String ADDRESS_DATA = "addressData";
    private static final String SHOULD_SAVE_ADDRESS = "shouldSaveAddress";
    private static final String SHOULD_SAVE_PAYMENT_INFO = "shouldSavePaymentInfo";
    public static final String USER_EMAIL = "userEmail";
    private static final String SAVE_ADDRESS_AND_PAYMENT_INFOS = "saveAddressAndPaymentInfos";
    private static final String SHOULD_SET_UP_PASSWORD_MESSAGE = "shouldShowSetUpPasswordMessage";
    private static final String SAVE_ADDRESS_AND_PAYMENT_INFOS_URL = "paypal/connect/noauthentication/saveAddressAndPaymentInfos";
    private static final String NOT_SAVE_PAYMENT_AND_ADDRESS_INFOS = "notSavePaymentAndAddressInfos";
    private static final String NOT_SAVE_PAYMENT_AND_ADDRESS_INFOS_URL = "paypal/connect/noauthentication/notSavePaymentAndAddressInfos";
    private static final String PAYPAL_CHECKOUT_DATA = "payPalCheckoutData";
    public static final String REDIRECT = "redirect:";
    public static final String CONNECT_LOGIN_SAVE_PAYMENT_INFO_AND_ADDRESS_PAGE = "/paypal/connect/noauthentication/savePaymentInfoAndAddressPage";
    private static final String SAVE_ADDRESS_AND_PAYMENT_INFOS_CHECKOUT_URL = "paypal/connect/noauthentication/saveAddressAndPaymentInfos/checkout";
    private static final String NOT_SAVE_PAYMENT_AND_ADDRESS_INFOS_CHECKOUT_URL = "paypal/connect/noauthentication/notSavePaymentAndAddressInfos/checkout";
    private static final String UNVERIFIED_ACCOUNT_ERROR = "braintree.paypal.unverified.account.error";

    @Resource(name = "brainTreePaymentFacadeImpl")
    private DefaultBrainTreePaymentFacade brainTreePaymentFacade;

    @Resource(name = "brainTreePaymentService")
    private BrainTreePaymentService brainTreePaymentService;

    @Resource(name = "braintreePayPalResponseExpressCheckoutHandler")
    BraintreePayPalResponseExpressCheckoutHandler braintreePayPalResponseExpressCheckoutHandler;

    @Resource(name = "brainTreeConfigService")
    private BrainTreeConfigService brainTreeConfigService;

    @Resource(name = "brainTreeCheckoutFacade")
    private DefaultBrainTreeCheckoutFacade brainTreeCheckoutFacade;

    @Resource(name = "checkoutFlowFacade")
    private CheckoutFlowFacade checkoutFlowFacade;

    @Resource(name = "brainTreeConnectWithPayPalFacade")
    private BrainTreeConnectWithPayPalFacade brainTreeConnectWithPayPalFacade;

    @Resource(name = "brainTreeRegistrationUserFacade")
    private BrainTreeRegistrationUserFacade brainTreeRegistrationUserFacade;

    @Resource(name = "b2bRegistrationFacade")
    private B2BRegistrationFacade b2bRegistrationFacade;

    @Resource(name = "braintreeB2BRegistrationUserFacade")
    private BraintreeB2BRegistrationUserFacade brainTreeB2BRegistrationUserFacade;

    @Resource(name = "userFacade")
    private BrainTreeUserFacade brainTreeUserFacade;

    @Resource(name = "brainTreeSetUpPasswordFacade")
    private BrainTreeSetUpPasswordFacade brainTreeSetUpPasswordFacade;

    @Resource(name = "payPalAutoLoginStrategy")
    private PayPalAutoLoginStrategy payPalAutoLoginStrategy;

    @Resource(name = "sessionService")
    private SessionService sessionService;

    @Resource(name = "commonI18NService")
    private CommonI18NService commonI18NService;

    @Resource(name = "countryConverter")
    private Converter<CountryModel, CountryData> countryConverter;

    @Resource(name = "brainTreeConnectWithPayPalRegistrationValidator")
    private Validator registrationValidator;

    @GetMapping(value = "/register")
    public String getRegister(final RegisterForm form, final HttpServletRequest request,
                              final HttpServletResponse response, final HttpSession session, final Model model,
                              @RequestParam(required = false) String code, final RedirectAttributes redirectAttributes) throws CMSItemNotFoundException {
        sessionService.setAttribute(IS_USER_LOGGING_IN_FROM_LOGIN_PAGE, true);
        return getRegistrationFlow(form, request, response, session, model, code, redirectAttributes);
    }

    @GetMapping(value = "/register/checkout")
    public String getCheckoutRegister(final RegisterForm form, final HttpServletRequest request,
                                      final HttpServletResponse response, final HttpSession session, final Model model,
                                      @RequestParam(required = false) String code, final RedirectAttributes redirectAttributes) throws CMSItemNotFoundException {
        sessionService.setAttribute(IS_USER_LOGGING_IN_FROM_LOGIN_PAGE, false);
        model.addAttribute("expressCheckoutAllowed", checkoutFlowFacade.isExpressCheckoutEnabledForStore());
        return getRegistrationFlow(form, request, response, session, model, code, redirectAttributes);
    }


    @PostMapping(value = {"/saveAddressAndPaymentInfos", "/saveAddressAndPaymentInfos/checkout"})
    public String doHandlePayPalResponse(final HttpServletRequest request, final HttpServletResponse response,
                                         final Model model, RedirectAttributes redirectAttributes) {
        final String email = sessionService.getAttribute(USER_EMAIL);
        final BrainTreeConnectWithPayPalAddressData addressData = sessionService.getAttribute(ADDRESS_DATA);
        if ((boolean) sessionService.getAttribute(SHOULD_SAVE_PAYMENT_INFO)) {
            try {
                savePayPalPayment(request, email);
                GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.INFO_MESSAGES_HOLDER, "paypal.connect.saved.payment.method");
            } catch (Exception e) {
                LOG.error("Error during adding payment method for new customer", e);
                GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.INFO_MESSAGES_HOLDER, "paypal.connect.saved.payment.method.error");
            }
        }

        if ((boolean) sessionService.getAttribute(SHOULD_SAVE_ADDRESS)) {
            try {
                if (addressData != null) {
                    brainTreeConnectWithPayPalFacade.addAddressForNewUser(addressData, email);
                    GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.INFO_MESSAGES_HOLDER, "paypal.connect.saved.address");
                }
            } catch (Exception e) {
                LOG.error("Error during adding address for new customer", e);
                GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.INFO_MESSAGES_HOLDER, "paypal.connect.saved.address.error");
            }
        }
        if ((boolean) sessionService.getAttribute(IS_USER_APPROVED)) {
            String payerId = sessionService.getAttribute(PAYER_ID);
            GlobalMessages.addFlashMessage(RequestContextUtils.getOutputFlashMap(request),
                    GlobalMessages.INFO_MESSAGES_HOLDER,
                    Localization.getLocalizedString(BRAINTREE_PAYPAL_CONNECT_LOGIN_LINKED_ACCOUNTS), null);
            removeSessionAttribute();
            return processLogin(request, response, payerId);
        }
        GlobalMessages.addFlashMessage(RequestContextUtils.getOutputFlashMap(request),
                GlobalMessages.INFO_MESSAGES_HOLDER,
                Localization.getLocalizedString(REGISTER_REQUEST_NOR_APPROVED), null);
        removeSessionAttribute();
        return LOGIN_PAGE_REDIRECT;
    }

    @PostMapping(value = {"/notSavePaymentAndAddressInfos", "/notSavePaymentAndAddressInfos/checkout"})
    public String notSavePaymentInfo(final RegisterForm form, final HttpServletRequest request,
                                     final HttpServletResponse response,
                                     final Model model, RedirectAttributes redirectAttributes) {
        model.addAttribute(form);
        if ((boolean) sessionService.getAttribute(IS_USER_APPROVED)) {
            String payerId = sessionService.getAttribute(PAYER_ID);
            GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.INFO_MESSAGES_HOLDER,
                    Localization.getLocalizedString(BRAINTREE_PAYPAL_CONNECT_LOGIN_LINKED_ACCOUNTS));
            removeSessionAttribute();
            return processLogin(request, response, payerId);
        }
        GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.INFO_MESSAGES_HOLDER,
                Localization.getLocalizedString(REGISTER_REQUEST_CREATED));
        removeSessionAttribute();
        return LOGIN_PAGE_REDIRECT;
    }

    @RequestMapping(value = "/savePaymentInfoAndAddressPage", method = RequestMethod.GET)
    public String getSavePaymentInfoAndAddressPage(final Model model, HttpServletRequest request) throws CMSItemNotFoundException {
        setModelAttributesForPaymentAndAddressInfo(request, model);
        storeCmsPageInModel(model, getCmsPage());
        setUpMetaDataForContentPage(model, (ContentPageModel) getCmsPage());
        return SAVE_PAYMENT_INFO_PAGE;
    }


    @PostMapping(value = "/register")
    public String doRegister(final RegistrationForm form, final BindingResult bindingResult,
                             final HttpSession session, final Model model, final HttpServletRequest request,
                             final RedirectAttributes redirectAttributes)
            throws CMSItemNotFoundException {
        return doRegistrationFlow(form, bindingResult, session, model, request, redirectAttributes);
    }

    @PostMapping(value = "/register/checkout")
    public String doCheckoutRegister(final RegistrationForm form, final BindingResult bindingResult,
                                     final HttpSession session, final Model model, final HttpServletRequest request,
                                     final RedirectAttributes redirectAttributes)
            throws CMSItemNotFoundException {
        return doRegistrationFlow(form, bindingResult, session, model, request, redirectAttributes);
    }

    private String getRegistrationFlow(final RegisterForm form, final HttpServletRequest request,
                                       final HttpServletResponse response, final HttpSession session, final Model model,
                                       final String code, RedirectAttributes redirectAttributes) throws CMSItemNotFoundException {
        if (code == null) {
            return getDefaultLoginPage(false, session, model);
        }
        String accessToken;
        String email;
        BrainTreeConnectWithPayPalUserExistB2BData userExistB2BData;
        BrainTreeConnectWithPayPalUserData userData;
        try {
            accessToken = brainTreeConnectWithPayPalFacade.exchangeAuthorizationCodeToAccessToken(code);
            userData = brainTreeConnectWithPayPalFacade.getUserDataByAccessToken(accessToken);
            email = brainTreeRegistrationUserFacade.getEmailFromPayPalUserData(userData);
            userExistB2BData = brainTreeB2BRegistrationUserFacade.isCustomerExistAndApprovedByUID(email);
        } catch (BrainTreeUnverifiedAccountPayPalException e) {
            GlobalMessages.addFlashMessage(redirectAttributes,
                    GlobalMessages.ERROR_MESSAGES_HOLDER,
                    getLocalizedString(UNVERIFIED_ACCOUNT_ERROR));
            return LOGIN_PAGE_REDIRECT;
        } catch (AdapterException e) {
            LOG.error("Adapter exception", e);
            form.setTermsCheck(false);
            model.addAttribute(form);
            model.addAttribute(new LoginForm());
            model.addAttribute(new GuestForm());
            return handleRegistrationError(model);
        }


        if (userExistB2BData.isExist() && userExistB2BData.isApproved()) {
            if (!brainTreeUserFacade.isCustomerWithPayerIdExist(userData.getPayerId())) {
                if (!brainTreeConfigService.isEnableLoginViaPayPalToTheExistingUser()) {
                    GlobalMessages.addFlashMessage(RequestContextUtils.getOutputFlashMap(request),
                            GlobalMessages.ERROR_MESSAGES_HOLDER,
                            getLocalizedString(DISABLE_LOGIN_VIA_PAYPAL_TO_THE_EXISTING_USER_MESSAGE), null);
                    return LOGIN_PAGE_REDIRECT;
                }
                brainTreeRegistrationUserFacade.setPayerIdToUser(email, userData.getPayerId());
                final boolean isPayPalAddressPresent = brainTreeConnectWithPayPalFacade.isPayPalAddressPresent(email, userData.getAddress());
                final boolean isPayPalPaymentMethodPresent = brainTreeConnectWithPayPalFacade.isPayPalPaymentMethodPresentOrIntentOrder(email);
                if (brainTreeConfigService.isPayPalConnectAddPaymentMethodFlow() && paypalMethodAndAddressPresent(isPayPalAddressPresent, isPayPalPaymentMethodPresent)) {
                    sessionService.setAttribute(IS_USER_APPROVED, true);

                    GlobalMessages.addFlashMessage(RequestContextUtils.getOutputFlashMap(request),
                            GlobalMessages.INFO_MESSAGES_HOLDER,
                            Localization.getLocalizedString(BRAINTREE_PAYPAL_CONNECT_LOGIN_LINKED_ACCOUNTS), null);
                    processLogin(request, response, userData.getPayerId());
                } else if (brainTreeConfigService.isPayPalConnectAddPaymentMethodFlow() && !paypalMethodAndAddressPresent(isPayPalAddressPresent, isPayPalPaymentMethodPresent)) {
                    setSessionAttributeForPaymentAndAddressInfo(!isPayPalAddressPresent, !isPayPalPaymentMethodPresent, true, userData, email);
                    return REDIRECT + CONNECT_LOGIN_SAVE_PAYMENT_INFO_AND_ADDRESS_PAGE;
                }

                GlobalMessages.addFlashMessage(RequestContextUtils.getOutputFlashMap(request),
                        GlobalMessages.INFO_MESSAGES_HOLDER,
                        Localization.getLocalizedString(BRAINTREE_PAYPAL_CONNECT_LOGIN_LINKED_ACCOUNTS), null);
            }
            if (Objects.nonNull(sessionService.getAttribute(SHOULD_SET_UP_PASSWORD_MESSAGE))) {
                GlobalMessages.addFlashMessage(RequestContextUtils.getOutputFlashMap(request),
                        GlobalMessages.INFO_MESSAGES_HOLDER,
                        getLocalizedString(BRAINTREE_PAYPAL_CONNECT_REGISTRATION_MESSAGE_SET_UP_PASSWORD_FOR_NEW_USER), null);
                sessionService.removeAttribute(SHOULD_SET_UP_PASSWORD_MESSAGE);
            }
            return processLogin(request, response, userData.getPayerId());
        } else if (userExistB2BData.isExist()) {
            if (!brainTreeConfigService.isEnableLoginViaPayPalToTheExistingUser() && userExistB2BData.isApproved()) {
                GlobalMessages.addFlashMessage(RequestContextUtils.getOutputFlashMap(request),
                        GlobalMessages.ERROR_MESSAGES_HOLDER,
                        getLocalizedString(DISABLE_LOGIN_VIA_PAYPAL_TO_THE_EXISTING_USER_MESSAGE), null);
                return LOGIN_PAGE_REDIRECT;
            }
            final boolean isPayPalAddressPresent = brainTreeConnectWithPayPalFacade.isPayPalAddressPresent(email, userData.getAddress());
            final boolean isPayPalPaymentMethodPresent = brainTreeConnectWithPayPalFacade.isPayPalPaymentMethodPresentOrIntentOrder(email);
            if (brainTreeConfigService.isPayPalConnectAddPaymentMethodFlow() && !brainTreeUserFacade.isCustomerWithPayerIdExist(userData.getPayerId())
                    && paypalMethodAndAddressPresent(isPayPalPaymentMethodPresent, isPayPalAddressPresent)) {
                return LOGIN_PAGE_REDIRECT;
            } else if (brainTreeConfigService.isPayPalConnectAddPaymentMethodFlow() && !brainTreeUserFacade.isCustomerWithPayerIdExist(userData.getPayerId())
                    && !paypalMethodAndAddressPresent(isPayPalPaymentMethodPresent, isPayPalAddressPresent)) {
                setSessionAttributeForPaymentAndAddressInfo(!isPayPalAddressPresent, !isPayPalPaymentMethodPresent, false, userData, email);
                return REDIRECT + CONNECT_LOGIN_SAVE_PAYMENT_INFO_AND_ADDRESS_PAGE;
            }
            GlobalMessages.addFlashMessage(RequestContextUtils.getOutputFlashMap(request),
                    GlobalMessages.INFO_MESSAGES_HOLDER,
                    Localization.getLocalizedString(REGISTER_REQUEST_CREATED), null);
            return (boolean) sessionService.getAttribute(IS_USER_LOGGING_IN_FROM_LOGIN_PAGE)
                    ? LOGIN_PAGE_REDIRECT : CHECKOUT_LOGIN_PAGE_REDIRECT;
        }

        sessionService.setAttribute(ACCESS_TOKEN, accessToken);
        final List<CountryData> countries = Converters
                .convertAll(commonI18NService.getAllCountries(), countryConverter);
        model.addAttribute("titles", brainTreeUserFacade.getTitles());
        model.addAttribute("countries", countries);
        populateModelCmsContent(model,
                getContentPageForLabelOrId(B2bacceleratoraddonWebConstants.CMS_REGISTER_PAGE_NAME));
        model.addAttribute(new RegistrationForm());
        return BRAINTREE_CONNECT_WITH_PAYPAL_ADDITIONAL_DATA;
    }

    private String doRegistrationFlow(final RegistrationForm form, final BindingResult bindingResult,
                                      final HttpSession session, final Model model, HttpServletRequest request, final RedirectAttributes redirectAttributes)
            throws CMSItemNotFoundException {
        populateModelCmsContent(model, getContentPageForLabelOrId(B2bacceleratoraddonWebConstants.CMS_REGISTER_PAGE_NAME));
        model.addAttribute(form);

        registrationValidator.validate(form, bindingResult);
        if (bindingResult.hasErrors()) {
            final List<CountryData> countries = Converters
                    .convertAll(commonI18NService.getAllCountries(), countryConverter);
            model.addAttribute("titles", brainTreeUserFacade.getTitles());
            model.addAttribute("countries", countries);
            return BRAINTREE_CONNECT_WITH_PAYPAL_ADDITIONAL_DATA;
        }

        final String accessToken = sessionService.getAttribute(ACCESS_TOKEN);
        final BrainTreeConnectWithPayPalUserData userData =
                brainTreeConnectWithPayPalFacade.getUserDataByAccessToken(accessToken);
        session.removeAttribute(ACCESS_TOKEN);

        try {
            b2bRegistrationFacade.register(convertFormAndPayPalDataToRegistrationData(form, userData));
            final String email = brainTreeRegistrationUserFacade.getEmailFromPayPalUserData(userData);
            brainTreeRegistrationUserFacade.setPayerIdToUser(email, userData.getPayerId());
            brainTreeSetUpPasswordFacade.setPasswordForSingleAuthenticationActiveToCustomer(email, false);
            sessionService.setAttribute(SHOULD_SET_UP_PASSWORD_MESSAGE, Boolean.TRUE);
        } catch (BrainTreeUnverifiedAccountPayPalException e) {
            GlobalMessages.addFlashMessage(redirectAttributes,
                    GlobalMessages.ERROR_MESSAGES_HOLDER,
                    Localization.getLocalizedString(UNVERIFIED_ACCOUNT_ERROR));
            return LOGIN_PAGE_REDIRECT;
        } catch (final CustomerAlreadyExistsException | RegistrationNotEnabledException e) {
            LOG.error("Failed to register account. Account already exists.", e);
            GlobalMessages.addErrorMessage(model, Localization.getLocalizedString(PAYPAL_CONNECT_GLOBAL_ERROR));
            return getRegistrationView();
        }
        final String email = sessionService.getAttribute(USER_EMAIL);
        if (brainTreeConfigService.isPayPalConnectAddPaymentMethodFlow()) {
            final boolean shutSavePaymentInfo = brainTreeConnectWithPayPalFacade.shouldSavePayPalPaymentInfo();
            setSessionAttributeForPaymentAndAddressInfo(true, shutSavePaymentInfo, false, userData, email);
            return REDIRECT + CONNECT_LOGIN_SAVE_PAYMENT_INFO_AND_ADDRESS_PAGE;
        }
        GlobalMessages.addFlashMessage(RequestContextUtils.getOutputFlashMap(request),
                GlobalMessages.INFO_MESSAGES_HOLDER, getLocalizedString(REGISTER_REQUEST_CREATED), null);
        return (boolean) sessionService.getAttribute(IS_USER_LOGGING_IN_FROM_LOGIN_PAGE)
                ? REDIRECT_TO_NOAUTHENTICATION_REGISTER : REDIRECT_TO_NOAUTHENTICATION_REGISTER_CHECKOUT;
    }

    private String processLogin(final HttpServletRequest request, final HttpServletResponse response, String payerId) {
        final CustomerData customer = brainTreeUserFacade.getCustomerByPayerId(payerId);
        payPalAutoLoginStrategy.login(customer.getUid(), request, response);
        return getSuccessRedirect(request, response);
    }

    @Override
    protected AbstractPageModel getCmsPage() throws CMSItemNotFoundException {
        final boolean isLoginPage = sessionService.getAttribute(IS_USER_LOGGING_IN_FROM_LOGIN_PAGE);
        if (isLoginPage) {
            return getContentPageForLabelOrId(LOGIN_LABEL);
        }
        return getContentPageForLabelOrId(CHECKOUT_LOGIN_LABEL);
    }

    @Override
    protected String getSuccessRedirect(HttpServletRequest request, HttpServletResponse response) {
        if (request.getServletPath().contains(CHECKOUT)) {
            return getCheckoutRedirectUrl();
        }
        return REDIRECT_PREFIX + ROOT;
    }

    @Override
    protected String getView() {
        final boolean isLoginPage = sessionService.getAttribute(IS_USER_LOGGING_IN_FROM_LOGIN_PAGE);
        sessionService.removeAttribute(IS_USER_LOGGING_IN_FROM_LOGIN_PAGE);
        if (isLoginPage) {
            return AccountLoginPage;
        }
        return CheckoutLoginPage;
    }

    @Override
    protected String getCheckoutRedirectUrl() {
        if (getUserFacade().isAnonymousUser()) {
            return REDIRECT_TO_LOGIN_FOR_CHECKOUT;
        }
        return REDIRECT_TO_MULTISTEP_CHECKOUT_FIRST_STEP;
    }

    protected void populateModelCmsContent(final Model model, final ContentPageModel contentPageModel) {

        storeCmsPageInModel(model, contentPageModel);
        setUpMetaDataForContentPage(model, contentPageModel);

        final Breadcrumb registrationBreadcrumbEntry = new Breadcrumb("#",
                getMessageSource().getMessage(SCP_LINK_CREATE_ACCOUNT, null, getI18nService().getCurrentLocale()), null);
        model.addAttribute(BREADCRUBMS, Collections.singletonList(registrationBreadcrumbEntry));
    }

    private B2BRegistrationData convertFormAndPayPalDataToRegistrationData(RegistrationForm form,
                                                                           BrainTreeConnectWithPayPalUserData userData) {
        final B2BRegistrationData registrationData = new B2BRegistrationData();
        BeanUtils.copyProperties(form, registrationData);
        registrationData.setName(userData.getName());
        registrationData.setEmail(brainTreeRegistrationUserFacade.getEmailFromPayPalUserData(userData));
        return registrationData;
    }

    private String getRegistrationView() {
        return B2bacceleratoraddonWebConstants.Views.REGISTRATION_PAGE;
    }

    private void setSessionAttributeForPaymentAndAddressInfo(final boolean shutSaveAddress,
                                                             final boolean shutSavePaymentInfo,
                                                             final boolean isUserApproved,
                                                             final BrainTreeConnectWithPayPalUserData userData,
                                                             final String email) {

        sessionService.setAttribute(SHOULD_SAVE_ADDRESS, shutSaveAddress);
        sessionService.setAttribute(SHOULD_SAVE_PAYMENT_INFO, shutSavePaymentInfo);
        sessionService.setAttribute(IS_USER_APPROVED, isUserApproved);
        sessionService.setAttribute(PAYER_ID, userData.getPayerId());
        sessionService.setAttribute(ADDRESS_DATA, userData.getAddress());
        sessionService.setAttribute(USER_EMAIL, email);
    }

    private void removeSessionAttribute() {
        sessionService.removeAttribute(PAYER_ID);
        sessionService.removeAttribute(ADDRESS_DATA);
        sessionService.removeAttribute(SHOULD_SAVE_ADDRESS);
        sessionService.removeAttribute(SHOULD_SAVE_PAYMENT_INFO);
        sessionService.removeAttribute(USER_EMAIL);
        sessionService.removeAttribute(IS_USER_APPROVED);
    }

    private void setModelAttributesForPaymentAndAddressInfo(HttpServletRequest request, Model model) {
        final PayPalCheckoutData checkoutData = brainTreeCheckoutFacade.getPayPalCheckoutData(BraintreePageType.LOGIN);
        model.addAttribute(PAYPAL_CHECKOUT_DATA, checkoutData);
        model.addAttribute(SHOULD_SAVE_ADDRESS, sessionService.getAttribute(SHOULD_SAVE_ADDRESS));
        model.addAttribute(SHOULD_SAVE_PAYMENT_INFO, sessionService.getAttribute(SHOULD_SAVE_PAYMENT_INFO));
        if ((boolean) sessionService.getAttribute(IS_USER_LOGGING_IN_FROM_LOGIN_PAGE)) {
            model.addAttribute(NOT_SAVE_PAYMENT_AND_ADDRESS_INFOS, NOT_SAVE_PAYMENT_AND_ADDRESS_INFOS_URL);
            model.addAttribute(SAVE_ADDRESS_AND_PAYMENT_INFOS, SAVE_ADDRESS_AND_PAYMENT_INFOS_URL);
        } else {
            model.addAttribute(NOT_SAVE_PAYMENT_AND_ADDRESS_INFOS, NOT_SAVE_PAYMENT_AND_ADDRESS_INFOS_CHECKOUT_URL);
            model.addAttribute(SAVE_ADDRESS_AND_PAYMENT_INFOS, SAVE_ADDRESS_AND_PAYMENT_INFOS_CHECKOUT_URL);
        }
    }

    private void savePayPalPayment(final HttpServletRequest request, String email) {
        PayPalExpressResponse payPalExpressResponse;
        AddressData hybrisBillingAddress;
        payPalExpressResponse = braintreePayPalResponseExpressCheckoutHandler.handlePayPalResponse(request);
        String payPalEmail = payPalExpressResponse.getDetails().getEmail();
        final String paymentProvider = Optional.ofNullable(payPalExpressResponse.getType())
                .orElse(BrainTreePaymentMethod.BRAINTREEPAYPALEXPRESS.getCode());
        if (Optional.ofNullable(payPalExpressResponse.getDetails()).map(PayPalDetails::getBillingAddress).isPresent()) {
            hybrisBillingAddress = braintreePayPalResponseExpressCheckoutHandler.getPayPalAddress(
                    payPalExpressResponse.getDetails(), payPalExpressResponse.getDetails().getBillingAddress());
            hybrisBillingAddress.setEmail(payPalExpressResponse.getDetails().getEmail());
        } else {
            hybrisBillingAddress = new AddressData();
            hybrisBillingAddress.setEmail(payPalEmail);
        }

        final BrainTreeSubscriptionInfoData paymentInfoData = GenericBuilder.of(BrainTreeSubscriptionInfoData::new)
                .with(BrainTreeSubscriptionInfoData::setPaymentProvider, paymentProvider)
                .with(BrainTreeSubscriptionInfoData::setEmail, payPalEmail)
                .with(BrainTreeSubscriptionInfoData::setNonce, payPalExpressResponse.getNonce())
                .with(BrainTreeSubscriptionInfoData::setSavePaymentInfo, true)
                .with(BrainTreeSubscriptionInfoData::setPaypalFundingSource, payPalExpressResponse.getFundingSource())
                .with(BrainTreeSubscriptionInfoData::setAddressData, hybrisBillingAddress)
                .build();

        Optional.ofNullable(payPalExpressResponse.getCardDetails()).ifPresent(data -> {
            paymentInfoData.setCardType(data.getCardType());
            paymentInfoData.setExpirationYear(data.getExpirationYear());
            paymentInfoData.setExpirationMonth(data.getExpirationMonth());
            paymentInfoData.setCardNumber(BrainTreeUtils.formatCardNumber(data.getLastFour()));
            if (BrainTreeUtils.isSrcPayment(paymentProvider)) {
                paymentInfoData.setImageSource(brainTreePaymentService
                        .getImageSourceForPaymentMethod(data.getCardType(), BRAINTREE_ACCEPTED_CREDIT_CARD_PAYMENT_METHODS));
            }
        });
        brainTreePaymentFacade.createPaymentInfoForLogin(paymentInfoData, email);
    }

    private boolean paypalMethodAndAddressPresent(boolean isPayPalPaymentMethodPresent, boolean isPayPalAddressPresent) {
        return isPayPalPaymentMethodPresent && isPayPalAddressPresent;
    }
}
