package com.braintree.facade.impl;

import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.when;

import com.braintree.constants.BraintreeConstants;
import com.braintree.enums.BrainTreePaymentMethod;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.b2bacceleratorfacades.order.data.ScheduledCartData;
import de.hybris.platform.b2bacceleratorservices.customer.B2BCustomerAccountService;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.orderscheduling.model.CartToOrderCronJobModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.user.UserService;
import java.util.ArrayList;
import java.util.List;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;

@UnitTest
public class DefaultBrainTreeB2BOrderFacadeTest {

    @Mock
    private UserService userService;

    @Mock
    private B2BCustomerAccountService customerAccountService;

    @Mock
    private Converter<CartToOrderCronJobModel, ScheduledCartData> scheduledCartConverter;

    @InjectMocks
    private DefaultBrainTreeB2BOrderFacade unit;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
    }

    @Test
    public void isCustomerHasActiveReplenishmentShouldReturnFalseWhenCustomerDoesNotHaveReplenishment() {
        CustomerModel customerModel = Mockito.mock(CustomerModel.class);
        when(userService.getCurrentUser()).thenReturn(customerModel);
        List<CartToOrderCronJobModel> jobModels = new ArrayList<>();
        doReturn(jobModels).when(customerAccountService).getCartToOrderCronJobsForUser(customerModel);

        Assert.assertFalse(unit.isCustomerHasActiveReplenishmentForSavedPayPalOrCreditCard());
    }

    @Test
    public void isCustomerHasActiveReplenishmentShouldReturnFalseWhenCustomerHasGooglePayReplenishment() {
        CustomerModel customerModel = Mockito.mock(CustomerModel.class);
        when(userService.getCurrentUser()).thenReturn(customerModel);
        List<CartToOrderCronJobModel> jobModels = new ArrayList<>();
        CartToOrderCronJobModel googleReplenishment = Mockito.mock(CartToOrderCronJobModel.class);
        jobModels.add(googleReplenishment);
        doReturn(jobModels).when(customerAccountService).getCartToOrderCronJobsForUser(customerModel);

        ScheduledCartData cartData = Mockito.mock(ScheduledCartData.class);
        when(cartData.isActive()).thenReturn(Boolean.TRUE);
        CCPaymentInfoData ccPaymentInfoData = Mockito.mock(CCPaymentInfoData.class);
        when(ccPaymentInfoData.getSubscriptionId()).thenReturn(BrainTreePaymentMethod.ANDROIDPAYCARD.getCode());
        when(cartData.getPaymentInfo()).thenReturn(ccPaymentInfoData);

        when(scheduledCartConverter.convert(googleReplenishment)).thenReturn(cartData);

        Assert.assertFalse(unit.isCustomerHasActiveReplenishmentForSavedPayPalOrCreditCard());
    }

    @Test
    public void isCustomerHasActiveReplenishmentShouldReturnTrueWhenCustomerHasPayPalReplenishment() {
        CustomerModel customerModel = Mockito.mock(CustomerModel.class);
        when(userService.getCurrentUser()).thenReturn(customerModel);
        List<CartToOrderCronJobModel> jobModels = new ArrayList<>();
        CartToOrderCronJobModel googleReplenishment = Mockito.mock(CartToOrderCronJobModel.class);
        jobModels.add(googleReplenishment);
        doReturn(jobModels).when(customerAccountService).getCartToOrderCronJobsForUser(customerModel);

        ScheduledCartData cartData = Mockito.mock(ScheduledCartData.class);
        when(cartData.isActive()).thenReturn(Boolean.TRUE);
        CCPaymentInfoData ccPaymentInfoData = Mockito.mock(CCPaymentInfoData.class);
        when(ccPaymentInfoData.getSubscriptionId()).thenReturn(BrainTreePaymentMethod.PAYPALACCOUNT.getCode());
        when(cartData.getPaymentInfo()).thenReturn(ccPaymentInfoData);

        when(scheduledCartConverter.convert(googleReplenishment)).thenReturn(cartData);

        Assert.assertTrue(unit.isCustomerHasActiveReplenishmentForSavedPayPalOrCreditCard());
    }

    @Test
    public void isCustomerHasActiveReplenishmentShouldReturnTrueWhenCustomerHasNotActivePayPalReplenishment() {
        CustomerModel customerModel = Mockito.mock(CustomerModel.class);
        when(userService.getCurrentUser()).thenReturn(customerModel);
        List<CartToOrderCronJobModel> jobModels = new ArrayList<>();
        CartToOrderCronJobModel googleReplenishment = Mockito.mock(CartToOrderCronJobModel.class);
        jobModels.add(googleReplenishment);
        doReturn(jobModels).when(customerAccountService).getCartToOrderCronJobsForUser(customerModel);

        ScheduledCartData cartData = Mockito.mock(ScheduledCartData.class);
        when(cartData.isActive()).thenReturn(Boolean.FALSE);
        CCPaymentInfoData ccPaymentInfoData = Mockito.mock(CCPaymentInfoData.class);
        when(ccPaymentInfoData.getSubscriptionId()).thenReturn(BrainTreePaymentMethod.PAYPALACCOUNT.getCode());
        when(cartData.getPaymentInfo()).thenReturn(ccPaymentInfoData);

        when(scheduledCartConverter.convert(googleReplenishment)).thenReturn(cartData);

        Assert.assertFalse(unit.isCustomerHasActiveReplenishmentForSavedPayPalOrCreditCard());
    }

}
