package com.braintree.controllers.pages;

import static com.braintree.constants.BraintreeConstants.PropertyConfiguration.BRAINTREE_ACCEPTED_CREDIT_CARD_PAYMENT_METHODS;
import static com.braintree.controllers.BraintreeaddonControllerConstants.CLIENT_TOKEN;
import static de.hybris.platform.util.localization.Localization.getLocalizedString;

import com.braintree.exceptions.BraintreeDeletePaymentDetailsWithNotCapturedOrderException;
import com.braintree.facade.BrainTreeUserFacade;
import com.braintree.facade.impl.DefaultBrainTreeCheckoutFacade;
import com.braintree.facade.impl.DefaultBrainTreePaymentFacade;
import com.braintree.facade.impl.DefaultBrainTreeSetUpPasswordFacade;
import com.braintree.form.UpdateCreditCardBillingAddressForm;
import com.braintree.hybris.data.BrainTreeSubscriptionInfoData;
import com.braintree.method.BrainTreePaymentService;
import com.braintree.util.BrainTreeUtils;
import com.braintree.util.GenericBuilder;
import com.braintree.validator.BrainTreePasswordValidator;
import com.fasterxml.jackson.databind.ObjectMapper;
import de.hybris.platform.acceleratorstorefrontcommons.annotations.RequireHardLogIn;
import de.hybris.platform.acceleratorstorefrontcommons.breadcrumb.Breadcrumb;
import de.hybris.platform.acceleratorstorefrontcommons.breadcrumb.ResourceBreadcrumbBuilder;
import de.hybris.platform.acceleratorstorefrontcommons.controllers.ThirdPartyConstants;
import de.hybris.platform.acceleratorstorefrontcommons.controllers.pages.AbstractPageController;
import de.hybris.platform.acceleratorstorefrontcommons.controllers.util.GlobalMessages;
import de.hybris.platform.acceleratorstorefrontcommons.forms.AddressForm;
import de.hybris.platform.acceleratorstorefrontcommons.forms.UpdatePasswordForm;
import de.hybris.platform.acceleratorstorefrontcommons.forms.validation.AddressValidator;
import de.hybris.platform.acceleratorstorefrontcommons.util.AddressDataUtil;
import de.hybris.platform.cms2.exceptions.CMSItemNotFoundException;
import de.hybris.platform.cms2.model.pages.ContentPageModel;
import de.hybris.platform.commercefacades.i18n.I18NFacade;
import de.hybris.platform.commercefacades.order.CheckoutFacade;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.commercefacades.user.data.CountryData;
import de.hybris.platform.commercefacades.user.data.TitleData;
import de.hybris.platform.commerceservices.enums.CountryType;
import de.hybris.platform.payment.AdapterException;

import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import javax.annotation.Resource;

import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;


@Controller
@RequestMapping("/my-account")
public class BrainTreeAccountPageController extends AbstractPageController {

    private static final Logger LOGGER = Logger.getLogger(BrainTreeAccountPageController.class);
    private static final String REDIRECT_TO_PAYMENT_INFO_PAGE = REDIRECT_PREFIX + "/my-account/payment-details";
    private static final String REDIRECT_TO_ADD_PAYMENT_INFO_PAGE = REDIRECT_PREFIX + "/my-account/add-payment-method";
    private static final String COUNTRY_DATA_ATTR = "countryData";
    private static final String ADDRESS_FORM_ATTR = "addressForm";
    private static final String COUNTRY_ATTR = "country";
    private static final String REGIONS_ATTR = "regions";
    private static final String TITLE_DATA_ATTR = "titleData";
    private static final String ADDRESS_DATA_ATTR = "addressData";
    private static final String FORM_GLOBAL_ERROR = "form.global.error";
    private static final String UPDATE_PASSWORD_CMS_PAGE = "updatePassword";
    private static final String BREADCRUMBS_ATTR = "breadcrumbs";
    private static final String FAILED_VERIFICATION_ERROR_MESSAGE = "Payment method failed verification.";

    // CMS Pages
    private static final String ADD_EDIT_PAYMENT_METHOD_CMS_PAGE = "add-edit-payment-method";
    private static final String UPDATE_PAYMENT_ADDRESS_CMS_PAGE = "update-payment-address";
    private static final String REDIRECT_TO_PASSWORD_UPDATE_PAGE = REDIRECT_PREFIX + "/my-account/update-password";

    @Resource(name = "userFacade")
    protected BrainTreeUserFacade userFacade;

    @Resource(name = "i18NFacade")
    private I18NFacade i18NFacade;

    @Resource(name = "acceleratorCheckoutFacade")
    private CheckoutFacade checkoutFacade;

    @Resource(name = "brainTreeCheckoutFacade")
    private DefaultBrainTreeCheckoutFacade brainTreeCheckoutFacade;

    @Resource(name = "brainTreeSetUpPasswordFacade")
    private DefaultBrainTreeSetUpPasswordFacade defaultBrainTreeSetUpPasswordFacade;

    @Resource(name = "accountBreadcrumbBuilder")
    private ResourceBreadcrumbBuilder accountBreadcrumbBuilder;

    @Resource(name = "addressDataUtil")
    private AddressDataUtil addressDataUtil;

    @Resource(name = "addressValidator")
    private AddressValidator addressValidator;

    @Resource(name = "brainTreePasswordValidator")
    private BrainTreePasswordValidator brainTreePasswordValidator;

    @Resource(name = "brainTreePaymentService")
    private BrainTreePaymentService brainTreePaymentService;

    @Resource(name = "brainTreePaymentFacadeImpl")
    private DefaultBrainTreePaymentFacade brainTreePaymentFacade;

    @ModelAttribute("countries")
    public Collection<CountryData> getCountries() {
        return checkoutFacade.getCountries(CountryType.SHIPPING);
    }

    @ModelAttribute("titles")
    public Collection<TitleData> getTitles() {
        return userFacade.getTitles();
    }

    @ModelAttribute("countryDataMap")
    public Map<String, CountryData> getCountryDataMap() {
        final Map<String, CountryData> countryDataMap = new HashMap<>();
        for (final CountryData countryData : getCountries()) {
            countryDataMap.put(countryData.getIsocode(), countryData);
        }
        return countryDataMap;
    }

    @RequestMapping(value = "/set-password", method = RequestMethod.POST)
    @RequireHardLogIn
    public String setPassword(final UpdatePasswordForm updatePasswordForm, final BindingResult bindingResult,
                              final Model model,
                              final RedirectAttributes redirectAttributes) throws CMSItemNotFoundException {
        brainTreePasswordValidator.validate(updatePasswordForm, bindingResult);
        if (!bindingResult.hasErrors()) {
            if (updatePasswordForm.getNewPassword().equals(updatePasswordForm.getCheckNewPassword())) {
                defaultBrainTreeSetUpPasswordFacade.setUpPassword(updatePasswordForm.getNewPassword());
            } else {
                bindingResult.rejectValue("checkNewPassword", "validation.checkPwd.equals", new Object[]{},
                        "validation.checkPwd.equals");
            }
        }
        if (bindingResult.hasErrors()) {
            GlobalMessages.addErrorMessage(model, FORM_GLOBAL_ERROR);
            final ContentPageModel updatePasswordPage = getContentPageForLabelOrId(UPDATE_PASSWORD_CMS_PAGE);
            storeCmsPageInModel(model, updatePasswordPage);
            setUpMetaDataForContentPage(model, updatePasswordPage);
            model.addAttribute(BREADCRUMBS_ATTR,
                    accountBreadcrumbBuilder.getBreadcrumbs("text.account.profile.updatePasswordForm"));
            return getViewForPage(model);
        } else {
            GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.CONF_MESSAGES_HOLDER,
                    "text.account.confirmation.password.set", null);
            return REDIRECT_TO_PASSWORD_UPDATE_PAGE;
        }
    }

    @RequestMapping(value = "/remove-payment-method-bt", method = RequestMethod.POST)
    @RequireHardLogIn
    public String removePaymentMethod(@RequestParam(value = "paymentInfoId") final String paymentInfoId,
                                      final RedirectAttributes redirectAttributes)
            throws CMSItemNotFoundException {
        try {
            userFacade.removeCCPaymentInfo(paymentInfoId);
        } catch (BraintreeDeletePaymentDetailsWithNotCapturedOrderException ex) {
            GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.ERROR_MESSAGES_HOLDER,
                    getLocalizedString("braintree.delete.payment.details.with.not.captured.order.error"));
            LOGGER.error(ex);
            return REDIRECT_TO_PAYMENT_INFO_PAGE;
        }
        GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.CONF_MESSAGES_HOLDER,
                getLocalizedString("text.account.profile.paymentCart.remove"));

        return REDIRECT_TO_PAYMENT_INFO_PAGE;
    }

    @RequestMapping(value = "/update-payment-address", method = RequestMethod.GET)
    @RequireHardLogIn
    public String updateAddress(@RequestParam(value = "paymentInfoId") final String paymentMethodId, final Model model)
            throws CMSItemNotFoundException {
        final AddressForm addressForm = new AddressForm();
        model.addAttribute(COUNTRY_DATA_ATTR, checkoutFacade.getCountries(CountryType.SHIPPING));
        model.addAttribute(TITLE_DATA_ATTR, userFacade.getTitles());
        model.addAttribute(ADDRESS_FORM_ATTR, addressForm);
        prefillAddressForm(paymentMethodId, addressForm, model);

        final ContentPageModel addEditAddressPage = getContentPageForLabelOrId(UPDATE_PAYMENT_ADDRESS_CMS_PAGE);
        storeCmsPageInModel(model, addEditAddressPage);
        setUpMetaDataForContentPage(model, addEditAddressPage);
        final List<Breadcrumb> breadcrumbs = accountBreadcrumbBuilder.getBreadcrumbs(null);

        breadcrumbs.add(new Breadcrumb("/my-account/payment-details",
                getMessageSource()
                        .getMessage("text.account.paymentDetails", null, getI18nService().getCurrentLocale()), null));
        breadcrumbs.add(new Breadcrumb("#",
                getMessageSource()
                        .getMessage("text.account.profile.paymentCart.update", null, getI18nService().getCurrentLocale()),
                null));

        model.addAttribute("paymentInfoId", paymentMethodId);
        model.addAttribute(BREADCRUMBS_ATTR, breadcrumbs);
        model.addAttribute(ThirdPartyConstants.SeoRobots.META_ROBOTS, ThirdPartyConstants.SeoRobots.NOINDEX_NOFOLLOW);
        return getViewForPage(model);
    }

    @RequestMapping(value = "/update-payment-address", method = RequestMethod.POST)
    @RequireHardLogIn
    public String updatePaymentAddress(@RequestParam(value = "paymentInfoId") final String paymentMethodId,
                                       final Model model, final AddressForm addressForm,
                                       final BindingResult bindingResult, final RedirectAttributes redirectAttributes) throws CMSItemNotFoundException {
        getAddressValidator().validate(addressForm, bindingResult);
        populateModelRegionAndCountry(model, addressForm.getCountryIso());
        CCPaymentInfoData paymentInfoData = userFacade.getCCPaymentInfoForCode(paymentMethodId);
        AddressData currentAddress = paymentInfoData.getBillingAddress();
        AddressData newAddress = addressDataUtil.convertToVisibleAddressData(addressForm);
        if (bindingResult.hasErrors()) {
            return handleAddressInputErrors(model, addressForm, paymentMethodId, "text.account.profile.address.update.error");
        } else if (areAddressesIdentical(currentAddress, newAddress)) {
            return handleAddressInputErrors(model, addressForm, paymentMethodId, "text.account.profile.address.update.sameAddressError");
        }
        try {
            UpdateCreditCardBillingAddressForm updateCreditCardBillingAddressForm = new UpdateCreditCardBillingAddressForm(newAddress, paymentMethodId);
            userFacade.editPaymentMethod(updateCreditCardBillingAddressForm);
        } catch (final AdapterException e) {
            LOGGER.error(
                    "Error occurred while payment address update: " + e.getMessage(), e);
            GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.ERROR_MESSAGES_HOLDER,
                    getLocalizedString("text.account.profile.paymentCart.addPaymentMethod.general.error"));
            return REDIRECT_TO_PAYMENT_INFO_PAGE;
        }
        GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.CONF_MESSAGES_HOLDER,
                getLocalizedString("text.account.profile.paymentCart.update.success"));
        return REDIRECT_TO_PAYMENT_INFO_PAGE;
    }

    @RequestMapping(value = "/receive-address", method = RequestMethod.GET)
    @RequireHardLogIn
    @ResponseBody
    public String receiveAddress(
            @RequestParam(value = "selectedAddressCode", required = false) String selectedAddressCode)
            throws CMSItemNotFoundException, IOException {
        if (StringUtils.isNotBlank(selectedAddressCode)) {
            final ObjectMapper mapper = new ObjectMapper();
            final List<AddressData> addressBook = userFacade.getAddressBook();
            final AddressData addressForPaymentInfo = getAddressForPaymentInfo(selectedAddressCode, addressBook);
            return mapper.writeValueAsString(addressForPaymentInfo);
        }
        return StringUtils.EMPTY;
    }

    @RequestMapping(value = "/add-payment-method", method = RequestMethod.GET)
    @RequireHardLogIn
    public String addPaymentMethod(final Model model, final String selectedAddressCode)
            throws CMSItemNotFoundException {

        final List<Breadcrumb> breadcrumbs = accountBreadcrumbBuilder.getBreadcrumbs(null);
        breadcrumbs.add(new Breadcrumb("/my-account/payment-details",
                getMessageSource().getMessage("text.account.paymentDetails", null, getI18nService().getCurrentLocale()),
                null));
        breadcrumbs.add(new Breadcrumb("#",
                getMessageSource()
                        .getMessage("text.account.profile.paymentMethod.add", null, getI18nService().getCurrentLocale()),
                null));

        storeCmsPageInModel(model, getContentPageForLabelOrId(ADD_EDIT_PAYMENT_METHOD_CMS_PAGE));
        setUpMetaDataForContentPage(model, getContentPageForLabelOrId(ADD_EDIT_PAYMENT_METHOD_CMS_PAGE));
        setupAdditionalFields(model);
        final List<AddressData> addressBook = userFacade.getAddressBook();
        model.addAttribute("deliveryAddresses", addressBook);
        model.addAttribute("selectedAddressCode", selectedAddressCode);
        model.addAttribute("addressForm", new AddressForm());
        model.addAttribute("breadcrumbs", breadcrumbs);
        model.addAttribute("metaRobots", "noindex,nofollow");
        return getViewForPage(model);
    }

    @RequestMapping(value = "/add-payment-method", method = RequestMethod.POST)
    @RequireHardLogIn
    public String addPaymentMethod(@RequestParam(value = "bt_payment_method_nonce") final String nonce,
                                   @RequestParam(value = "payment_type") final String paymentProvider,
                                   @RequestParam(value = "paypal_email") final String payPalEmail,
                                   @RequestParam(value = "card_type") final String cardType,
                                   @RequestParam(value = "card_details") final String cardDetails,
                                   @RequestParam(value = "card_expires_month") final String expiresMonth,
                                   @RequestParam(value = "card_expires_year") final String expiresYear,
                                   @RequestParam(value = "device_data") final String deviceData,
                                   @RequestParam(value = "liability_shifted") final String liabilityShifted,
                                   @RequestParam(value = "selectedAddressCode") final String selectedAddressCode,
                                   @RequestParam(value = "cardholder", required = false) final String cardholder,
                                   final RedirectAttributes redirectAttributes) throws CMSItemNotFoundException {
        if (StringUtils.isEmpty(nonce)) {
            LOGGER.error("Failed to create payment method. Error occurred while contacting external payment services.");
            GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.ERROR_MESSAGES_HOLDER,
                    getLocalizedString("text.account.profile.paymentCart.addPaymentMethod.noNonce"));
            return REDIRECT_TO_ADD_PAYMENT_INFO_PAGE;
        }

        final String storeInVault = getSiteConfigService().getProperty("braintree.store.in.vault");
        if (storeInVault != null && !Boolean.parseBoolean(storeInVault)) {
            LOGGER.error(
                    "Failed to create payment method. Store in vault is forbidden. For adding new payment methods enable store in vault.");
            GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.ERROR_MESSAGES_HOLDER,
                    getLocalizedString("text.account.profile.paymentCart.addPaymentMethod.forbidden"));
            return REDIRECT_TO_PAYMENT_INFO_PAGE;
        }

        final AddressData addressData = userFacade.getAddressForCode(selectedAddressCode);
        if (addressData == null) {
            LOGGER.error("Failed to create payment method. Error occurred while address selection");
            GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.ERROR_MESSAGES_HOLDER,
                    getLocalizedString("text.account.profile.paymentCart.addPaymentMethod.noAddress"));
            return REDIRECT_TO_ADD_PAYMENT_INFO_PAGE;
        }

        final BrainTreeSubscriptionInfoData paymentInfoData = GenericBuilder.of(BrainTreeSubscriptionInfoData::new)
                .with(BrainTreeSubscriptionInfoData::setCardType, cardType)
                .with(BrainTreeSubscriptionInfoData::setPaymentProvider, paymentProvider)
                .with(BrainTreeSubscriptionInfoData::setCardNumber,
                        BrainTreeUtils.formatCardNumber(cardDetails))
                .with(BrainTreeSubscriptionInfoData::setNonce, nonce)
                .with(BrainTreeSubscriptionInfoData::setExpirationMonth, expiresMonth)
                .with(BrainTreeSubscriptionInfoData::setExpirationYear, expiresYear)
                .with(BrainTreeSubscriptionInfoData::setCardholder, cardholder)
                .with(BrainTreeSubscriptionInfoData::setEmail, payPalEmail)
                .with(BrainTreeSubscriptionInfoData::setLiabilityShifted, Boolean.valueOf(liabilityShifted))
                .with(BrainTreeSubscriptionInfoData::setSavePaymentInfo, true)
                .with(BrainTreeSubscriptionInfoData::setDeviceData, deviceData)
                .with(BrainTreeSubscriptionInfoData::setImageSource,
                        brainTreePaymentService.getImageSourceForPaymentMethod(cardType,
                                BRAINTREE_ACCEPTED_CREDIT_CARD_PAYMENT_METHODS))
                .with(BrainTreeSubscriptionInfoData::setAddressData, addressData)
                .build();

        try {
            final CCPaymentInfoData paymentMethod = brainTreePaymentFacade.completeCreateSubscriptionForMyAccount(paymentInfoData);

            if (paymentMethod == null) {
                LOGGER.error("Failed to create payment method. Error occurred while payment method creation");
                GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.ERROR_MESSAGES_HOLDER,
                        getLocalizedString("text.account.profile.paymentCart.addPaymentMethod.error"));
                return REDIRECT_TO_ADD_PAYMENT_INFO_PAGE;
            }
        } catch (final AdapterException e) {
            String error;
            if (e.getMessage().equals(FAILED_VERIFICATION_ERROR_MESSAGE)) {
                error = "text.account.profile.paymentCart.addPaymentMethod.failed.verification.error";
            } else {
                error = "text.account.profile.paymentCart.addPaymentMethod.general.error";
            }
            LOGGER.error(
                    "Failed to create payment method. Error occurred while payment method creation: " + e.getMessage(), e);
            GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.ERROR_MESSAGES_HOLDER,
                    getLocalizedString(error));
            return REDIRECT_TO_ADD_PAYMENT_INFO_PAGE;
        }

        GlobalMessages.addFlashMessage(redirectAttributes, GlobalMessages.CONF_MESSAGES_HOLDER,
                getLocalizedString("text.account.profile.paymentCart.addPaymentMethod.success"));
        return REDIRECT_TO_PAYMENT_INFO_PAGE;
    }

    protected void setUpAddressFormAfterError(AddressForm addressForm, Model model, String paymentMethodId) {
        model.addAttribute(COUNTRY_DATA_ATTR, checkoutFacade.getCountries(CountryType.SHIPPING));
        model.addAttribute("metaRobots", "noindex,nofollow");
        model.addAttribute("paymentInfoId", paymentMethodId);
        model.addAttribute("titles", userFacade.getTitles());
        model.addAttribute("titleData", userFacade.getTitles());
        if (addressForm.getCountryIso() != null) {
            populateModelRegionAndCountry(model, addressForm.getCountryIso());
        }
    }

    protected void prefillAddressForm(String paymentMethodId, AddressForm addressForm, Model model) {
        CCPaymentInfoData paymentInfoData = userFacade.getCCPaymentInfoForCode(paymentMethodId);
        AddressData addressData = paymentInfoData.getBillingAddress();
        model.addAttribute(REGIONS_ATTR, getI18NFacade().getRegionsForCountryIso(addressData.getCountry().getIsocode()));
        model.addAttribute(COUNTRY_ATTR, addressData.getCountry().getIsocode());
        model.addAttribute(ADDRESS_DATA_ATTR, addressData);
        addressDataUtil.convert(addressData, addressForm);
    }

    private String handleAddressInputErrors(Model model, AddressForm addressForm, String paymentMethodId, String messageKey) throws CMSItemNotFoundException {
        GlobalMessages.addErrorMessage(model, messageKey);
        final ContentPageModel addEditAddressPage = getContentPageForLabelOrId(UPDATE_PAYMENT_ADDRESS_CMS_PAGE);
        storeCmsPageInModel(model, addEditAddressPage);
        setUpMetaDataForContentPage(model, addEditAddressPage);
        setUpAddressFormAfterError(addressForm, model, paymentMethodId);
        return getViewForPage(model);
    }

    protected void populateModelRegionAndCountry(final Model model, final String countryIsoCode) {
        model.addAttribute(REGIONS_ATTR, getI18NFacade().getRegionsForCountryIso(countryIsoCode));
        model.addAttribute(COUNTRY_ATTR, countryIsoCode);
    }

    private void setupAdditionalFields(final Model model) {
        String clientToken = StringUtils.EMPTY;

        try {
            clientToken = brainTreeCheckoutFacade.generateClientToken(Boolean.TRUE);
        } catch (final AdapterException exception) {
            LOGGER.error("[Brain Tree Controller] Error during token generation!", exception);
        }

        model.addAttribute(CLIENT_TOKEN, clientToken);
    }

    private boolean areAddressesIdentical(AddressData currentAddress, AddressData newAddress) {
        return currentAddress.getCountry().getIsocode().equals(newAddress.getCountry().getIsocode()) &&
                currentAddress.getFirstName().equals(newAddress.getFirstName()) &&
                currentAddress.getLastName().equals(newAddress.getLastName()) &&
                currentAddress.getLine1().equals(newAddress.getLine1()) &&
                StringUtils.equals(currentAddress.getLine2(), newAddress.getLine2()) &&
                currentAddress.getTown().equals(newAddress.getTown()) &&
                currentAddress.getPostalCode().equals(newAddress.getPostalCode()) &&
                areRegionsIdentical(currentAddress, newAddress);
    }

    private boolean areRegionsIdentical(AddressData currentAddress, AddressData newAddress) {
        boolean isCurrentRegionPresent = Objects.nonNull(currentAddress.getRegion());
        boolean isNewRegionPresent = Objects.nonNull(newAddress.getRegion());
        if (isCurrentRegionPresent && isNewRegionPresent) {
            return StringUtils.equals(currentAddress.getRegion().getIsocode(), newAddress.getRegion().getIsocode());
        } else if (!isCurrentRegionPresent && !isNewRegionPresent) {
            return Boolean.TRUE;
        }
        return Boolean.FALSE;
    }

    private AddressData getAddressForPaymentInfo(String selectedAddressCode, List<AddressData> addressBook) {
        if (StringUtils.isNotBlank(selectedAddressCode)) {
            for (final AddressData addressData : addressBook) {
                if (selectedAddressCode.equals(addressData.getId())) {
                    return addressData;
                }
            }
        }
        return null;
    }

    protected AddressValidator getAddressValidator() {
        return addressValidator;
    }

    protected I18NFacade getI18NFacade() {
        return i18NFacade;
    }
}
