package com.braintree.controllers;

import braintreehac.services.BrainTreeDropInConfigurationService;
import braintreehac.services.BrainTreeButtonConfigurationService;
import com.braintree.enums.BraintreePageType;
import com.braintree.facade.impl.DefaultBrainTreeCheckoutFacade;
import com.braintree.hybris.data.PayPalCheckoutData;
import com.braintree.hybris.data.ws.BraintreeAssetsWsDTO;
import com.braintree.hybris.data.ws.DropInButtonStyleWsDTO;
import com.braintree.hybris.data.ws.BraintreeButtonStyleWsDTO;
import de.hybris.platform.braintreehac.data.BraintreeButtonStyleData;
import com.braintree.hybris.data.ws.PayPalCheckoutWsDTO;
import com.braintree.validators.BraintreeButtonTypeValidator;
import de.hybris.platform.braintreehac.data.BraintreeButtonStyleData;
import de.hybris.platform.braintreehac.data.DropInButtonStyleData;
import de.hybris.platform.braintreehac.data.GooglePayButtonStyleData;
import de.hybris.platform.braintreehac.data.PayPalButtonStyleData;
import de.hybris.platform.braintreehac.data.ApplePayButtonStyleData;
import de.hybris.platform.webservicescommons.swagger.ApiBaseSiteIdParam;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import javax.annotation.Resource;

import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import static braintreehac.services.BrainTreeDropInConfigurationService.CART;

@Controller
@RequestMapping(value = "/{baseSiteId}")
@Api(tags = "Braintree Payment")
public class BraintreePaymentController extends BraintreeBaseController {

    @Resource(name = "braintreeButtonTypeValidator")
    private BraintreeButtonTypeValidator buttonTypeValidator;

    @Resource(name = "brainTreeCheckoutFacade")
    private DefaultBrainTreeCheckoutFacade brainTreeCheckoutFacade;

    @Resource(name = "brainTreeButtonConfigurationService")
    private BrainTreeButtonConfigurationService payPalButtonConfigurationService;

    @Resource(name = "brainTreeDropInConfigurationService")
    private BrainTreeDropInConfigurationService dropInConfigurationService;

    @GetMapping("/users/{userId}/braintree/payment")
    @ResponseStatus(value = HttpStatus.OK)
    @ResponseBody
    @ApiBaseSiteIdParam
    @ApiOperation(value = " Get the Braintree data for client-side payment initialization",
        notes = "Get a data for all payment methods")
    public PayPalCheckoutWsDTO getPaymentData(
        @ApiParam(value = "Payment method identifier.", allowableValues = "paypal, googlepay, venmo, applepay," +
            " localpayments, creditcard, configData, full", required = true) @RequestParam final String paymentMethod,
        @ApiParam(value = "Page type", defaultValue = "MARK")
        @RequestParam(defaultValue = "MARK", required = false) final String pageType) {
        final BraintreePageType braintreePageType = BraintreePageType.valueOf(pageType);
        final PayPalCheckoutData payPalCheckoutData = brainTreeCheckoutFacade.getPayPalCheckoutData(braintreePageType);
        return getDataMapper().map(payPalCheckoutData, PayPalCheckoutWsDTO.class, paymentMethod);
    }

    @GetMapping("/users/{userId}/braintree/payment/button")
    @ResponseStatus(value = HttpStatus.OK)
    @ResponseBody
    @ApiBaseSiteIdParam
    @ApiOperation(value = " Get the PayPal Smart Button style data.")
    public BraintreeButtonStyleWsDTO getButtonStyle(
        @ApiParam(value = "Button type identifier.", allowableValues = "minicart, cart, mark",
            required = true) @RequestParam final String buttonType) {
        validate(buttonType, buttonType, buttonTypeValidator);
        final PayPalButtonStyleData payPalButtonStyleData = payPalButtonConfigurationService.
            getPayPalButtonStyleByPageType(BraintreePageType.valueOf(buttonType.toUpperCase()));
        final GooglePayButtonStyleData googlePayButtonStyleData = payPalButtonConfigurationService.
                getGooglePayButtonStyleByPageType(BraintreePageType.valueOf(buttonType.toUpperCase()));
        final ApplePayButtonStyleData applePayButtonStyleData = payPalButtonConfigurationService.
                getApplePayButtonStyleByPageType(BraintreePageType.valueOf(buttonType.toUpperCase()));

        BraintreeButtonStyleData braintreeButtonStyle = new BraintreeButtonStyleData();
        braintreeButtonStyle.setPayPalButtonStyle(payPalButtonStyleData);
        braintreeButtonStyle.setGooglePayButtonStyle(googlePayButtonStyleData);
        braintreeButtonStyle.setApplePayButtonStyle(applePayButtonStyleData);

        return getDataMapper().map(braintreeButtonStyle, BraintreeButtonStyleWsDTO.class);
    }

    @GetMapping("/braintree/payment/assets")
    @ResponseStatus(value = HttpStatus.OK)
    @ResponseBody
    @ApiBaseSiteIdParam
    @ApiOperation(value = " Get assets for checkout.")
    public BraintreeAssetsWsDTO getAssets() {
        return getDataMapper().map(brainTreeCheckoutFacade.getBraintreeAssets(), BraintreeAssetsWsDTO.class);
    }

    @GetMapping("/users/{userId}/braintree/payment/dropInButton")
    @ResponseStatus(value = HttpStatus.OK)
    @ResponseBody
    @ApiBaseSiteIdParam
    @ApiOperation(value = " Get the Drop-in UI button style data.")
    public DropInButtonStyleWsDTO getDropInButtonStyle(
            @ApiParam(value = "Button type identifier.", allowableValues = "Cart, Mini cart, Billing",
                    required = true) @RequestParam(required = false, defaultValue = CART) final String buttonType) {
        final DropInButtonStyleData dropInButtonStyleData = dropInConfigurationService.getDropInButtonStyleData(buttonType);
        final DropInButtonStyleWsDTO dropInButtonStyleWsDTO = getDataMapper().map(dropInButtonStyleData, DropInButtonStyleWsDTO.class);
        return dropInButtonStyleWsDTO;
    }

}
