import { Observable } from 'rxjs';
import {Component, OnInit, ChangeDetectionStrategy, Input} from '@angular/core';
import * as braintree from 'braintree-web';
import {
  BraintreeUtilsService,
  BraintreePaymentDetailsService,
  CheckoutData,
  PayPalCreditMessageComponent,
  PageType
} from 'braintree-spartacus-core';
import * as CONST from 'braintree-spartacus-core';
import { map } from 'rxjs/operators';
import {ActiveCartService, CmsService, OCC_USER_ID_ANONYMOUS, UserIdService} from '@spartacus/core';
import { CmsComponentData } from '@spartacus/storefront';


@Component({
  selector: 'bt-paypal-credit-message',
  templateUrl: './paypal-credit-message.component.html',
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class PaypalCreditMessageComponent implements OnInit {

  @Input()
  private pageType: PageType;

  loadedCheckoutData$: Observable<CheckoutData>;
  amount$: Observable<number>;
  componentData$: Observable<PayPalCreditMessageComponent>;

  constructor(
    protected component: CmsComponentData<PayPalCreditMessageComponent>,
    protected braintreePaymentDetailsService: BraintreePaymentDetailsService,
    protected braintreeUtils: BraintreeUtilsService,
    protected activeCartService: ActiveCartService,
    protected cmsService: CmsService,
    protected userIdService: UserIdService
  ) {}

  ngOnInit(): void {
    if (this.pageType === PageType.CART) {
      this.componentData$ = this.cmsService.getComponentData('PayPalCartPageCreditMessageComponentSPA');
    }else{
      this.componentData$ = this.component.data$;
    }
    this.loadedCheckoutData$ = this.braintreePaymentDetailsService
      .loadPaymentDetails('full')
      .pipe(
        map((checkoutData) => {
          this.initializePayPalCreditMessage(checkoutData);
          return checkoutData;
        })
      );

    this.amount$ = this.activeCartService
    .getActive()
    .pipe(
      map((cart) => {
        if (cart.totalPrice !== undefined){
          return cart.totalPrice.value;
        }else{
          return 0.0;
        }
      })
    );
  }

  initializePayPalCreditMessage(checkoutData: CheckoutData): void {
    this.braintreeUtils.createClientInstance(
      checkoutData.configurationData,
      (client: any) => {
        braintree.paypalCheckout.create(
          {
            client,
          },
          (paypalCheckoutErr: any, paypalCheckoutInstance: any) => {
            if (typeof (window as any).paypalSdk === 'undefined' && !this.sdkLoadsOnCartPage(checkoutData)) {
              const commit =
                checkoutData.configurationData.intent === CONST.INTENT_SALE && checkoutData.configurationData.userAction === 'true';

              this.braintreeUtils.loadPayPalSDK(checkoutData, paypalCheckoutInstance,
                  checkoutData.configurationData.intent, commit, this.braintreeUtils.getPaypalFlow(checkoutData), this.pageType,
                  () => {console.log('PayPal Sdk was loaded');
                });
            }
          }
        );
      }
    );
  }

  sdkLoadsOnCartPage(checkoutData: CheckoutData): boolean{
    return this.pageType === PageType.CART && checkoutData.payPalPaymentMethod.payPalExpressEnabled
        && !checkoutData.dropIn.dropInEnabled && !(this.isAnonymousUser() && checkoutData.configurationData.intent === CONST.INTENT_ORDER);
  }

  isAnonymousUser(): boolean{
    let userId;
    this.userIdService
        .getUserId()
        .subscribe((occUserId) => (userId = occUserId))
        .unsubscribe();
    return OCC_USER_ID_ANONYMOUS === userId;
  }
}
