import {
  AfterViewInit,
  ChangeDetectorRef,
  Component,
  ComponentRef,
  ElementRef,
  EventEmitter,
  Input,
  OnDestroy,
  OnInit,
  Output,
  ViewChild,
  ViewContainerRef
} from '@angular/core';
import {combineLatest, Observable, Subject, Subscription} from 'rxjs';
import {map} from 'rxjs/operators';
import {ActiveCartService, OccConfig, RoutingService} from '@spartacus/core';
import {CheckoutFacade} from '@spartacus/checkout/root';
import {
  ApplePayCheckoutService,
  BraintreeAssets,
  BraintreeLocalPaymentMethodsService,
  BraintreePaymentDetailsService,
  BraintreePaymentMethodsUtilsService,
  BraintreeSrcCheckoutService,
  BraintreeVenmoService,
  CheckoutData,
  GooglePayCheckoutService,
  INTENT_ORDER,
  PageType,
  BraintreeButtonStyle,
  PayPalButtonStyle,
  PaypalCheckoutService, GooglePayButtonStyle, ApplePayButtonStyle
} from 'braintree-spartacus-core';
import {LAUNCH_CALLER, LaunchDialogService} from '@spartacus/storefront';

@Component({
  selector: 'bt-payment-method',
  templateUrl: './braintree-payment-method.component.html'
})
export class BraintreePaymentMethodComponent implements OnInit, OnDestroy, AfterViewInit {

  @ViewChild('paypalMarkButton') paypalButton: ElementRef;
  @ViewChild('venmoButton') venmoButton: ElementRef;
  @ViewChild('applePayButton') applePayButton: ElementRef;

  @ViewChild('paymentMethodSelection') set content(content: ElementRef) {
    if (content) {
        this.paymentMethodExist = true;
        this.changeDetectorRef.detectChanges();
    }
  }

  paymentMethodExist: boolean;
  subscription: Subscription = new Subscription();
  placedOrder: void | Observable<ComponentRef<any>>;
  loadedCheckoutData$: Observable<CheckoutData>;
  braintreeAssets$: Observable<BraintreeAssets>;
  braintreeButtonStyle: BraintreeButtonStyle;
  isApplePayAvailable = this.applePayService.isAvailableApplePay();
  isCreditCardAvailable: boolean;
  isApplePayInitialized = false;
  isVenmoInitialized = false;
  isSrcInitialized = false;
  isPaymentMethodSaved = false;
  radioButtonType: string;
  contextPath$: string;
  showSavePaymentInfo: boolean;
  isVaultingFlow: boolean;
  braintreeCheckoutData: CheckoutData;
  applePayStyle: string;
  applePayType: string;
  applePayRadius: string;

  @Input()
  pageType = PageType.BILLING;

  @Input()
  paymentMethodsCount: number;

  @Output()
  backToPaymentMethods = new EventEmitter<any>();

  constructor(
    protected paypalService: PaypalCheckoutService,
    protected venmoService: BraintreeVenmoService,
    protected googlePayService: GooglePayCheckoutService,
    protected applePayService: ApplePayCheckoutService,
    protected srcCheckoutService: BraintreeSrcCheckoutService,
    protected paymentMethodsUtilsService: BraintreePaymentMethodsUtilsService,
    protected braintreePaymentDetailsService: BraintreePaymentDetailsService,
    protected braintreeLocalPaymentMethodsService: BraintreeLocalPaymentMethodsService,
    protected config: OccConfig,
    protected checkoutService: CheckoutFacade,
    protected launchDialogService: LaunchDialogService,
    protected vcr: ViewContainerRef,
    protected routingService: RoutingService,
    protected activeCartService: ActiveCartService,
    protected changeDetectorRef: ChangeDetectorRef,
  ) { }

  private _viewInitWaiter$ = new Subject();

  ngOnInit(): void {
    this.contextPath$ = this.config.backend.occ.baseUrl;
    this.loadedCheckoutData$ = combineLatest(
      [this.braintreePaymentDetailsService.getBraintreeButtonStyles('mark'),
        this.braintreePaymentDetailsService.loadPaymentDetailsForBillingPage('full', this.pageType)])
      .pipe(
        map(
          ([buttonStyle, checkoutData]) => {
            this.braintreeButtonStyle = buttonStyle;
            this.initializeApplePayStyle(buttonStyle);
            this.configurePaymentPageProperties(checkoutData);
            this.braintreeCheckoutData = checkoutData;
            this.triggerRadioButton(checkoutData, buttonStyle);
            return checkoutData;
          }));
    this.paymentMethodsUtilsService.setBackToPaymentMethods(this.backToPaymentMethods);
    this.braintreeAssets$ = this.braintreePaymentDetailsService.getBraintreeAssets();
  }

  ngAfterViewInit(): void{
    this._viewInitWaiter$.complete();
  }

  private _executeAfterViewInit(func: () => any): any {
    this._viewInitWaiter$.subscribe(null, null, () => {
      return func();
    });
  }

  onChangePayPal(checkoutData: CheckoutData, buttonStyle: PayPalButtonStyle): void{
    this.radioButtonType = 'paypal';
    this.showSavePaymentInfo = this.isVaultingFlow && !(checkoutData.configurationData.intent === INTENT_ORDER);
    if (checkoutData.payPalPaymentMethod.payPalStandardEnabled && !this.paypalButton.nativeElement.hasChildNodes()){
      this.paypalService.initializePayPalCheckout(this.paypalButton.nativeElement, checkoutData, buttonStyle, this.pageType);
    }
  }

  onChangeGooglePay(checkoutData: CheckoutData, buttonStyle: GooglePayButtonStyle): void{
    this.radioButtonType = 'googlePay';
    this.showSavePaymentInfo = this.isVaultingFlow;
    if (checkoutData.googlePayPaymentMethod.googlePayEnabled){
      this.googlePayService.loadGooglePay(() => {
        this.googlePayService.initialiseGooglePay(checkoutData, buttonStyle, this.pageType);
      });
    }
  }

  onChangeVenmo(checkoutData: CheckoutData): void{
    this.radioButtonType = 'venmo';
    this.showSavePaymentInfo = this.isVaultingFlow;
    if (checkoutData.venmoPaymentMethod.venmoEnabled && !this.isVenmoInitialized) {
      this.venmoService.initializeVenmo(checkoutData, this.venmoButton.nativeElement, this.pageType);
    }
  }

  onChangeApplePay(checkoutData: CheckoutData): void{
    this.radioButtonType = 'applePay';
    this.showSavePaymentInfo = false;
    if (checkoutData.applePayPaymentMethod.applePayEnabled === 'true' && this.isApplePayAvailable && !this.isApplePayInitialized) {
      this.applePayService.initalizeApplePay(checkoutData, PageType.BILLING, this.applePayButton.nativeElement);
    }
  }

  onChangeCreditCard(checkoutData: CheckoutData): void{
    this.radioButtonType = 'hostedFields';
    this.showSavePaymentInfo = this.isVaultingFlow;
  }

  onChangeUsBankAccount(checkoutData: CheckoutData): void{
    this.radioButtonType = 'usBankAccount';
    this.showSavePaymentInfo = this.isVaultingFlow;
  }

  onChangeSRC(checkoutData: CheckoutData): void{
    this.radioButtonType = 'src';
    this.showSavePaymentInfo = this.isVaultingFlow;
    if (checkoutData.srcPaymentMethod.srcEnabled && !this.isSrcInitialized) {
      this.srcCheckoutService.initializeSrc(checkoutData, this.pageType);
    }
  }

  triggerRadioButton(checkoutData: CheckoutData, buttonStyle: BraintreeButtonStyle): void {
    this._executeAfterViewInit(() =>
    {
      if (checkoutData.payPalPaymentMethod.payPalStandardEnabled && !(this.isMyAccountPage() && checkoutData.configurationData.intent === INTENT_ORDER)) {
        this.onChangePayPal(checkoutData, buttonStyle.payPalButtonStyle);
      } else if (checkoutData.applePayPaymentMethod.applePayEnabled === 'true' && this.isApplePayAvailable && this.isBillingPage()) {
        this.onChangeApplePay(checkoutData);
      } else if (checkoutData.googlePayPaymentMethod.googlePayEnabled) {
        this.onChangeGooglePay(checkoutData, buttonStyle.googlePayButtonStyle);
      } else if (this.isCreditCardAvailable && checkoutData.creditCardPaymentMethod.paymentsImagesUrl) {
        this.onChangeCreditCard(checkoutData);
      } else if (checkoutData.venmoPaymentMethod.venmoEnabled) {
        this.onChangeVenmo(checkoutData);
      } else if (checkoutData.srcPaymentMethod.srcEnabled) {
        this.onChangeSRC(checkoutData);
      } else if (
          checkoutData.localPaymentMethod?.localPaymentMethodsEnabled && checkoutData.configurationData.intent === 'sale'
          && checkoutData.localPaymentMethod?.localPaymentMethods.length > 0 && this.isBillingPage()) {
        this.onChangeLPM();
      }
    });

  }

  onChangeLPM(): void{
        this.radioButtonType = 'lpm';
        this.showSavePaymentInfo = false;
        this.subscription.add(
      combineLatest([
        this.checkoutService.getPlaceOrderLoading(),
        this.checkoutService.getPlaceOrderSuccess(),
        this.checkoutService.getPlaceOrderError(),
      ]).subscribe(([orderLoading, orderSuccess, orderError]) => {
        if (orderLoading) {
          this.placedOrder = this.launchDialogService.launch(
            LAUNCH_CALLER.PLACE_ORDER_SPINNER,
            this.vcr
          );
        }

        if (orderError) {
          if (this.placedOrder) {
            this.placedOrder
              .subscribe((component) => {
                this.launchDialogService.clear(
                  LAUNCH_CALLER.PLACE_ORDER_SPINNER
                );
                component.destroy();
              })
              .unsubscribe();
            this.checkoutService.clearPlaceOrderState();
          }
        }

        if (orderSuccess) {
          this.routingService.go({ cxRoute: 'orderConfirmation' });
        }
      })
    );
  }

  onLocalPaymentSelect(checkoutData: CheckoutData, lpmType: string): void{
    if (checkoutData.localPaymentMethod?.localPaymentMethodsEnabled && checkoutData.configurationData.intent === 'sale'){
      this.braintreeLocalPaymentMethodsService.initializeLocalPaymentMethod(checkoutData, lpmType);
    }
  }

  configurePaymentPageProperties(checkoutData: CheckoutData): void{
    this.isVaultingFlow = checkoutData.configurationData.storeInVault.toLowerCase() === 'true' &&
      !this.activeCartService.isGuestCart();
    this.isCreditCardAvailable = checkoutData.creditCardPaymentMethod.hostedFieldsEnable;
  }

  back(): void{
    this.backToPaymentMethods.emit();
  }

  toggleSavePaymentInfo(event): void {
    this.paymentMethodsUtilsService.setSavePaymentInfo(event.target.checked);
    this.isPaymentMethodSaved = event.target.checked;
  }

  isBillingPage(): boolean{
    return this.pageType === PageType.BILLING;
  }

  isMyAccountPage(): boolean{
    return this.pageType === PageType.MY_ACCOUNT;
  }

  isBackButtonAvailable(): boolean{
    return this.isBillingPage() ? this.paymentMethodsCount > 0 : true;
  }

  getApplePayBorderRadius(shape: string): string {
    if (shape === 'squareCorners') {
      return '0px';
    } else if (shape === 'pill-shaped') {
      return '15px';
    }
  }

  initializeApplePayStyle(buttonStyle: BraintreeButtonStyle) {
    this.applePayStyle = buttonStyle.applePayButtonStyle.color;
    this.applePayType = buttonStyle.applePayButtonStyle.type;
    this.applePayRadius = this.getApplePayBorderRadius(buttonStyle.applePayButtonStyle.shape);
  }

  ngOnDestroy(): void {
    this.subscription.unsubscribe();
    this.launchDialogService.clear(LAUNCH_CALLER.PLACE_ORDER_SPINNER);
    this.checkoutService.clearPlaceOrderState();
    this.paymentMethodsUtilsService.setSavePaymentInfo(false);
  }

}
