import {BraintreePaymentDetails, PageType} from 'braintree-spartacus-core';
import {
    BraintreePaymentMethodsUtilsService
} from 'braintree-spartacus-core';
import {Component, Input} from '@angular/core';
import {Card, OrderOverviewComponent} from '@spartacus/storefront';
import {Address, TranslationService} from '@spartacus/core';
import {combineLatest, Observable} from 'rxjs';
import {filter, map} from 'rxjs/operators';
import {forEach} from "@angular-devkit/schematics";


@Component({
    selector: 'bt-order-overview',
    templateUrl: './braintree-order-overview.component.html'
})
export class BraintreeOrderOverviewComponent extends OrderOverviewComponent {

    @Input()
    pageType: string;

    @Input('order')
    set setOrder(order: any) {
        super.order = order;
    }

    constructor(
        protected translation: TranslationService,
        protected braintreePaymentMethodsUtilsService: BraintreePaymentMethodsUtilsService
    ) {
        super(translation);
    }

    getVenmoBillingAddressCardContent(): Observable<Card> {
        return combineLatest([
            this.translation.translate('paymentDetails.isNotRequiredForVenmo'),
            this.translation.translate('paymentForm.billingAddress'),
        ]).pipe(
            map(([textVenmo, textTitle]) => {
                return {
                    title: textTitle,
                    textBold: textVenmo,
                } as Card;
            })
        );
    }

    getPayPalBillingAddressCardContent(deliveryAddress: Address): Observable<Card> {
        return this.translation.translate('paymentForm.billingAddress').pipe(
            filter(() => Boolean(deliveryAddress)),
            map((textTitle) => {
                const formattedAddress = this.normalizePayPalFormattedAddress(
                    deliveryAddress.formattedAddress ?? ''
                );

                return {
                    title: textTitle,
                    textBold: `${deliveryAddress.firstName} ${deliveryAddress.lastName}`,
                    text: [formattedAddress, deliveryAddress.country?.name],
                } as Card;
            })
        );
    }

    private normalizePayPalFormattedAddress(formattedAddress: string): string {
        const addresses = formattedAddress
            .split(',')
            .map((address) => address.trim());

        const newFormattedAddress = addresses.filter(Boolean).join(', ');

        return newFormattedAddress;
    }

    getPaymentInfoCardContent(paymentDetails: BraintreePaymentDetails): Observable<Card> {
        return combineLatest([
            this.translation.translate('paymentForm.payment'),
            this.translation.translate('paymentCard.expires', {
                month: paymentDetails.expiryMonth,
                year: paymentDetails.expiryYear,
            }),
        ]).pipe(
            map(([textTitle, textExpires]) => {
                return {
                    title: textTitle,
                    textBold: paymentDetails.subscriptionId,
                    img: paymentDetails.accountHolderName,
                    text: this.pageType != null && this.pageType === PageType.CONFIRMATION ?
                        this.braintreePaymentMethodsUtilsService.configurePaymentMethodCardText(paymentDetails, textExpires, PageType.CONFIRMATION) :
                        this.braintreePaymentMethodsUtilsService.configurePaymentMethodCardText(paymentDetails, textExpires, PageType.OVERVIEW)
                };
            })
        );
    }

}
