import {BraintreeUserService, SetUpPasswordFormData} from 'braintree-spartacus-core';
import {Component} from '@angular/core';
import {UpdatePasswordComponent, UpdatePasswordComponentService} from '@spartacus/user/profile/components';
import {BehaviorSubject, Observable} from 'rxjs';
import {tap} from 'rxjs/operators';
import {FormControl, FormGroup, Validators} from '@angular/forms';
import {CustomFormValidators} from '@spartacus/storefront';
import {GlobalMessageService, GlobalMessageType, RoutingService} from "@spartacus/core";

@Component({
    selector: 'braintree-set-up-password-for-single-auth',
    templateUrl: './braintree-set-up-password-for-single-auth.component.html'
})
export class BraintreeSetUpPasswordForSingleAuthComponent extends UpdatePasswordComponent {

    setUpPasswordFormData$: Observable<SetUpPasswordFormData>;

    constructor(
        protected service: UpdatePasswordComponentService,
        protected braintreeUserService: BraintreeUserService,
        protected routingService: RoutingService,
        protected globalMessageService: GlobalMessageService
    ) {
        super(service);
        this.setUpPasswordFormData$ = this.braintreeUserService.getSetUpPasswordFormData();
    }

    protected busy$ = new BehaviorSubject(false);

    isUpdatingSetPasswordForm$ = this.busy$.pipe(
        tap((state) => (state === true ? this.setPasswordForm.disable() : this.setPasswordForm.enable()))
    );

    setPasswordForm: FormGroup = new FormGroup(
        {
            newPassword: new FormControl('', [
                Validators.required,
                CustomFormValidators.passwordValidator,
            ]),
            newPasswordConfirm: new FormControl('', Validators.required),
        },
        {
            validators: CustomFormValidators.passwordsMustMatch(
                'newPassword',
                'newPasswordConfirm'
            ),
        }
    );

    setUpPassword() {
        if (!this.setPasswordForm.valid) {
            this.setPasswordForm.markAllAsTouched();
            return;
        }
        this.busy$.next(true);
        const newPassword = this.setPasswordForm.get('newPassword')?.value;
        this.braintreeUserService.setUpPasswordForPayPalCustomer(newPassword).subscribe({
            next: () => this.passwordAdded(),
            error: (error: Error) => this.handleError(error),
        });
    }

    private passwordAdded(): void {
        this.addInfoMessage('myAccount.passwordForSingleAuthenticationAdded');
        this.busy$.next(false);
        this.setPasswordForm.reset();
        this.routingService.go({cxRoute: 'home'});
    }

    private handleError(_error: Error): void {
        this.busy$.next(false);
        this.setPasswordForm.reset();
        this.addErrorMessage('connectRegister.error.somethingWentWrong')
    }

    private addInfoMessage(key: string) {
        this.globalMessageService.add(
            {key: key},
            GlobalMessageType.MSG_TYPE_INFO,
            10000
        );
    }

    private addErrorMessage(key: string) {
        this.globalMessageService.add(
            {key: key},
            GlobalMessageType.MSG_TYPE_WARNING,
            10000
        );
    }
}