import {Component, EventEmitter, Output, OnInit} from '@angular/core';
import {
    Address, StateWithProcess, StateWithUser,
    TranslationService,
    UserAddressService,
} from '@spartacus/core';
import {ActivatedRoute} from '@angular/router';
import {AddressBookComponentService, Card} from '@spartacus/storefront';
import {Observable, of, combineLatest} from 'rxjs';
import { map } from 'rxjs/operators';
import {Store} from '@ngrx/store';

export interface CardWithAddress {
    card: Card;
    address: Address;
}

@Component({
    selector: 'bt-select-address-myaccount',
    templateUrl: './braintree-select-address-myaccount.component.html'
})
export class BraintreeSelectAddressMyaccountComponent implements OnInit {

    selectedBilling$: Observable<Address>;
    addresses$: Observable<Address[]>;
    addresses: Address[];
    addressesStateLoading$: Observable<boolean>;
    selectedAddress: Address;
    doneAutoSelect: boolean;

    @Output()
    backToAddPaymentMethods = new EventEmitter<any>();

    @Output()
    selectedAddressForPayment: EventEmitter<Address> = new EventEmitter();

    constructor(
        protected store: Store<StateWithUser | StateWithProcess<void>>,
        protected userAddressService: UserAddressService,
        protected activatedRoute: ActivatedRoute,
        protected translation: TranslationService,
        public service: AddressBookComponentService,
    ) {}

    ngOnInit(): void {
        this.addresses$ = this.service.getAddresses();
        this.addressesStateLoading$ = this.service.getAddressesStateLoading();
        this.service.loadAddresses();
        this.addresses$.subscribe(addresses => {
                this.addresses = addresses as Address[]
            });
    }

    get cards$(): Observable<CardWithAddress[]> {
        return combineLatest([
            this.addresses$,
            this.translation.translate('checkoutAddress.defaultShippingAddress'),
            this.translation.translate('addressCard.selected'),
        ]).pipe(
            map(([addresses, textDefault, textSelected]) =>
                (<any>addresses).map((address) => ({
                    address,
                    card: this.getCardContent(
                        address,
                        textDefault,
                        textSelected
                    ),
                }))
            )
        );
    }

    selectedBillingAddress(address: Address): void {
        this.selectedBilling$ = of(address);
        this.selectedAddressForPayment.emit(address);
    }

    selectAddress(address: Address): void {
        this.selectedAddress = address;
    }

    getCardContent(
        address: Address,
        textDefaultShippingAddress: string,
        textSelected: string
    ): Card {
        let region = '';
        if (address.region && address.region.isocode) {
            region = address.region.isocode + ', ';
        }
        this.selectDefaultAddress(this.addresses);
        return {
            title: address.defaultAddress ? textDefaultShippingAddress : '',
            textBold: address.firstName + ' ' + address.lastName,
            text: [
                address.line1,
                address.line2,
                address.town + ', ' + region + address.country.isocode,
                address.postalCode,
                address.phone,
            ],
            header: this.selectedAddress && this.selectedAddress.id === address.id ? textSelected : '',
        };
    }

    selectDefaultAddress(addresses: Address[]) {
        if (!this.doneAutoSelect && addresses && addresses.length) {
            let selected: Address;
            if (addresses.length === 1) {
                this.selectAddress(addresses[0]);
            } else {
                selected = addresses.find((address) => address.defaultAddress);
                if (selected) {
                    this.selectAddress(selected);
                }
            }
            this.doneAutoSelect = true;
        }
    }

    back() {
        this.backToAddPaymentMethods.emit();
    }
}
