import {Component, OnDestroy, OnInit} from '@angular/core';
import {FormBuilder, FormGroup, Validators} from "@angular/forms";
import {
  Address, AddressValidation,
  GlobalMessageService, GlobalMessageType,
  Occ,
  Region,
  UserAddressService,
} from "@spartacus/core";
import {ActivatedRoute, Router} from '@angular/router';
import {ModalRef, ModalService, SuggestedAddressDialogComponent} from "@spartacus/storefront";
import {switchMap, take, tap} from "rxjs/operators";
import Country = Occ.Country;
import {BehaviorSubject, Observable, Subscription} from "rxjs";
import {BraintreeCreditCardService, BraintreePaymentDetailsService} from 'braintree-spartacus-core';
import {HttpErrorResponse} from "@angular/common/http";


@Component({
  selector: 'braintree-edit-billing-address-for-payment-method-myaccount',
  templateUrl: './braintree-edit-billing-address-for-payment-method-myaccount.component.html',
})
export class BraintreeEditBillingAddressForPaymentMethodMyaccountComponent implements OnInit, OnDestroy {

  paymentMethodId: string;
  countries$: Observable<Country[]>;
  regions$: Observable<Region[]>;
  selectedCountry$: BehaviorSubject<string> = new BehaviorSubject<string>('');
  regionsSub: Subscription;
  suggestedAddressModalRef: ModalRef;
  currentAddress: Address;
  regionEnable: boolean = false;

  addressForm: FormGroup = this.fb.group({
    country: this.fb.group({
      isocode: [null, Validators.required],
    }),
    firstName: ['', Validators.required],
    lastName: ['', Validators.required],
    line1: ['', Validators.required],
    line2: [''],
    town: ['', Validators.required],
    region: this.fb.group({
      isocode: [null, Validators.required],
    }),
    postalCode: ['', Validators.required],
  });

  constructor(
      protected fb: FormBuilder,
      protected userAddressService: UserAddressService,
      protected globalMessageService: GlobalMessageService,
      protected modalService: ModalService,
      protected route: ActivatedRoute,
      protected router: Router,
      protected creditCardService: BraintreeCreditCardService,
      protected paymentMethodService: BraintreePaymentDetailsService
  ) {
  }

  ngOnInit() {
    // Fetching countries
    this.countries$ = this.userAddressService.getDeliveryCountries().pipe(
        tap((countries: Country[]) => {
          if (Object.keys(countries).length === 0) {
            this.userAddressService.loadDeliveryCountries();
          }
        })
    );

    this.route.queryParams.subscribe(params => {
      this.paymentMethodId = params['paymentMethodId']
    });

    this.paymentMethodService.getBillingAddressByPaymentMethodId(this.paymentMethodId).subscribe(address=>{
      this.currentAddress=address;
      this.countrySelected(address.country);
      this.addressForm.patchValue({firstName: this.currentAddress.firstName});
      this.addressForm.patchValue({lastName: this.currentAddress.lastName});
      this.addressForm.patchValue({line1: this.currentAddress.line1});
      this.addressForm.patchValue({line2: this.currentAddress.line2});
      this.addressForm.patchValue({town: this.currentAddress.town});
      if(this.currentAddress.region!== undefined){
        this.addressForm.get('region')?.get('isocode')?.setValue(this.currentAddress.region.isocode);
      }
      this.addressForm.patchValue({postalCode: this.currentAddress.postalCode});
      this.addressForm.markAllAsTouched();
    });

    // Fetching regions
    this.regions$ = this.selectedCountry$.pipe(
        switchMap((country) => this.userAddressService.getRegions(country)),
        tap((regions: Region[]) => {
          const regionControl = this.addressForm.get('region.isocode');
          if (regions && regions.length > 0) {
            regionControl.enable();
            this.regionEnable=true;
          } else {
            regionControl.disable();
            this.regionEnable=false;
          }
        })
    );

  }


  updateBillingAddress(address: Address): void {
    if(!this.isSameAddressAlreadySaved(address)){
      this.creditCardService.updateCreditCardBillingAddress(address, this.paymentMethodId).subscribe({
        next: ()=>this.billingAddressUpdated(),
        error: (error: HttpErrorResponse) => this.handleError(error)
      });
    }
  }

  private billingAddressUpdated():void {
    this.globalMessageService.add(
        {key: 'paymentDetails.paymentMethodUpdated'},
        GlobalMessageType.MSG_TYPE_INFO,
        10000
    );
    this.router.navigate(['/my-account/payment-details']);
  }

  private handleError(_error: HttpErrorResponse): void {
    this.globalMessageService.add(
        {key: 'error.client_request_error'},
        GlobalMessageType.MSG_TYPE_ERROR,
        15000
    );
    this.router.navigate(['/my-account/payment-details']);
  }

  verifyAddress(): void {
    if (this.addressForm.valid) {
      if(this.isSameAddressAlreadySaved(this.addressForm.value)){
        this.globalMessageService.add(
            {key: 'paymentDetails.sameAddressAlreadySaved'},
            GlobalMessageType.MSG_TYPE_WARNING
        );
      }
      if (this.addressForm.get('region').value.isocode) {
        this.regionsSub = this.regions$.pipe(take(1)).subscribe((regions) => {
          const obj = regions.find(
              (region) =>
                  region.isocode ===
                  this.addressForm.controls['region'].value.isocode
          );
          Object.assign(this.addressForm.value.region, {
            isocodeShort: obj.isocodeShort,
          });
        });
      }

      if (this.addressForm.dirty) {
        this.userAddressService
        .verifyAddress(this.addressForm.value)
        .subscribe((result) => {
          this.handleAddressVerificationResults(result);
        });
      }
    } else {
      this.addressForm.markAllAsTouched();
    }
  }

  protected handleAddressVerificationResults(results: AddressValidation) {
    if (results.decision === 'ACCEPT') {
      this.updateBillingAddress(this.addressForm.value);
    } else if (results.decision === 'REJECT') {
      this.globalMessageService.add(
          {key: 'addressForm.invalidAddress'},
          GlobalMessageType.MSG_TYPE_ERROR
      );

    } else if (results.decision === 'REVIEW') {
      this.openSuggestedAddress(results);
    }
  }

  openSuggestedAddress(results: AddressValidation): void {
    if (!this.suggestedAddressModalRef) {
      this.suggestedAddressModalRef = this.modalService.open(
          SuggestedAddressDialogComponent,
          {centered: true, size: 'lg'}
      );
      this.suggestedAddressModalRef.componentInstance.enteredAddress =
          this.addressForm.value;
      this.suggestedAddressModalRef.componentInstance.suggestedAddresses =
          results.suggestedAddresses;
      this.suggestedAddressModalRef.result
      .then((address) => {
        if (address) {
          address = Object.assign(
              {
                titleCode: this.addressForm.value.titleCode,
                phone: this.addressForm.value.phone,
                selected: true,
              },
              address
          );
          this.updateBillingAddress(address);
        }
        this.suggestedAddressModalRef = null;
      })
      .catch(() => {
        // this  callback is called when modal is closed with Esc key or clicking backdrop
        const address = Object.assign(
            {
              selected: true,
            },
            this.addressForm.value
        );
        this.updateBillingAddress(address);
        this.suggestedAddressModalRef = null;
      });
    }
  }

  countrySelected(country: Country): void {
    this.addressForm.get('country')?.get('isocode')?.setValue(country.isocode);
    this.selectedCountry$.next(country.isocode);
  }

  back(): void {
    this.router.navigate(['my-account/payment-details']);
  }

  private isSameAddressAlreadySaved(address: Address): boolean{
    let res = address.firstName === this.currentAddress.firstName &&
        address.lastName === this.currentAddress.lastName &&
        address.line1 === this.currentAddress.line1 &&
        address.line2 === this.currentAddress.line2 &&
        address.town === this.currentAddress.town &&
        address.postalCode === this.currentAddress.postalCode &&
        address.country.isocode === this.currentAddress.country.isocode;

    return this.regionEnable ? res && address.region.isocode === this.currentAddress.region.isocode : res;

  }

  ngOnDestroy(): void {
    if (this.regionsSub) {
      this.regionsSub.unsubscribe();
    }
  }

}
