import { Component, ElementRef, OnInit, ViewChild, ChangeDetectorRef, OnDestroy} from '@angular/core';
import {ActivatedRoute, Router} from '@angular/router';
import {Observable, combineLatest} from 'rxjs';
import {
  ActiveCartService,
  AuthRedirectStorageService,
  GlobalMessageService,
  AuthService,
  GlobalMessageType,
  TranslationService,
} from '@spartacus/core';
import {first} from 'rxjs/operators';
import {
  PaypalConnectService,
  PaypalPaymentDetailsService,
  BraintreeLoginService,
  AddingPaymentMethodAndAddressStatusData,
  PaypalCheckoutService,
  BraintreePaymentDetailsService,
  PageType
} from "braintree-spartacus-core";



@Component({
  selector: 'paypal-connect-waiting',
  templateUrl: './paypal-connect-waiting.component.html'
})
export class PaypalConnectWaitingComponent implements OnInit, OnDestroy {


  @ViewChild('paypalButton') paypalButton: ElementRef;
  keySessionAccessTokenGuid = 'accessTokenGuid';
  keySessionPayerId = 'payerId';
  wasUserRegistered = 'wasUserRegistered';
  didCustomerHavePayerId = 'didCustomerHavePayerId';
  redirectUrl = 'redirectUrl';
  accessTokenGuid: string;
  isCustomerRegistered = false;
  public message: Observable<string>;
  isRenderPayPalButton = true;
  isShouldSaveAddress = 'isShouldSaveAddress';
  isShouldSavePaymentInfo = 'isShouldSavePaymentInfo';
  isPayPalLoaded = false;


  constructor(
      protected paypalConnectService: PaypalConnectService,
      protected router: Router,
      protected auth: AuthService,
      protected paypalPaymentDetailsService: PaypalPaymentDetailsService,
      protected activatedRoute: ActivatedRoute,
      protected activeCartService: ActiveCartService,
      protected globalMessageService: GlobalMessageService,
      protected translation: TranslationService,
      protected authRedirectStorageService: AuthRedirectStorageService,
      protected cdr: ChangeDetectorRef,
      protected braintreePaymentDetailsService: BraintreePaymentDetailsService,
      protected braintreeLoginService: BraintreeLoginService,
  ) {
  }

  ngOnInit(): void {
    this.message = this.translation.translate('connectRegister.addPaymentMethodAndAddressMessage');
    this.authRedirectStorageService.getRedirectUrl()
    .pipe(first())
    .subscribe(url => {
      if(url.startsWith('/connect')) {
        this.authRedirectStorageService.setRedirectUrl(localStorage.getItem(this.redirectUrl))
      }
    });
    this.activatedRoute.queryParams
    .pipe(first())
    .subscribe(params => {
      const authorizationCode = params['code'];
      if (authorizationCode) {
        const storageAccessTokenGuid = localStorage.getItem(this.keySessionAccessTokenGuid);
        const storagePayerId = localStorage.getItem(this.keySessionPayerId);
        if (storageAccessTokenGuid && storagePayerId) {
          this.paypalPaymentDetailsService.loadPaymentDetails('login').subscribe(checkoutData => {
            if (checkoutData.configurationData.payPalConnectAddPaymentMethodFlow) {
              this.accessTokenGuid = storageAccessTokenGuid;
              this.isCustomerRegistered = true;
              this.cdr.detectChanges();
              this.prepareFormAddAddressAndPaymentInfo()
            } else {
                this.loginUser();
            }
          });
        } else {
          this.paypalConnectService.exchangeAuthorizationCode(authorizationCode)
          .subscribe(connectData => {
            this.accessTokenGuid = connectData.accessTokenGuid;
            localStorage.setItem(this.keySessionAccessTokenGuid, this.accessTokenGuid);
            localStorage.setItem(this.wasUserRegistered, String(connectData.isRegistered));
            localStorage.setItem(this.didCustomerHavePayerId, String(connectData.isUserAlreadyHaveAccount));
            localStorage.setItem(this.isShouldSavePaymentInfo, String(connectData.shouldSavePaymentInfo))
            localStorage.setItem(this.isShouldSaveAddress, String(connectData.shouldSaveAddress))
            if (connectData.isRegistered) {
              if (connectData.isUserAlreadyHaveAccount) {
                  this.removeLocalStorage();
                  this.paypalConnectService.login(this.accessTokenGuid).subscribe(flag => {
                  });
              } else {
                    this.paypalPaymentDetailsService.loadPaymentDetails('login').subscribe(checkoutData => {
                      if (!checkoutData.configurationData.enableLoginViaPayPalToTheExistingUser) {
                        this.addInfoMessage("connectRegister.accountAlreadyExists");
                        this.router.navigate(['/login']);
                      } else {
                      if (checkoutData.configurationData.payPalConnectAddPaymentMethodFlow) {
                        localStorage.setItem(this.keySessionPayerId, connectData.payerId);
                        this.prepareFormAddAddressAndPaymentInfo();
                      } else {
                        this.loginUser();
                      }
                    }
                    });
              }
            } else {
              this.paypalConnectService.register(this.accessTokenGuid).subscribe(payerId => {
                if (payerId) {
                  this.paypalPaymentDetailsService.loadPaymentDetails('login').subscribe(checkoutData => {
                    if (checkoutData.configurationData.payPalConnectAddPaymentMethodFlow) {
                        this.isCustomerRegistered = true;
                        this.cdr.detectChanges();
                        localStorage.setItem(this.keySessionPayerId, payerId);
                        this.prepareFormAddAddressAndPaymentInfo();
                    } else {
                        this.loginUser();
                    }
                  });
                } else {
                  this.addErrorMessage('connectRegister.error.somethingWentWrong');
                  this.router.navigate(['/login']);
                }
              }, err => {
                this.addErrorMessage('connectRegister.error.somethingWentWrong');
                this.router.navigate(['/login']);
              });
            }

          });
        }
      }else {
        const forced = localStorage.getItem('forced');
        localStorage.removeItem('forced');
        if (forced === 'true') {
          this.router.navigateByUrl('/login?forced=true');
        } else {
          this.router.navigate(['/login']);
        }
      }
    });
  }

  private addErrorMessage(key: string) {
    this.globalMessageService.add(
        {key: key},
        GlobalMessageType.MSG_TYPE_WARNING,
        10000
    );
    this.removeLocalStorage();
  }

  private addInfoMessage(key: string) {
    this.globalMessageService.add(
        {key: key},
        GlobalMessageType.MSG_TYPE_INFO,
        15000
    );
  }

  private addConfirmationMessage(key: string) {
    this.globalMessageService.add(
        {key: key},
        GlobalMessageType.MSG_TYPE_CONFIRMATION,
        10000
    );
  }

  private addWarningMessage(key: string) {
    this.globalMessageService.add(
        {key: key},
        GlobalMessageType.MSG_TYPE_WARNING,
        10000
    );
  }

  prepareFormAddAddressAndPaymentInfo() {
    this.isCustomerRegistered = true;

    const isShouldSaveAddress = localStorage.getItem(this.isShouldSaveAddress) == 'true'
    const isShouldSavePaymentInfo = localStorage.getItem(this.isShouldSavePaymentInfo) == 'true'

    if(isShouldSaveAddress && isShouldSavePaymentInfo){
      this.onSuccessRegistration(isShouldSaveAddress, isShouldSavePaymentInfo);
    }else if(isShouldSavePaymentInfo){
      this.message = this.translation.translate('connectRegister.addPaymentMethodMessage');
      this.onSuccessRegistration(isShouldSaveAddress, isShouldSavePaymentInfo);
    }else if(isShouldSaveAddress){
      this.message = this.translation.translate('connectRegister.addAddressMessage');
      this.isRenderPayPalButton = false;
      this.cdr.detectChanges();
    }else {
      this.addInfoMessage('connectRegister.youHaveAccountSoYouWillLogged');
      this.removeLocalStorage();
      this.paypalConnectService.login(this.accessTokenGuid).subscribe(flag => {
      });
    }
  }

  onSuccessRegistration(shouldSaveAddressInfo: boolean, shouldSavePaymentInfo: boolean) {
    combineLatest(
        [this.braintreePaymentDetailsService.getBraintreeButtonStyles('mark'),
          this.braintreePaymentDetailsService.loadPaymentDetailsForBillingPage('full', PageType.MY_ACCOUNT)])
    .subscribe(([buttonStyle, checkoutData]) => {
      this.paypalConnectService.initializePayPalAutomaticSavePaymentMethod(this.paypalButton.nativeElement, checkoutData,
          buttonStyle.payPalButtonStyle,PageType.LOGIN, (status: AddingPaymentMethodAndAddressStatusData) => {
            if (shouldSaveAddressInfo){
              if (status.isAddressAdded) {
                this.addConfirmationMessage('connectRegister.addressWasAddedSuccess');
              } else {
                this.addWarningMessage('connectRegister.error.addressNotAdded');
              }
            }
            if (shouldSavePaymentInfo) {
              if (status.isPaymentMethodAdded) {
                this.addConfirmationMessage('connectRegister.paymentMethodWasAddedSuccess');
              } else {
                this.addWarningMessage('connectRegister.error.paymentMethodNotAdded');
              }
            }
            this.paypalConnectService.login(this.accessTokenGuid).subscribe(flag => {
              this.displayingGlobalMessageToRegisteredAndNotRegisteredUser();
            });
            this.removeLocalStorage();
          }, shouldSaveAddressInfo, shouldSavePaymentInfo, this.accessTokenGuid);
      this.cdr.detectChanges();
      this.isPayPalLoaded = true;
    });
  }

  addOnlyAddress(){
    const onSuccessCallback = (status: AddingPaymentMethodAndAddressStatusData) => {
      if (status.isAddressAdded) {
        this.addConfirmationMessage('connectRegister.addressWasAddedSuccess');
      } else {
        this.addWarningMessage('connectRegister.error.addressNotAdded');
      }
      this.removeLocalStorage();
      this.paypalConnectService.login(this.accessTokenGuid).subscribe(flag => {
      });
      this.displayingGlobalMessageToRegisteredAndNotRegisteredUser();
    };
    this.braintreePaymentDetailsService.savePaymentDetailsAndAddressForNewUser(
        this.accessTokenGuid,
        onSuccessCallback,
        undefined,
        true,
        false,
        undefined,
        undefined
    );
  }


  loginUser() {
    this.removeLocalStorage();
    this.paypalConnectService.login(this.accessTokenGuid).subscribe(flag => {
      this.displayingGlobalMessageToRegisteredAndNotRegisteredUser();
    });
  }

  removeLocalStorage() {
    localStorage.removeItem(this.keySessionPayerId);
    localStorage.removeItem(this.keySessionAccessTokenGuid);
    localStorage.removeItem(this.redirectUrl);
    localStorage.removeItem(this.isShouldSaveAddress);
    localStorage.removeItem(this.isShouldSavePaymentInfo);
  }

  displayingGlobalMessageToRegisteredAndNotRegisteredUser() {
    const wasUserRegisteredMessage = localStorage.getItem(this.wasUserRegistered) == 'true';
    const didCustomerHavePayerIdMessage = localStorage.getItem(this.didCustomerHavePayerId) == 'true';
    if (!wasUserRegisteredMessage) {
      this.addInfoMessage('myAccount.passwordForSingleAuthentication');
    } else if (wasUserRegisteredMessage && !didCustomerHavePayerIdMessage) {
      this.addInfoMessage('connectRegister.youHaveAccountSoYouWillLogged');
    }
    localStorage.removeItem(this.wasUserRegistered);
    localStorage.removeItem(this.didCustomerHavePayerId);
  }

  ngOnDestroy(): void {
    if(this.auth.isUserLoggedIn()){
      this.braintreeLoginService.showExpireStatusMessage();
    }
  }



}

