import {Component, OnInit, ViewChild, ElementRef, ChangeDetectorRef, OnDestroy} from '@angular/core';
import {FormBuilder, FormGroup, Validators} from '@angular/forms';
import {ActivatedRoute, Router} from '@angular/router';
import { Observable, of, combineLatest} from 'rxjs';
import {first, map, switchMap, tap} from 'rxjs/operators';
import {
  AuthRedirectStorageService,
  Country,
  GlobalMessageService,
  GlobalMessageType,
  AuthService,
  Title, TranslationService,
  UserAddressService,
  UserService
} from '@spartacus/core';
import {sortTitles} from '@spartacus/storefront';

import {
  PayPalConnectB2BData,
  PayPalConnectB2BRegisterData,
  PaypalConnectService,
  PaypalPaymentDetailsService,
  AddingPaymentMethodAndAddressStatusData,
  PageType,
  BraintreeLoginService,
  BraintreePaymentDetailsService
} from 'braintree-spartacus-core';


@Component({
  selector: 'paypal-connect-additional',
  templateUrl: './paypal-connect-additional.component.html'
})
export class PaypalConnectAdditionalComponent implements OnInit, OnDestroy {

  @ViewChild('paypalButton') paypalButton: ElementRef;

  payPalConnectData$: Observable<PayPalConnectB2BData>;
  countries$: Observable<Country[]>;
  titles$: Observable<Title[]>;
  accessTokenGuid: string;
  public message: Observable<string>;

  keySessionAccessTokenGuid = 'accessTokenGuid';
  keySessionPayerId = 'payerId';
  isUserApproved = 'isUserApproved';
  wasUserRegistered = 'wasUserRegistered';
  didCustomerHavePayerId = 'didCustomerHavePayerId';
  redirectUrl = 'redirectUrl';
  isCustomerRegistered = false;
  showRegistrationSpinner = false;
  isShouldSaveAddress = 'isShouldSaveAddress';
  isShouldSavePaymentInfo = 'isShouldSavePaymentInfo';
  isRenderPayPalButton = true;
  shouldShowSetUpPasswordMessage = 'shouldShowSetUpPasswordMessage';

  registerForm: FormGroup = this.fb.group(
      {
        country: this.fb.group({
          isocode: [null, Validators.required],
        }),
        titleCode: [''],
        company: ['', Validators.required],
        addressLine1: ['', Validators.required],
        addressLine2: [''],
        city: ['', Validators.required],
        postcode: ['', Validators.required],
        yourPosition: ['', Validators.required],
        telephone: ['', Validators.required],
      }
  );

  constructor(
      protected translation: TranslationService,
      protected globalMessageService: GlobalMessageService,
      protected fb: FormBuilder,
      protected paypalConnectService: PaypalConnectService,
      protected activatedRoute: ActivatedRoute,
      protected userAddressService: UserAddressService,
      protected userRegister: UserService,
      protected authRedirectStorageService: AuthRedirectStorageService,
      protected router: Router,
      protected paypalPaymentDetailsService: PaypalPaymentDetailsService,
      protected cdr: ChangeDetectorRef,
      protected braintreePaymentDetailsService: BraintreePaymentDetailsService,
      protected braintreeLoginService: BraintreeLoginService,
      protected auth: AuthService,
  ) {
  }

  ngOnInit() {
    this.message = this.translation.translate('connectRegister.addPaymentMethodAndAddressMessage');
    this.authRedirectStorageService.getRedirectUrl()
    .pipe(first())
    .subscribe(url => {
      if(url.startsWith('/connect/additional')) {
        this.authRedirectStorageService.setRedirectUrl(localStorage.getItem(this.redirectUrl));
      }
    });

    const storageAccessTokenGuid = localStorage.getItem(this.keySessionAccessTokenGuid);
    if (storageAccessTokenGuid) {
      this.accessTokenGuid = storageAccessTokenGuid;
      this.payPalConnectData$ = of({});
    } else {
      this.payPalConnectData$ = this.activatedRoute.queryParams.pipe(
          first(),
          switchMap(params => {
            const authorizationCode = params['code'];
            if (authorizationCode) {
              return this.paypalConnectService.exchangeAuthorizationCodeB2B(authorizationCode);
            } else {
              this.router.navigate(['/login']);
            }
            return of(null);
          }),
          map((connectData: PayPalConnectB2BData) => {
            this.accessTokenGuid = connectData.accessTokenGuid;
            localStorage.setItem(this.keySessionAccessTokenGuid, connectData.accessTokenGuid);
            localStorage.setItem(this.wasUserRegistered, String(connectData.isRegistered));
            localStorage.setItem(this.didCustomerHavePayerId, String(connectData.isUserAlreadyConnectPayPalAccount));
            localStorage.setItem(this.isShouldSavePaymentInfo, String(connectData.shouldSavePaymentInfo))
            localStorage.setItem(this.isShouldSaveAddress, String(connectData.shouldSaveAddress))
            if (connectData.isRegistered && !connectData.isApproved) {
              if (connectData.isUserAlreadyConnectPayPalAccount) {
                this.globalMessageService.add(
                    {key: 'connectRegister.accountNotApproved'},
                    GlobalMessageType.MSG_TYPE_WARNING,
                    10000
                );
                this.removeLocalStorage();
                this.router.navigate(['/login']);
                return null;
              } else {
                this.paypalPaymentDetailsService.loadPaymentDetails('login').subscribe( checkoutData => {
                  if (!checkoutData.configurationData.enableLoginViaPayPalToTheExistingUser) {
                    this.removeLocalStorage();
                    this.addInfoMessage("connectRegister.accountAlreadyExists");
                    this.router.navigate(['/login']);
                  } else {
                  if (checkoutData.configurationData.payPalConnectAddPaymentMethodFlow) {
                    localStorage.setItem(this.keySessionPayerId, connectData.payerId);
                    this.prepareFormAddAddressAndPaymentInfo();
                  } else {
                    this.processLogin();
                  }
                }
                });
              }
            } else if (connectData.isRegistered) {
              localStorage.setItem(this.isUserApproved, String(connectData.isApproved));
              if (connectData.isUserAlreadyConnectPayPalAccount) {
                this.removeLocalStorage();
                if (localStorage.getItem(this.shouldShowSetUpPasswordMessage) == 'true') {
                  this.addInfoMessage('myAccount.passwordForSingleAuthentication');
                  localStorage.removeItem(this.shouldShowSetUpPasswordMessage);
                }
                this.paypalConnectService.login(connectData.accessTokenGuid).subscribe(flag => {
                });
                this.displayingGlobalMessageToRegisteredAndNotRegisteredUser();
                return null;
              } else {
                  this.paypalPaymentDetailsService.loadPaymentDetails('login').subscribe(checkoutData => {
                    if (!checkoutData.configurationData.enableLoginViaPayPalToTheExistingUser) {
                      this.removeLocalStorage();
                      this.addInfoMessage("connectRegister.accountAlreadyExists");
                      this.router.navigate(['/login']);
                    } else {
                    if (checkoutData.configurationData.payPalConnectAddPaymentMethodFlow) {
                      localStorage.setItem(this.keySessionPayerId, connectData.payerId);
                      this.prepareFormAddAddressAndPaymentInfo();
                    } else {
                      this.processLogin();
                    }
                    }
                  });
              }
            }
            return connectData;
          })
      );
    }

    this.countries$ = this.userAddressService.getDeliveryCountries().pipe(
        tap((countries: Country[]) => {
          if (Object.keys(countries).length === 0) {
            this.userAddressService.loadDeliveryCountries();
          }
        })
    );

    this.titles$ = this.userRegister.getTitles().pipe(
        map((titles: Title[]) => {
          return titles.sort(sortTitles);
        })
    );

    const storagePayerId = localStorage.getItem(this.keySessionPayerId);
    this.paypalPaymentDetailsService.loadPaymentDetails('login').subscribe( checkoutData => {
      if (storageAccessTokenGuid && storagePayerId && checkoutData.configurationData.payPalConnectAddPaymentMethodFlow) {
        this.isCustomerRegistered = true;
        this.cdr.detectChanges();
        this.prepareFormAddAddressAndPaymentInfo()
      }
    });

  }

  countrySelected(country: Country): void {
    this.registerForm.get('country')?.get('isocode')?.setValue(country.isocode);
  }

  submitForm(): void {
    if (this.registerForm.valid) {
      localStorage.setItem(this.shouldShowSetUpPasswordMessage, 'true');
      this.showRegistrationSpinner = true;
      this.paypalConnectService.registerB2BCustomer(this.accessTokenGuid, this.collectDataFromForm(this.registerForm)).subscribe(payerId => {
        if (payerId) {
          this.paypalPaymentDetailsService.loadPaymentDetails('login').subscribe(checkoutData => {
              if (checkoutData.configurationData.payPalConnectAddPaymentMethodFlow) {
                this.isCustomerRegistered = true;
                this.cdr.detectChanges();
                this.addInfoMessage('connectRegister.successfulRegistration');
                localStorage.setItem(this.keySessionPayerId, payerId);
                this.prepareFormAddAddressAndPaymentInfo();
              } else {
                this.addInfoMessage('connectRegister.successfulRegistration');
                this.processLogin();
              }
          });
        } else {
          this.addErrorMessage('connectRegister.error.somethingWentWrong');
          this.removeLocalStorage();
          this.router.navigate(['/login']);

        }
      }, err => {
        this.addErrorMessage('connectRegister.error.somethingWentWrong');
        this.router.navigate(['/login']);
      });
    } else {
      this.registerForm.markAllAsTouched();
    }
  }

  onSuccessRegistration(shouldSaveAddressInfo: boolean, shouldSavePaymentInfo: boolean) {
    combineLatest(
        [this.braintreePaymentDetailsService.getBraintreeButtonStyles('mark'),
          this.braintreePaymentDetailsService.loadPaymentDetailsForBillingPage('full', PageType.MY_ACCOUNT)])
    .subscribe(([buttonStyle, checkoutData]) => {
      this.paypalConnectService.initializePayPalAutomaticSavePaymentMethod(this.paypalButton.nativeElement, checkoutData,
          buttonStyle.payPalButtonStyle,PageType.LOGIN, (status: AddingPaymentMethodAndAddressStatusData) => {
            if (shouldSaveAddressInfo){
              if (status.isAddressAdded) {
                this.addConfirmationMessage('connectRegister.addressWasAddedSuccess');
              } else {
                this.addWarningMessage('connectRegister.error.addressNotAdded');
              }
            }
            if (shouldSavePaymentInfo) {
              if (status.isPaymentMethodAdded) {
                this.addConfirmationMessage('connectRegister.paymentMethodWasAddedSuccess');
              } else {
                this.addWarningMessage('connectRegister.error.paymentMethodNotAdded');
              }
            }
            this.processLogin();
          }, shouldSaveAddressInfo, shouldSavePaymentInfo, this.accessTokenGuid);
    });
    this.showRegistrationSpinner = false;
    this.cdr.detectChanges();
  }

  addOnlyAddress(){
    const onSuccessCallback = (status: AddingPaymentMethodAndAddressStatusData) => {
      if (status.isAddressAdded) {
        this.addConfirmationMessage('connectRegister.addressWasAddedSuccess');
      } else {
        this.addWarningMessage('connectRegister.error.addressNotAdded');
      }
      this.processLogin();
      this.displayingGlobalMessageToRegisteredAndNotRegisteredUser();
    };
    this.braintreePaymentDetailsService.savePaymentDetailsAndAddressForNewUser(
        this.accessTokenGuid,
        onSuccessCallback,
        undefined,
        true,
        false,
        undefined,
        undefined
    );
  }


  collectDataFromForm(formData: FormGroup): PayPalConnectB2BRegisterData {
    return {
      companyAddressCountryIso: formData.value.country.isocode,
      titleCode: formData.value.titleCode,
      companyName: formData.value.company,
      companyAddressStreet: formData.value.addressLine1,
      companyAddressStreetLine2: formData.value.addressLine2,
      companyAddressCity: formData.value.city,
      companyAddressPostalCode: formData.value.postcode,
      position: formData.value.yourPosition,
      telephone: formData.value.telephone
    };
  }

  processLogin(){
    const storageIsUserApproved = localStorage.getItem(this.isUserApproved);
    this.removeLocalStorage();
    if(storageIsUserApproved) {
      this.paypalConnectService.login(this.accessTokenGuid).subscribe(flag => {
      });
    }else {
      this.router.navigate(['/login']);
    }
    this.displayingGlobalMessageToRegisteredAndNotRegisteredUser();
  }

  private addErrorMessage(key: string) {
    this.globalMessageService.add(
        {key: key},
        GlobalMessageType.MSG_TYPE_WARNING,
        10000
    );
    this.removeLocalStorage();
  }

  private addInfoMessage(key: string) {
    this.globalMessageService.add(
        {key: key},
        GlobalMessageType.MSG_TYPE_INFO, 15000
    );
  }

  private addWarningMessage(key: string) {
    this.globalMessageService.add(
        {key: key},
        GlobalMessageType.MSG_TYPE_WARNING
    );
  }

  private addConfirmationMessage(key: string) {
    this.globalMessageService.add(
        {key: key},
        GlobalMessageType.MSG_TYPE_CONFIRMATION,
        10000
    );
  }



  removeLocalStorage() {
    localStorage.removeItem(this.keySessionPayerId);
    localStorage.removeItem(this.keySessionAccessTokenGuid);
    localStorage.removeItem(this.isUserApproved);
    localStorage.removeItem(this.redirectUrl);
    localStorage.removeItem(this.isShouldSaveAddress);
    localStorage.removeItem(this.isShouldSavePaymentInfo);
  }

  displayingGlobalMessageToRegisteredAndNotRegisteredUser() {
    const wasUserRegisteredMessage = localStorage.getItem(this.wasUserRegistered) == 'true';
    const didCustomerHavePayerIdMessage = localStorage.getItem(this.didCustomerHavePayerId) == 'true';
    if (wasUserRegisteredMessage && !didCustomerHavePayerIdMessage) {
      this.addInfoMessage('connectRegister.youHaveAccountSoYouWillLogged');
    }
    localStorage.removeItem(this.wasUserRegistered);
    localStorage.removeItem(this.didCustomerHavePayerId);
  }

  prepareFormAddAddressAndPaymentInfo() {
    this.isCustomerRegistered = true;

    const isShouldSaveAddress = localStorage.getItem(this.isShouldSaveAddress) == 'true'
    const isShouldSavePaymentInfo = localStorage.getItem(this.isShouldSavePaymentInfo) == 'true'

    if(isShouldSaveAddress && isShouldSavePaymentInfo){
      this.onSuccessRegistration(isShouldSaveAddress, isShouldSavePaymentInfo);
    }else if(isShouldSavePaymentInfo){
      this.message = this.translation.translate('connectRegister.addPaymentMethodMessage');
      this.onSuccessRegistration(isShouldSaveAddress, isShouldSavePaymentInfo);
    }else if(isShouldSaveAddress){
      this.message = this.translation.translate('connectRegister.addAddressMessage');
      this.isRenderPayPalButton = false;
      this.showRegistrationSpinner = false;
      this.cdr.detectChanges();
    } else {
      this.processLogin();
    }
  }

  ngOnDestroy(): void {
    if(this.auth.isUserLoggedIn()){
      this.braintreeLoginService.showExpireStatusMessage();
    }
  }

}

