import {Component, ElementRef, OnInit, ViewChild} from '@angular/core';
import {combineLatest, Observable} from 'rxjs';
import {map} from 'rxjs/operators';
import {
  GooglePayCheckoutService,
  PageType,
  CheckoutData,
  PayPalButtonStyle,
  BraintreeButtonStyle,
  PaypalCheckoutService,
  BraintreePaymentDetailsService,
  ApplePayCheckoutService, BraintreeSrcCheckoutService
} from 'braintree-spartacus-core';
import {BraintreeCheckoutConfigService} from '../../../services/braintree-checkout-config.service';
import {
  OCC_USER_ID_ANONYMOUS,
  UserIdService
} from '@spartacus/core';

@Component({
  selector: 'bt-cart',
  templateUrl: './braintree-cart.component.html'
})
export class BraintreeCartComponent implements OnInit {

  @ViewChild('paypalButtonContainer') paypalButtonContainer: ElementRef;
  @ViewChild('applePayButtonContainer') applePayButtonContainer: ElementRef;

  loadedCheckoutData: Observable<CheckoutData>;
  pageType = PageType.CART;

  braintreeButtonStyle: BraintreeButtonStyle;

  applePayStyle: string;
  applePayType: string;
  applePayRadius: string;

  constructor(
    protected paypalService: PaypalCheckoutService,
    protected applePayService: ApplePayCheckoutService,
    protected googlePayService: GooglePayCheckoutService,
    protected srcCheckoutService: BraintreeSrcCheckoutService,
    protected braintreePaymentDetailsService: BraintreePaymentDetailsService,
    protected braintreeCheckoutServ: BraintreeCheckoutConfigService,
    protected userIdService: UserIdService
) { }

  ngOnInit(): void {
      this.loadedCheckoutData = combineLatest(
        [this.braintreePaymentDetailsService.getBraintreeButtonStyles('cart'),
          this.braintreePaymentDetailsService.loadPaymentDetails('full', PageType.CART)])
        .pipe(
          map(
            ([buttonStyle, checkoutData]) => {
              this.braintreeButtonStyle = buttonStyle;
              this.initializeApplePayStyle(buttonStyle);
              this.initializeCartPage(checkoutData, buttonStyle);
              return checkoutData;
            }));
  }

  isPaymentAllowed(): boolean{
    return !this.isAnonymousUser() || this.braintreeCheckoutServ.isAllowedGuestExpressCheckout();
  }

  initializeCartPage(checkoutData: CheckoutData, buttonStyle: BraintreeButtonStyle): void {
    if (this.isPaymentAllowed() && checkoutData.payPalPaymentMethod.payPalExpressEnabled) {
      this.paypalService.initializePayPalCheckout(
        this.paypalButtonContainer.nativeElement,
        checkoutData,
        buttonStyle.payPalButtonStyle,
        this.pageType
      );
    } else{
      this.paypalButtonContainer.nativeElement.remove();
    }
    if (this.isPaymentAllowed() && checkoutData.googlePayPaymentMethod.googlePayEnabled &&
        checkoutData.googlePayPaymentMethod.googlePayOnExpressCheckoutEnabled) {
      this.googlePayService.loadGooglePay(() => {
        this.googlePayService.initialiseGooglePay(checkoutData, buttonStyle.googlePayButtonStyle, this.pageType);
      });
    }
    if (this.isPaymentAllowed() && checkoutData.srcPaymentMethod.srcEnabled) {
      this.srcCheckoutService.initializeSrc(checkoutData, this.pageType);
    }
    if (this.isPaymentAllowed() && checkoutData.applePayPaymentMethod.applePayEnabled === 'true'
        && checkoutData.applePayPaymentMethod.applePayOnExpressCheckoutEnabled && this.applePayService.isAvailableApplePay()) {
      this.applePayService.initalizeApplePay(checkoutData, this.pageType, this.applePayButtonContainer.nativeElement);
    } else {
      this.applePayButtonContainer.nativeElement.remove();
    }
  }

  isAnonymousUser(): boolean{
    let userId;
    this.userIdService
        .getUserId()
        .subscribe((occUserId) => (userId = occUserId))
        .unsubscribe();
    return OCC_USER_ID_ANONYMOUS === userId;
  }

  getApplePayBorderRadius(shape: string): string {
    if (shape === 'squareCorners') {
      return '0px';
    } else if (shape === 'pill-shaped') {
      return '15px';
    }
  }

  initializeApplePayStyle(buttonStyle: BraintreeButtonStyle) {
    this.applePayStyle = buttonStyle.applePayButtonStyle.color;
    this.applePayType = buttonStyle.applePayButtonStyle.type;
    this.applePayRadius = this.getApplePayBorderRadius(buttonStyle.applePayButtonStyle.shape);
  }
}
