import {ChangeDetectorRef, Component, ElementRef, EventEmitter, Input, OnInit, Output, ViewChild} from '@angular/core';
import {combineLatest, Observable} from 'rxjs';
import {map} from 'rxjs/operators';
import {
    BraintreeCheckoutService,
    BraintreeDropInService, BraintreeExpressCheckoutService,
    BraintreePaymentDetailsService, BraintreeUtilsService,
    CheckoutData,
    DropInButtonStyle,
    PageType
} from 'braintree-spartacus-core';
import * as CONST from 'braintree-spartacus-core';
import {BraintreeCheckoutConfigService} from '../../../services/braintree-checkout-config.service';
import {GlobalMessageService, GlobalMessageType, OCC_USER_ID_ANONYMOUS, UserIdService} from '@spartacus/core';


@Component({
  selector: 'bt-cart-drop-in',
  templateUrl: './braintree-cart-drop-in.component.html'
})
export class BraintreeCartDropInComponent implements OnInit {

  @ViewChild('dropInButtonContainer') dropInButton: ElementRef;
  @ViewChild('dropInSubmitButton') dropInSubmitButton: ElementRef;

  loadedCheckoutData$: Observable<CheckoutData>;
  pageType = PageType.CART;
  hideButtons = true;
  isAvailableSavedPaymentMethods = false;
  isDropInLoaded = false;
  payPalOptionsAmount: number;
  hideSpinner = true;


    constructor(
      protected braintreeUtils: BraintreeUtilsService,
      protected braintreeExpressCheckoutService: BraintreeExpressCheckoutService,
      protected braintreeDropInService: BraintreeDropInService,
      protected braintreePaymentDetailsService: BraintreePaymentDetailsService,
      protected braintreeCheckoutServ: BraintreeCheckoutConfigService,
      protected userIdService: UserIdService,
      private cdr: ChangeDetectorRef,
      protected globalMessageService: GlobalMessageService,
  ) {
  }

  ngOnInit(): void {
    if (!this.isAnonymousUser() || this.braintreeCheckoutServ.isAllowedGuestExpressCheckout()) {
      this.hideSpinner = false;
      this.loadedCheckoutData$ = combineLatest(
          [ this.braintreeDropInService.getDropInButtonStyle('Cart'),
              this.braintreePaymentDetailsService.loadPaymentDetailsForBillingPage('full', this.pageType)])
      .pipe(
          map(([dropInButtonStyle , checkoutData]) => {
              if (checkoutData.payPalPaymentMethod.payPalExpressEnabled
                  || checkoutData.googlePayPaymentMethod.googlePayEnabled
                  || (checkoutData.applePayPaymentMethod.applePayEnabled === 'true')
              ) {
                  this.braintreeDropInService.initializeDropIn(this.dropInButton.nativeElement, this.dropInSubmitButton.nativeElement,
                      checkoutData, dropInButtonStyle, this.pageType, (dropInInstance) => {
                          this.isDropInLoaded = true;
                          dropInInstance.on('changeActiveView', (event) => {
                              if (event.newViewId === 'methods') {
                                  if (dropInInstance._model._activePaymentMethod != null
                                          && (event.previousViewId === 'paypal'
                                          || event.previousViewId === 'paypalCredit'
                                          || event.previousViewId === 'googlePay'
                                          || event.previousViewId === 'applePay')) {
                                      dropInInstance.requestPaymentMethod((err, payload) => {
                                          if (err) {
                                              console.log(err);
                                              this.showErrorMessage(err.message);
                                              return;
                                          }
                                          this.braintreeDropInService.sendPayloadNonceToServer(payload, checkoutData, this.pageType);
                                      });
                                  }
                                  this.changeCorrectView(dropInInstance, checkoutData);
                              }
                              this.cdr.detectChanges();
                          });
                          dropInInstance.on('paymentOptionSelected', (event) => {
                              if ((event.paymentOption === 'paypal' || event.paymentOption === 'paypalCredit')
                                  && checkoutData.payPalPaymentMethod.shouldShowShippingAddressMessage
                                  && (checkoutData.configurationData.intent === CONST.INTENT_ORDER)) {
                                  this.changeCorrectView(dropInInstance, checkoutData);
                                  this.showErrorMessage('error.braintree_message_no_shipping_address');
                              }
                              this.cdr.detectChanges();
                          });
                          dropInInstance.clearSelectedPaymentMethod();
                          this.changeCorrectView(dropInInstance, checkoutData);
                          this.cdr.detectChanges();
                  });
                  this.hideButtons = false;
              }
              return checkoutData;
          })
      );
    }
  }

  private showErrorMessage(message): void {
    this.globalMessageService.add(
        {
          key: message,
        },
        GlobalMessageType.MSG_TYPE_ERROR
    );
  }

  isAnonymousUser(): boolean {
    let userId;
    this.userIdService
    .getUserId()
    .subscribe((occUserId) => (userId = occUserId))
    .unsubscribe();
    return OCC_USER_ID_ANONYMOUS === userId;
  }

  private changeCorrectView(dropInInstance: any, checkoutData: CheckoutData): void {
    if(dropInInstance._model.supportedPaymentOptions.length > 1){
      dropInInstance._mainView.setPrimaryView('options', undefined);
    }else{
      dropInInstance._mainView.setPrimaryView(dropInInstance._model.supportedPaymentOptions[0], undefined);
    }
  }
}
