import {Component, Input, OnInit} from '@angular/core';
import { FormBuilder, FormGroup, AbstractControl, FormControl, Validators} from '@angular/forms';
import { DatePipe } from '@angular/common';
import {
  PageType,
  UsBankAccountCheckoutService,
  CheckoutData
} from 'braintree-spartacus-core';
import {Observable, BehaviorSubject} from 'rxjs';
import {ActiveCartService,  UserAddressService, Region} from '@spartacus/core';
import {switchMap,tap} from 'rxjs/operators';
import {CheckoutFacade} from '@spartacus/checkout/root';

@Component({
  selector: 'bt-us-bank-account',
  templateUrl: './braintree-us-bank-account.component.html'
})
export class BrainTreeUsBankAccountComponent implements OnInit {

  @Input() checkoutData: CheckoutData;
  @Input() pageType = PageType.BILLING;
  @Input() isSaveUnchecked = false;


  accountTypes = [
    {type: 'Checking'},
    {type: 'Savings'},
];

  ownershipTypes = [
    {type: 'Personal'},
    {type: 'Business'},
];

  usBankAccountForm:FormGroup;
  accountType = this.accountTypes[0];
  ownershipType = this.ownershipTypes[0];
  firstName = '';
  lastName = '';
  depositoryName = '';
  routingNumber = '';
  accountNumber = '';
  date = this.setCurrentDate();
  usBankAccountMessage: String;
  order$: Observable<any>;
  cart$: Observable<any>;
  selectedCountry$: BehaviorSubject<string> = new BehaviorSubject<string>('');
  regions$: Observable<Region[]>;
  isGuestCustomer = false;
  orderGuid: string;
  price: string;


  ngOnInit() {
    this.setOrderPrice();
    this.regions$ = this.userAddressService.getRegions('US');
  }

  constructor(
    protected formBuilder: FormBuilder,
    protected usBankAccountCheckoutService: UsBankAccountCheckoutService,
    protected checkoutFacade: CheckoutFacade,
    protected userAddressService: UserAddressService,
    protected activeCartService: ActiveCartService,
    ) {
    this.usBankAccountForm = this.formBuilder.group({
        accountNumber: this.formBuilder.control('', [
            Validators.minLength(4),
            Validators.maxLength(17)
        ]),
        routingNumber: this.formBuilder.control('', [
            Validators.minLength(9),
            Validators.maxLength(9)
        ]),
        accountType: '',
        ownershipType: '',
        firstName: this.formBuilder.control('', [
            Validators.required
        ]),
        lastName: this.formBuilder.control('', [
            Validators.required
        ]),
        businessName: this.formBuilder.control('', [
            Validators.required
        ]),
        streetAddress: this.formBuilder.control('', [
            Validators.required
        ]),
        extendedAddress: '',
        locality: this.formBuilder.control('', [
            Validators.required
        ]),
        region: this.formBuilder.group({
            isocodeShort: 'AL',
        }),
        postalCode: this.formBuilder.control('', [
            Validators.minLength(5),
            Validators.maxLength(10)
        ])
    });
  }

  onChangeAccountNumber(event: any) {
    event.target.value = event.target.value.replace(/[^0-9.]/g, '').replace(/(\..*)\./g, '$1');
    this.accountNumber = event.target.value;
    this.usBankAccountForm.controls['accountNumber'].setValue(event.target.value);
  }

  onChangeRoutingNumber(event: any) {
    event.target.value = event.target.value.replace(/[^0-9.]/g, '').replace(/(\..*)\./g, '$1');
    this.routingNumber = event.target.value;
    this.usBankAccountForm.controls['routingNumber'].setValue(event.target.value);
  }

  onChangeOwnershipType(event: any) {
    this.firstName = '';
    this.lastName = '';
    this.depositoryName = '';
    this.usBankAccountForm.controls['firstName'].setValue('');
    this.usBankAccountForm.controls['lastName'].setValue('');
    this.usBankAccountForm.controls['businessName'].setValue('');
  }

  onChangeFirstName(event: any) {
    this.firstName = event.target.value;
    this.depositoryName = event.target.value + ' ' + this.lastName;
  }

  onChangeLastName(event: any) {
    this.lastName = event.target.value;
    this.depositoryName = this.firstName + ' ' + event.target.value;
  }

  onChangeBusinessName(event: any) {
    this.depositoryName = event.target.value;
  }

  onSubmitusBankAccountFrom() {
    if (this.dataVerified()){
      this.usBankAccountCheckoutService.tokenizeUsBankAccount(this.checkoutData, this.usBankAccountForm, document.getElementById('mandate-text').innerText, this.pageType);
    } else {
      this.usBankAccountCheckoutService.errorInvalidPaymentDetails();
    }
  }

  private dataVerified() {
    if (this.usBankAccountForm.value.ownershipType.type === 'Personal') {
      this.usBankAccountForm.controls['businessName'].setErrors(null);
    } else {
      this.usBankAccountForm.controls['firstName'].setErrors(null);
      this.usBankAccountForm.controls['lastName'].setErrors(null);
    }

    if (this.usBankAccountForm.invalid) {
      this.usBankAccountCheckoutService.errorInvalidPaymentDetails();
      Object.values(this.usBankAccountForm.controls).forEach(control => control.markAsTouched());
      return false;
    } else {
      return true;
    }
  }

  setOrderPrice() {
    this.order$ = this.checkoutFacade.getOrderDetails().pipe(
      tap((order) => {
          this.isGuestCustomer =
              'guestCustomer' in order ? order.guestCustomer ?? false : false;
          this.orderGuid = order.guid as string;
      })
  );

  this.activeCartService.getActive().subscribe((res) => {
    this.price = res.totalPrice.formattedValue;
  });
  }

  setCurrentDate(): string {
    var datePipe = new DatePipe("en-US");
    return datePipe.transform(new Date(), "MM-dd-yyyy");
  }
}
