import {AfterViewInit, Component, ElementRef, EventEmitter, Input, OnInit, Output, ViewChild} from '@angular/core';
import { FormBuilder, FormGroup, Validators } from '@angular/forms';
import { Address, Country, Region, UserAddressService, UserPaymentService } from '@spartacus/core';
import {
  CheckoutDeliveryFacade,
  CheckoutPaymentFacade,
} from '@spartacus/checkout/root';

import { Card, ICON_TYPE } from '@spartacus/storefront';
import { BehaviorSubject, Observable } from 'rxjs';
import { switchMap, tap } from 'rxjs/operators';
import {
  BraintreePaymentMethodsUtilsService,
  BraintreePaymentDetailsService,
  PageType,
  BraintreeHostedFieldsCheckoutService,
  CheckoutData
} from 'braintree-spartacus-core';

@Component({
  selector: 'bt-hosted-fields',
  templateUrl: './braintree-hosted-fields.component.html'
})
export class BraintreeHostedFieldsComponent implements OnInit, AfterViewInit {
  iconTypes = ICON_TYPE;

  @Input() checkoutData: CheckoutData;
  @Input() pageType = PageType.BILLING;
  @ViewChild('buttonContainer') buttonContainer: ElementRef;

  shippingAddress$: Observable<Address>;
  regions$: Observable<Region[]>;
  countries$: Observable<Country[]>;
  selectedCountry$: BehaviorSubject<string> = new BehaviorSubject<string>('');
  sameAsShippingAddress = true;
  isSubmitted = false;

  billingAddressForm: FormGroup = this.fb.group({
    firstName: ['', Validators.required],
    lastName: ['', Validators.required],
    line1: ['', Validators.required],
    line2: [''],
    town: ['', Validators.required],
    region: this.fb.group({
      isocodeShort: [null, Validators.required],
    }),
    country: this.fb.group({
      isocode: [null, Validators.required],
    }),
    postalCode: ['', Validators.required],
  });

  constructor(
    protected checkoutPaymentService: CheckoutPaymentFacade,
    protected checkoutDeliveryService: CheckoutDeliveryFacade,
    protected userPaymentService: UserPaymentService,
    protected fb: FormBuilder,
    protected braintreeHostedFieldsCheckoutService: BraintreeHostedFieldsCheckoutService,
    protected userAddressService: UserAddressService,
    protected paymentMethodsUtilsService: BraintreePaymentMethodsUtilsService,
  ) { }

  ngAfterViewInit(): void {
    this.braintreeHostedFieldsCheckoutService.initializeHostedFields(this.checkoutData,
      this.buttonContainer.nativeElement, this.pageType);

  }

  ngOnInit(): void {
    this.shippingAddress$ = this.checkoutDeliveryService.getDeliveryAddress();

    this.countries$ = this.userPaymentService.getAllBillingCountries().pipe(
      tap((countries) => {
        // If the store is empty fetch countries. This is also used when changing language.
        if (Object.keys(countries).length === 0) {
          this.userPaymentService.loadBillingCountries();
        }
      })
    );

    this.regions$ = this.selectedCountry$.pipe(
      switchMap((country) => this.userAddressService.getRegions(country)),
      tap((regions) => {
        const regionControl = this.billingAddressForm.get(
          'region.isocodeShort'
        );
        if (regions.length > 0) {
          regionControl.enable();
        } else {
          regionControl.disable();
        }
      })
    );
  }

  toggleSameAsShippingAddress(): void {
    this.sameAsShippingAddress = !this.sameAsShippingAddress;
  }

  processHostedFieldsPayment(billingAddress: Address): void{
  if (this.braintreeHostedFieldsCheckoutService.isHostedFieldsValid()) {
    if (PageType.MY_ACCOUNT === this.pageType){
      this.braintreeHostedFieldsCheckoutService.tokenizeHostedFields(this.sameAsShippingAddress,
          this.paymentMethodsUtilsService.getBillingAddress());
    } else if (this.sameAsShippingAddress) {
        this.braintreeHostedFieldsCheckoutService.tokenizeHostedFields(this.sameAsShippingAddress, billingAddress);
      } else {
        if (this.billingAddressForm.valid) {
          this.braintreeHostedFieldsCheckoutService.tokenizeHostedFields(this.sameAsShippingAddress, billingAddress);
        } else {
          this.billingAddressForm.markAllAsTouched();
        }
      }
    } else {

      if (!this.sameAsShippingAddress) {
        this.billingAddressForm.markAllAsTouched();
      }
    }
  }

  getAddressCardContent(address: Address): Card {
    let region = '';
    if (address.region && address.region.isocode) {
      region = address.region.isocode + ', ';
    }

    return {
      textBold: address.firstName + ' ' + address.lastName,
      text: [
        address.line1,
        address.line2,
        address.town + ', ' + region + address.country.isocode,
        address.postalCode,
        address.phone,
      ],
    };
  }

  countrySelected(country: Country): void {
    this.billingAddressForm.get('country.isocode').setValue(country.isocode);
    this.selectedCountry$.next(country.isocode);
  }

  isBillingPage(): boolean{
    return PageType.BILLING === this.pageType;
  }

  onSubmit(billingAddressForm: FormGroup) {
    this.processHostedFieldsPayment(billingAddressForm.value);
  }
}
