import {
  BraintreeChangedPaymentDetailsEvent,
  BraintreeHostedFieldsCheckoutService,
  BraintreePaymentDetails,
  BraintreePaymentDetailsService,
  BraintreePaymentMethodsUtilsService,
  CheckoutData,
  CreatedAndSetBraintreePaymentDetailsEvent,
  Fields,
  PageType
} from 'braintree-spartacus-core';
import {Component, OnDestroy, OnInit} from '@angular/core';
import {ActivatedRoute} from '@angular/router';
import {
  ActiveCartService,
  Address,
  EventService,
  GlobalMessageService,
  Occ,
  TranslationService,
  UserPaymentService,
} from '@spartacus/core';
import {
    CheckoutDeliveryFacade,
    CheckoutPaymentFacade,
    CheckoutFacade,
} from '@spartacus/checkout/root';
import {Card} from '@spartacus/storefront';
import {CheckoutStepService, PaymentMethodComponent} from '@spartacus/checkout/components';
import {combineLatest, Observable, of} from 'rxjs';
import {map, take, tap} from 'rxjs/operators';
import PaymentDetails = Occ.PaymentDetails;

@Component({
  selector: 'bt-billing',
  templateUrl: './braintree-billing.component.html'
})
export class BraintreeBillingComponent extends PaymentMethodComponent implements OnInit, OnDestroy {

  loadedCheckoutData$: Observable<CheckoutData>;
  paymentMethods: BraintreePaymentDetails[];
  firstTryPayCard: boolean;
  currentPaymentMethod: PaymentDetails;
  checkoutData: CheckoutData;
  showCtnButton = true;

  constructor(
      protected userPaymentService: UserPaymentService,
      protected checkoutService: CheckoutFacade,
      protected checkoutDeliveryService: CheckoutDeliveryFacade,
      protected checkoutPaymentService: CheckoutPaymentFacade,
      protected globalMessageService: GlobalMessageService,
      protected activatedRoute: ActivatedRoute,
      protected translation: TranslationService,
      protected activeCartService: ActiveCartService,
      protected checkoutStepService: CheckoutStepService,
      protected braintreeHostedFieldsService: BraintreeHostedFieldsCheckoutService,
      protected braintreePaymentDetailsService: BraintreePaymentDetailsService,
      protected braintreePaymentMethodUtilsService: BraintreePaymentMethodsUtilsService,
      protected eventService: EventService,
  ) {
    super(
        userPaymentService,
        checkoutService,
        checkoutDeliveryService,
        checkoutPaymentService,
        globalMessageService,
        activatedRoute,
        translation,
        activeCartService,
        checkoutStepService
    );
  }

  ngOnInit(): void {

    this.shouldRedirect = false;
    this.isLoading$ = this.userPaymentService.getPaymentMethodsLoading();

    this.loadedCheckoutData$ = this.braintreePaymentDetailsService.loadPaymentDetailsForBillingPage(
        'full', PageType.BILLING
    );

    if (!this.activeCartService.isGuestCart()) {
      this.userPaymentService.loadPaymentMethods();
      this.existingPaymentMethods$ = this.braintreePaymentDetailsService.getBraintreePaymentDetails(
          true,
          Fields.Full
      );

    } else {
      this.isGuestCheckout = true;
      this.existingPaymentMethods$ = of([]);
    }

    this.checkoutDeliveryService
    .getDeliveryAddress()
    .pipe(take(1))
    .subscribe((address: Address) => {
      this.deliveryAddress = address;
    });

    this.createdAndSetBraintreePaymentDetails();
    this.firstTryPayCard = true;
  }

  ngOnDestroy(): void {
    super.ngOnDestroy();
  }

  get selectedMethod(): Observable<PaymentDetails> {
    this.selectedMethod$ = this.checkoutPaymentService.getPaymentDetails().pipe(
        tap((paymentInfo) => {
          if (paymentInfo && !!Object.keys(paymentInfo).length) {
            if (paymentInfo['hasError']) {
              Object.keys(paymentInfo).forEach((key) => {
                if (key.startsWith('InvalidField')) {
                  this.sendPaymentMethodFailGlobalMessage(paymentInfo[key]);
                }
              });
              this.checkoutService.clearCheckoutStep(3);
            } else if (this.shouldRedirect) {
              this.next();
            }
          }
        })
    );
    if (this.currentPaymentMethod !== undefined && this.checkoutData.creditCardPaymentMethod.verifyFlow) {
      var isCurrentMethodCC= this.currentPaymentMethod.subscriptionId !== 'CreditCard';
      this.showCtnButton = isCurrentMethodCC;

      this.selectedMethod$.subscribe(nextPayment => {
        if (!isCurrentMethodCC && nextPayment.subscriptionId === 'CreditCard') {
          this.braintreeHostedFieldsService.initializeCVVCheck(this.checkoutData); 
        }
        this.currentPaymentMethod = nextPayment;
      });
    }    
    return this.selectedMethod$;
  }

  


  selectBraintreePaymentDetails(selectedPaymentDetails: BraintreePaymentDetails, checkoutData: CheckoutData): void {
    if (checkoutData.creditCardPaymentMethod.verifyFlow && selectedPaymentDetails.subscriptionId === 'CreditCard') {
        this.checkoutData = checkoutData;
        this.currentPaymentMethod = selectedPaymentDetails;
        this.showCtnButton = false;
        this.braintreeHostedFieldsService.initializeCVVCheck(checkoutData); 
    } else {
    if (
        (checkoutData.creditCardPaymentMethod.secure3d || checkoutData.creditCardPaymentMethod.secure3dFallback) &&
        (selectedPaymentDetails.subscriptionId === 'CreditCard'
            || selectedPaymentDetails.subscriptionId === 'AndroidPayCard'
            || selectedPaymentDetails.subscriptionId === 'VisaCheckoutCard'
        )) {
      this.braintreePaymentDetailsService.getBraintreePaymentDetailsById(selectedPaymentDetails.id, Fields.Full)
      .subscribe(paymentDetails => {
        if (checkoutData.creditCardPaymentMethod.secure3d || paymentDetails.shouldPerform3dSecure) {
          this.eventService.get(BraintreeChangedPaymentDetailsEvent).subscribe(
              () => this.next()
          );
          this.braintreeHostedFieldsService.initialise3dSecure(
              paymentDetails, 
              checkoutData,
              PageType.BILLING,
              undefined,
              this.activatedRoute);

        } else {
          this.checkoutStepService.next(this.activatedRoute);
        }
      });
    } else {
      this.checkoutStepService.next(this.activatedRoute);
    }
   }
  }

  getPaymentMethodCardBraintree(
      paymentDetails: BraintreePaymentDetails
  ): Observable<Card> {
    return combineLatest([
      this.translation.translate('paymentDetails.defaultPaymentMethod'),
      this.translation.translate('paymentForm.payment'),
      this.translation.translate('paymentCard.expires', {
        month: paymentDetails.expiryMonth,
        year: paymentDetails.expiryYear,
      }),
      this.translation.translate('paymentForm.useThisPayment'),
      this.translation.translate('paymentCard.selected'),
      this.selectedMethod
    ]).pipe(
        map(([
               textDefaultPaymentMethod,
               textTitle,
               textExpires,
               useThisPaymentText,
               selectedText,
               selectedPayment
             ]) => {
          let paymentTitle;
          if (paymentDetails.defaultPayment) {
            paymentTitle = textDefaultPaymentMethod;
          } else {
            paymentTitle = textTitle;
          }

          const region = paymentDetails.billingAddress?.region?.isocode
              ? paymentDetails.billingAddress?.region?.isocode + ', '
              : '';

          const billingActions: { name: string; event: string }[] = [];
          billingActions.push({name: useThisPaymentText, event: 'send'});

          if (!selectedPayment || Object.keys(selectedPayment).length === 0) {
            selectedPayment = this.selectDefaultPayment(this.paymentMethods);
          } else if (selectedPayment.id != null && (selectedPayment as BraintreePaymentDetails).paymentMethodNonce) {
            const updatedSelectedPayment = this.paymentMethods
            .find(paymentMethod => paymentMethod.id === selectedPayment.id);
            if (updatedSelectedPayment != null &&
                (selectedPayment as BraintreePaymentDetails).paymentMethodNonce !== updatedSelectedPayment.paymentMethodNonce) {
              this.selectPaymentMethod(updatedSelectedPayment);
            }
          }
          return {
            title: paymentTitle,
            textBold: paymentDetails.subscriptionId,
            text: this.braintreePaymentMethodUtilsService.configurePaymentMethodCardText(
                paymentDetails,
                textExpires,
                PageType.BILLING
            ),
            img: paymentDetails.accountHolderName,
            paragraphs: [
              {
                text: [
                  paymentDetails.billingAddress?.line1,
                  paymentDetails.billingAddress?.town +
                  ', ' +
                  region +
                  paymentDetails.billingAddress?.country?.name,
                  paymentDetails.billingAddress?.postalCode,
                ],
              },
            ],
            actions: billingActions,
            header:
                selectedPayment?.id === paymentDetails.id
                    ? selectedText
                    : undefined,
          };
        })
    );
  }

  selectDefaultPayment(payments: BraintreePaymentDetails[]): PaymentDetails {
    if (payments && payments.length) {
      let selected: BraintreePaymentDetails;
      if (payments.length === 1) {
        this.selectPaymentMethod(payments[0]);
        this.selectedMethod$ = of(payments[0]);
      } else {
        selected = payments.find((payments) => payments.defaultPayment);
        if (selected) {
          this.selectPaymentMethod(selected);
          this.selectedMethod$ = of(selected);
        }
        return selected;
      }
    }
    return undefined;
  }

  setBraintreePaymentDetails(payments: BraintreePaymentDetails[]): boolean {
    this.paymentMethods = payments;
    return true;
  }

  createdAndSetBraintreePaymentDetails(): void {
    this.eventService.get(CreatedAndSetBraintreePaymentDetailsEvent).subscribe(
        (event) => {
          this.next();
        }
    );
  }
}
