import {ConnectAdapter} from '../../braintree-checkout/connectors/connect.adapter';
import {Injectable} from '@angular/core';
import {Observable} from 'rxjs';
import {HttpClient, HttpHeaders, HttpParams} from '@angular/common/http';
import {ConverterService, OccEndpointsService} from '@spartacus/core';
import {
  PAYPAL_BOOLEAN_NORMALIZER,
  PAYPAL_CONNECT_B2B_DATA_NORMALIZER,
  PAYPAL_CONNECT_B2B_REGISTER_DATA_SERIALIZER,
  PAYPAL_CONNECT_DATA_NORMALIZER,
  PAYPAL_CONNECT_LOGIN_DATA_NORMALIZER,
  PAYPAL_STRING_NORMALIZER,
  PAYPAL_ADDING_PAYMENT_METHOD_AND_ADDRESS_STATUS_NORMALIZER,
} from '../../braintree-checkout/converters';
import {PAYPAL_PAYMENT_DETAILS_SERIALIZER} from '../../braintree-checkout/braintree-checkout-store/converters/converters';
import {
  AddingPaymentMethodAndAddressStatusData,
  Fields,
  PageType,
  PayPalConnectB2BData,
  PayPalConnectB2BRegisterData,
  PayPalConnectData,
  PayPalConnectLoginData, PayPalDetail, PayPalPaymentRequest
} from '../../braintree-checkout/models';

@Injectable()
export class PaypalConnectAdapter implements ConnectAdapter {

  constructor(
      protected http: HttpClient,
      protected occEndpoints: OccEndpointsService,
      protected converter: ConverterService
  ) {
  }

  protected getEndpoint(userId): string {
    return this.occEndpoints.buildUrl('users/${userId}', {
      urlParams: {userId},
    }) + '/connect';
  }

  exchangeAuthorizationCode(
      authorizationToken: string,
      userId: string): Observable<PayPalConnectData> {
    const headers = new HttpHeaders({'Content-Type': 'application/json'});

    const url = `${this.getEndpoint(userId)}/exchange`;

    return this.http.post(url, {'authorizationCode': authorizationToken}, {headers})
    .pipe(this.converter.pipeable(PAYPAL_CONNECT_DATA_NORMALIZER));
  }

  afterLogin(
      accessTokenGuid: string,
      userId: string
  ): Observable<boolean> {
    const headers = new HttpHeaders({'Content-Type': 'application/json'});

    const url = `${this.getEndpoint(userId)}/afterLogin`;

    return this.http.post(url, {'accessTokenGuid': accessTokenGuid}, {headers})
    .pipe(this.converter.pipeable(PAYPAL_BOOLEAN_NORMALIZER));
  }

  login(
      accessTokenGuid: string,
      userId: string
  ): Observable<PayPalConnectLoginData> {
    const headers = new HttpHeaders({'Content-Type': 'application/json'});

    const url = `${this.getEndpoint(userId)}/login`;

    return this.http.post(url, {'accessTokenGuid': accessTokenGuid}, {headers})
    .pipe(this.converter.pipeable(PAYPAL_CONNECT_LOGIN_DATA_NORMALIZER));
  }

  register(
      accessTokenGuid: string,
      userId: string
  ): Observable<string> {
    const headers = new HttpHeaders({'Content-Type': 'application/json'});

    const url = `${this.getEndpoint(userId)}/register`;

    return this.http.post(url, {'accessTokenGuid': accessTokenGuid}, {headers})
    .pipe(this.converter.pipeable(PAYPAL_STRING_NORMALIZER));
  }

  exchangeAuthorizationCodeB2B(
      authorizationToken: string,
      userId: string
  ): Observable<PayPalConnectB2BData> {
    const headers = new HttpHeaders({'Content-Type': 'application/json'});

    const url = `${this.getEndpoint(userId)}/exchange`;

    return this.http.post(url, {'authorizationCode': authorizationToken}, {headers})
    .pipe(this.converter.pipeable(PAYPAL_CONNECT_B2B_DATA_NORMALIZER));
  }

  registerB2BCustomer(
      registerData: PayPalConnectB2BRegisterData,
      userId: string
  ): Observable<string> {
    const headers = new HttpHeaders({'Content-Type': 'application/json'});

    const url = `${this.getEndpoint(userId)}/register`;

    registerData = this.converter.convert(registerData, PAYPAL_CONNECT_B2B_REGISTER_DATA_SERIALIZER);

    return this.http.post(url, registerData, {headers})
    .pipe(this.converter.pipeable(PAYPAL_STRING_NORMALIZER));
  }

  public savePaymentDetailsAndAddressForNewUser(
      accessTokenGuid:string,
      payPalRequest: PayPalPaymentRequest,
      shouldSaveAddressInfo: boolean,
      shouldSavePaymentInfo: boolean,
      userId: string,
      cartId?: string,
      deviceData?: string,
      fundingSource?: string
  ): Observable<AddingPaymentMethodAndAddressStatusData>{


    const url = `${this.getEndpoint(
        userId)}/createPaymentDetails`;

    if (shouldSavePaymentInfo) {
      payPalRequest = this.converter.convert(payPalRequest, PAYPAL_PAYMENT_DETAILS_SERIALIZER);
    } else {
      payPalRequest = {
        nonce: null,
        type: null,
        details: null
      }
    }
    
    let params: HttpParams = new HttpParams()
    .set('shouldSaveAddressInfo', shouldSaveAddressInfo)
    .set('shouldSavePaymentInfo', shouldSavePaymentInfo)
    .set('accessTokenGuid', accessTokenGuid);

    if (deviceData !== undefined){
      params = params.set('deviceData', deviceData);
    }

    if (fundingSource !== undefined){
      params = params.set('fundingSource', fundingSource);
    }

    const headers = new HttpHeaders({
      'Content-Type': 'application/json',
    });

    return this.http.post(url, payPalRequest, {headers, params})
    .pipe(this.converter.pipeable(PAYPAL_ADDING_PAYMENT_METHOD_AND_ADDRESS_STATUS_NORMALIZER));
  }



}
