import { BraintreePaymentDetails } from '../../models/braintree-payment-data.model';
import {EventEmitter, Injectable} from '@angular/core';
import { Occ, TranslationService, PaymentDetails } from '@spartacus/core';
import {combineLatest, Observable} from 'rxjs';
import {Card} from '@spartacus/storefront';
import {map} from 'rxjs/operators';
import {paymentText} from '../../models/braintree.constants';
import Address = Occ.Address;
import {PageType} from '../../models';

@Injectable({
  providedIn: 'root'
})
export class BraintreePaymentMethodsUtilsService {

  protected savePaymentInfo = false;
  protected billingAddress: Address;
  protected _backToPaymentMethods: EventEmitter<any>;

  constructor(
    protected translation: TranslationService
  ) { }

  toggleSavePaymentInfo(): void {
    this.savePaymentInfo = !this.savePaymentInfo;
    console.log(this.savePaymentInfo);
  }

  setSavePaymentInfo(savePaymentInfo: boolean): void{
    this.savePaymentInfo = savePaymentInfo;
  }

  getSavePaymentInfo(): boolean{
    return this.savePaymentInfo;
  }

  setBillingAddress(address: Address): void{
    this.billingAddress = address;
  }

  getBillingAddress(): Address{
    return this.billingAddress;
  }

  getPaymentMethodCardBraintreeMyAccount(paymentDetails: BraintreePaymentDetails): Observable<Card> {
    return combineLatest([
      this.translation.translate('paymentCard.defaultPaymentMethod'),
      this.translation.translate('paymentForm.payment'),
      this.translation.translate('paymentCard.expires', {
        month: paymentDetails.expiryMonth,
        year: paymentDetails.expiryYear,
      }),
      this.translation.translate('paymentForm.billingAddress'),
    ]).pipe(
      map(([textDefaultPaymentMethod, textTitle, textExpires, billingAddress]) => {
        const textDefault = paymentDetails.defaultPayment ? textDefaultPaymentMethod : '';

        const region = paymentDetails.billingAddress?.region?.isocode
          ? paymentDetails.billingAddress?.region?.isocode + ', '
          : '';

        const myAccountActions: { name: string; event: string }[] = [];
        if (!paymentDetails.defaultPayment) {
          myAccountActions.push({ name: 'Set as default', event: 'default' });
        }
        myAccountActions.push({ name: 'Delete', event: 'edit' });
        if (paymentDetails.subscriptionId === 'CreditCard') {
          myAccountActions.push({name: 'Update Billing Address', event: 'updateBillingAddress'});
        }

        return {
          header: textDefault,
          title: textTitle,
          textBold: paymentDetails.subscriptionId,
          text: this.configurePaymentMethodMyAccountCardText(paymentDetails, textExpires),
          img: paymentDetails.accountHolderName,
          paragraphs: [
            {
              text: [
                paymentDetails.billingAddress?.line1,
                paymentDetails.billingAddress?.town +
                ', ' +
                region +
                paymentDetails.billingAddress?.country?.isocode,
                paymentDetails.billingAddress?.postalCode,
              ],
            },
          ],
          actions: myAccountActions
        };
      })
    );
  }


  getBackToPaymentMethodsEmitter(): EventEmitter<any> {
    return this._backToPaymentMethods;
  }

  setBackToPaymentMethods(value: EventEmitter<any>) {
    this._backToPaymentMethods = value;
  }

  getPaymentMethodVenmoBraintreeReviewOrder(paymentDetails: BraintreePaymentDetails): Observable<Card> {
    return combineLatest([
      this.translation.translate('paymentForm.payment'),
      this.translation.translate('paymentCard.expires', {
        month: paymentDetails.expiryMonth,
        year: paymentDetails.expiryYear,
      }),
    ]).pipe(
        map(([textTitle, textExpires]) => {
          return {
            title: textTitle,
            textBold: paymentDetails.subscriptionId,
            text: this.configurePaymentMethodCardText(paymentDetails, textExpires, PageType.REVIEW),
            img: paymentDetails.accountHolderName,
          };
        })
    );
  }

  getPaymentMethodCardBraintreeReviewOrder(paymentDetails: BraintreePaymentDetails): Observable<Card> {
    if (paymentDetails.subscriptionId === 'VenmoAccount'){
      return this.getPaymentMethodVenmoBraintreeReviewOrder(paymentDetails);
    }
    return combineLatest([
      this.translation.translate('paymentForm.payment'),
      this.translation.translate('paymentCard.expires', {
        month: paymentDetails.expiryMonth,
        year: paymentDetails.expiryYear,
      }),
      this.translation.translate('paymentForm.billingAddress'),
    ]).pipe(
      map(([textTitle, textExpires, billingAddress]) => {
        const region = paymentDetails.billingAddress?.region?.isocode
          ? paymentDetails.billingAddress?.region?.isocode + ', '
          : '';
        return {
          title: textTitle,
          textBold: paymentDetails.subscriptionId,
          text: this.configurePaymentMethodCardText(paymentDetails, textExpires, PageType.REVIEW),
          img: paymentDetails.accountHolderName,
          paragraphs: [
            {
              text: [
                paymentDetails.billingAddress?.line1,
                paymentDetails.billingAddress?.town +
                ', ' +
                region +
                paymentDetails.billingAddress?.country?.name,
                paymentDetails.billingAddress?.postalCode,
              ],
            },
          ],
        };
      })
    );
  }

  configurePaymentMethodCardText(paymentDetails: BraintreePaymentDetails, textExpires: string, pageType?: PageType): string[]{
    if (paymentDetails.expiryMonth === undefined || paymentDetails.expiryYear === undefined) {
      textExpires = undefined;
    }

    if (paymentDetails.subscriptionId === 'AndroidPayCard'){
      if (paymentDetails.expiryMonth === '' && paymentDetails.expiryYear === '') {
        return [paymentText.GOOGLEPAY_ACCOUNT, paymentDetails.payer, paymentDetails.cardNumber, paymentDetails.cardType?.code];
      } else {
        return [paymentText.GOOGLEPAY_ACCOUNT, paymentDetails.payer, paymentDetails.cardNumber, paymentDetails.cardType?.code, textExpires];
      }
    }else if (paymentDetails.subscriptionId === 'PayPalAccount' || paymentDetails.subscriptionId === 'BrainTreePayPalExpress'){
      let paymentType;
      if (paymentDetails.paypalFundingSource === 'card') {
        paymentType = paymentText.PAYPAL_CREDIT_ACCOUNT;
      } else {
        paymentType = paymentDetails.subscriptionId;
      }
      return [paymentType, paymentDetails.payer];
    } else if (paymentDetails.subscriptionId === 'VenmoAccount'){
      return [paymentText.VENMO_ACCOUNT, paymentDetails.payer];
    }else if (paymentDetails.subscriptionId === 'CreditCard'){
      return pageType != null && pageType === PageType.CONFIRMATION ?
          [paymentText.CREDIT_CARD_TYPE, paymentDetails.billingAddress?.email,
            paymentDetails.cardholderName, paymentDetails.cardNumber,
            paymentDetails.cardType?.code, textExpires] :
          [paymentText.CREDIT_CARD_TYPE,
            paymentDetails.cardholderName, paymentDetails.cardNumber,
            paymentDetails.cardType?.code, textExpires];

    }else if (paymentDetails.subscriptionId === 'ApplePayCard'){
      return [paymentText.APPLEPAY_ACCOUNT, paymentDetails.payer];
    } else if (paymentDetails.subscriptionId === 'VisaCheckoutCard') {
      return [paymentText.SRC_ACCOUNT, paymentDetails.billingAddress?.email, paymentDetails.cardNumber,
        paymentDetails.cardType.code, textExpires];
    }
    return [paymentDetails.subscriptionId, paymentDetails.cardNumber];
  }

  configurePaymentMethodMyAccountCardText(paymentDetails: BraintreePaymentDetails, textExpires: string): string[]{
    if (paymentDetails.subscriptionId === 'AndroidPayCard'){
      if (paymentDetails.expiryMonth === '' && paymentDetails.expiryYear === '') {
        return [paymentText.GOOGLEPAY_ACCOUNT, paymentDetails.billingAddress?.email,
          paymentDetails.cardNumber, paymentDetails.cardType?.code];
      } else {
        return [paymentText.GOOGLEPAY_ACCOUNT, paymentDetails.billingAddress?.email,
          paymentDetails.cardNumber, paymentDetails.cardType?.code, textExpires];
      }
    }else if (paymentDetails.subscriptionId === 'PayPalAccount' || paymentDetails.subscriptionId === 'BrainTreePayPalExpress'){
      let paymentType;
      if (paymentDetails.paypalFundingSource === 'card') {
        paymentType = paymentText.PAYPAL_CREDIT_ACCOUNT;
      } else {
        paymentType = paymentDetails.subscriptionId;
      }
      return [paymentType, paymentDetails.billingAddress?.email];
    } else if (paymentDetails.subscriptionId === 'VenmoAccount'){
      return [paymentText.VENMO_ACCOUNT, paymentDetails.payer];
    }else if (paymentDetails.subscriptionId === 'CreditCard'){
      return [paymentText.CREDIT_CARD_TYPE, paymentDetails.cardholderName, paymentDetails.cardNumber,
        paymentDetails.cardType?.code, textExpires];
    }else if (paymentDetails.subscriptionId === 'ApplePayCard'){
      return [paymentText.APPLEPAY_ACCOUNT, paymentDetails.billingAddress?.email];
    } else if (paymentDetails.subscriptionId === 'VisaCheckoutCard') {
      return [paymentText.SRC_ACCOUNT, paymentDetails.billingAddress?.email, paymentDetails.cardNumber,
        paymentDetails.cardType.code, textExpires];
    }
    return [paymentDetails.subscriptionId, paymentDetails.cardNumber];
  }

}
